﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

/**
    @file
    @details nn::MemoryResource をサービスフレームワークで使用するためのユーティリティ群です。
*/

#include <nn/nn_Allocator.h>

#include <nn/sf/sf_AllocationPolicies.h>
#include <nn/nn_SdkAssert.h>

namespace nn { namespace sf {

/**
    @brief MemoryResource を使用する状態ありのアロケーションポリシーです。

    @details
     本クラスをアロケーションポリシーとして使用すると、
     MemoryResource を使用する状態ありのアロケーションポリシーとしてふるまいます。

     本アロケーションポリシーを指定した際には有効な MemoryResource* をアロケータとして渡す必要があります。

    @see MemoryResource, StandardAllocatorMemoryResource, UnitHeapMemoryResource, ExpHeapMemoryResource, GetNewDeleteMemoryResource()
*/
struct MemoryResourceAllocationPolicy
{
    static const bool HasStatefulAllocator = true;
    typedef MemoryResource Allocator;

    static void* AllocateAligned(MemoryResource* pMemoryResource, size_t size, size_t alignment) NN_NOEXCEPT
    {
        return pMemoryResource->allocate(size, alignment);
    }

    static void DeallocateAligned(MemoryResource* pMemoryResource, void* p, size_t size, size_t alignment) NN_NOEXCEPT
    {
        pMemoryResource->deallocate(p, size, alignment);
    }
};

/**
    @brief 特定の MemoryResource をサービスフレームワークのアロケータとして使用すためのアダプタです。

    @param Tag アロケータを静的に識別するためのタグです。

    @details
     使用前に Initialize() に有効な MemoryResource へのポインタを渡して呼ぶ必要があります。
     なお、この際に渡す MemoryResource は、サービスフレームワークから排他無しに呼び出されます。
     複数スレッドからの使用が見込まれる場合には、
     メモリ確保・解放関数を同時に呼び出すことが可能(スレッドセーフ)な実装を使用してください。

     テンプレート引数 Tag は、同じ MemoryResource を使いたい単位ごとに、ユーザ定義の空の struct 型などを指定してください。
     void や int などの一般的な型を指定はしないようにしてください。

     Policy メンバは、状態のアロケーションポリシーとして使用できます。

    @see MemoryResource, StandardAllocatorMemoryResource, UnitHeapMemoryResource, ExpHeapMemoryResource, GetNewDeleteMemoryResource()
*/
template <typename Tag>
struct MemoryResourceStaticAllocator
{
    /**
        @brief MemoryResource を使って初期化します。

        @param[in] pMemoryResource MemoryResource へのポインタを指定します。

        @pre pMemoryResource != nullptr

        @post アロケーション可能状態である
    */
    static void Initialize(MemoryResource* pMemoryResource) NN_NOEXCEPT
    {
        g_pMemoryResource = pMemoryResource;
    }

    /**
        @brief サービスフレームワークの状態なしのアロケーションポリシーとして使用できる型です。
    */
    struct Policy
    {
        static const bool HasStatefulAllocator = false;
        typedef MemoryResource Allocator;

        template <typename>
        struct GetAllocator
        {
            typedef Allocator type;
        };

        template <typename>
        static void* AllocateAligned(size_t size, size_t alignment) NN_NOEXCEPT
        {
            return g_pMemoryResource->allocate(size, alignment);
        }

        template <typename>
        static void DeallocateAligned(void* p, size_t size, size_t alignment) NN_NOEXCEPT
        {
            g_pMemoryResource->deallocate(p, size, alignment);
        }
    };

    static MemoryResource* g_pMemoryResource;
};

template <typename Tag>
MemoryResource* MemoryResourceStaticAllocator<Tag>::g_pMemoryResource;

}}
