﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

/**
    @file
    @brief lmem ライブラリをサービスフレームワーク上で使用するためのユーティリティです。

    @details
     このファイルをインクルードすることで、以下のファイルで定義されているものを間接的にインクルードします。

     - <nn/lmem/lmem_ExpHeap.h>
     - <nn/lmem/lmem_UnitHeap.h>
     - <nn/sf/sf_MemoryResource.h>

    @see nn::lmem
*/

#include <nn/nn_Common.h>
#include <nn/lmem/lmem_ExpHeap.h>
#include <nn/lmem/lmem_UnitHeap.h>
#include <nn/sf/sf_MemoryResource.h>
#include <nn/nn_Allocator.h>
#include <nn/nn_SdkAssert.h>

namespace nn { namespace sf {

/**
    @brief 拡張ヒープを MemoryResource として扱うためのアダプタです。

    @details
     本アダプタは内部で排他処理を行いません。
     サービスフレームワーク内におけるアロケーションは、一般に同時に呼び出される可能性があるため、
     本アダプタに与える拡張ヒープハンドルは lmem::CreationOption_ThreadSafe を指定して作成される必要があります。

    @see lmem::CreateExpHeap, MemoryResourceAllocator, MemoryResourceStaticAllocator, DefaultAllocationPolicy
*/
class ExpHeapMemoryResource
    : public MemoryResource
{
public:

    /**
        @brief 指定した拡張ヒープハンドルを用いて初期化するコンストラクタです。

        @param[in] handle 拡張ヒープハンドルを指定します。

        @pre handle が有効な拡張ヒープハンドルである

        @post this->GetHandle() == handle
    */
    explicit ExpHeapMemoryResource(lmem::HeapHandle handle) NN_NOEXCEPT
        : m_Handle(handle)
    {
    }

    /**
        @brief 関連付けられている拡張ヒープハンドルを取得します。

        @return 関連付けられている拡張ヒープハンドルを返します。
    */
    lmem::HeapHandle GetHandle() const NN_NOEXCEPT
    {
        return m_Handle;
    }

private:

    virtual void* do_allocate(std::size_t bytes, std::size_t alignment) NN_NOEXCEPT NN_OVERRIDE
    {
        return lmem::AllocateFromExpHeap(m_Handle, bytes, static_cast<int>(alignment));
    }

    virtual void do_deallocate(void* p, std::size_t, std::size_t) NN_NOEXCEPT NN_OVERRIDE
    {
        lmem::FreeToExpHeap(m_Handle, p);
    }

    virtual bool do_is_equal(const MemoryResource& other) const NN_NOEXCEPT NN_OVERRIDE
    {
        return this == &other;
    }

    lmem::HeapHandle m_Handle;

};

/**
    @brief ユニットヒープを MemoryResource として扱うためのアダプタです。

    @details
     ユニットヒープを使用してメモリーリソースを作成しますが、
     ユニットヒープはその仕様上、ユニットサイズ以下のサイズのメモリしか確保できません。
     ユニットサイズより大きいサイズのメモリ確保要求があった際の動作は保証されません。
     よって、サービスフレームワークのアロケータとして使用する際には十分に注意する必要があるため、
     サービスフレームワークのマニュアルなどで指示がある場合を除いて、使用しないようにしてください。

     また、アライメントを明示的に指定してユニットヒープを作成する場合には、
     標準アラインメント DefaultAlignment を使用するようにしてください。

     本アダプタは内部で排他処理を行いません。
     サービスフレームワーク内におけるアロケーションは、一般に同時に呼び出される可能性があるため、
     本アダプタに与えるユニットヒープハンドルは lmem::CreationOption_ThreadSafe を指定して作成される必要があります。

    @see lmem::CreateUnitHeap, MemoryResourceAllocator, MemoryResourceStaticAllocator, DefaultAllocationPolicy
*/
class UnitHeapMemoryResource
    : public MemoryResource
{
public:

    /**
        @brief 指定したユニットヒープハンドルを用いて初期化するコンストラクタです。

        @param[in] handle ユニットヒープハンドルを指定します。

        @pre handle が有効なユニットヒープハンドルである

        @post this->GetHandle() == handle
    */
    explicit UnitHeapMemoryResource(lmem::HeapHandle handle) NN_NOEXCEPT
        : m_Handle(handle)
    {
    }

    /**
        @brief 関連付けられているユニットヒープハンドルを取得します。

        @return 関連付けられているユニットヒープハンドルを返します。
    */
    lmem::HeapHandle GetHandle() const NN_NOEXCEPT
    {
        return m_Handle;
    }

private:

    virtual void* do_allocate(std::size_t bytes, std::size_t alignment) NN_NOEXCEPT NN_OVERRIDE
    {
        NN_SDK_REQUIRES(bytes <= lmem::GetUnitHeapUnitSize(m_Handle));
        NN_SDK_REQUIRES(alignment <= static_cast<size_t>(lmem::GetUnitHeapAlignment(m_Handle)));
        NN_UNUSED(bytes);
        NN_UNUSED(alignment);
        return lmem::AllocateFromUnitHeap(m_Handle);
    }

    virtual void do_deallocate(void* p, std::size_t, std::size_t) NN_NOEXCEPT NN_OVERRIDE
    {
        lmem::FreeToUnitHeap(m_Handle, p);
    }

    virtual bool do_is_equal(const MemoryResource& other) const NN_NOEXCEPT NN_OVERRIDE
    {
        return this == &other;
    }

    lmem::HeapHandle m_Handle;

};

}}
