﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

/**
    @file
    @brief HIPC におけるサーバ処理への追加ハンドラを提供します。
*/

#include <nn/nn_Common.h>
#include <nn/nn_Result.h>

namespace nn { namespace sf {

/**
    @brief HIPC サーバの追加ハンドラ機能を初期化し、使用可能にします。

    @post HIPC サーバの追加ハンドラ機能を初期化されている。

    @details
     本関数は複数回呼び出すことが可能で、2 度目以降の呼び出しでは何も処理が行われません。
     ただし、本関数を複数のスレッドから同時に呼び出すことはできません。
     テストなどを除いて、プログラムの最初などで一度だけ呼び出すようにしてください。
*/
void InitializeForHipcServerSessionManagerHandler() NN_NOEXCEPT;

/**
    @brief HIPC サーバの追加ハンドラを表すインターフェイスクラスです。

    @details
     本クラスを基底とするクラスインスタンスを SetCurrentHipcServerSessionManagerHandler() に渡すことで、
     そのスレッドにおける HIPC サーバの挙動を変更することができます。
*/
class IHipcServerSessionManagerHandler
{
public:

    /**
        @brief リクエストに対する実処理を行う前に呼ばれる関数です。

        @return 処理結果を返します。

        @details
         リクエストに対する実処理を行う前に呼ばれる関数です。
         本関数が成功を返した場合、以降の実処理が行われ、その後 AfterInvoke() が呼ばれます。
         本関数が失敗を返した場合、以降の実処理は行われません。AfterInvoke() も呼び出されません。

         DeferProcess() を返した場合には、リクエストが遅延されます。
    */
    virtual Result BeforeInvoke() NN_NOEXCEPT
    {
        return ResultSuccess();
    }

    /**
        @brief リクエストに対する実処理を行った後に呼ばれる関数です。

        @see BeforeInvoke()
    */
    virtual void AfterInvoke() NN_NOEXCEPT
    {
    }

    /**
        @brief クライアントが追加された際に呼ばれる関数です。

        @param[in] clientId クライアントを識別する識別子を指定します。

        @details
         クライアントが追加された際に、そのクライアントを識別する clientId で呼ばれる関数です。

         本関数は同一とみなされるクライアントが複数のセッションを張った場合などに、同じ clientId に対して複数回呼ばれることがあります。
         この場合、クライアントが完全に利用を終了する時点までに同じ回数だけ、同じ clientId で OnRemoveClient() も呼ばれます。
    */
    virtual void OnAddClient(Bit64 clientId) NN_NOEXCEPT
    {
        NN_UNUSED(clientId);
    }

    /**
        @brief クライアントが削除された際に呼ばれる関数です。

        @param[in] clientId クライアントを識別する識別子を指定します。

        @details
         クライアントがいなくなった際に、そのクライアントを識別する clientId で呼ばれる関数です。
    */
    virtual void OnRemoveClient(Bit64 clientId) NN_NOEXCEPT
    {
        NN_UNUSED(clientId);
    }

protected:

    /**
        @brief デストラクタです。
    */
    ~IHipcServerSessionManagerHandler() NN_NOEXCEPT = default;

};

/**
    @brief 現在 HIPC サーバで処理中のクライアント ID を取得します。

    @return クライアント ID を返します。

    @pre HIPC サーバの追加ハンドラ機能を初期化されている。
*/
Bit64 GetCurrentHipcClientIdOnServer() NN_NOEXCEPT;

}}
