﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/sf/detail/sf_BinaryVersion.h>
#if NN_SF_DETAIL_BINARY_VERSION_EFFECTIVE == NN_SF_LATEST_BINARY_VERSION

#include <nn/nn_Common.h>
#include <nn/nn_Result.h>
#include <nn/os/os_MultipleWaitTypes.h>
#include <nn/sf/cmif/server/sf_CmifServerObjectInfo.h>
#include <nn/sf/hipc/sf_HipcHandleTypes.h>
#include <nn/sf/hipc/server/sf_HipcServerApiModel.h>
#include <nn/sf/sf_HipcServerSessionManagerHandler.h>

#include <utility>
#include <nn/util/util_ScopeExit.h>
#include <nn/sf/hipc/sf_HipcResult.h>

namespace nn { namespace sf { namespace hipc { namespace server { inline namespace v1 {

class HipcServerSessionManagerBase;

class HipcServerSessionBase
    : public nn::os::MultiWaitHolderType // WaitAny 可能
{
    friend class HipcServerSessionManagerBase;
private:
    bool m_Closed;
    HipcServerSessionHandle m_ServerHandle;
public:
    Result Reply(void* messageBuffer, size_t messageBufferSize) NN_NOEXCEPT;
};

class HipcServerSessionManagerBase
{
public:

    Result ProcessRequest(HipcServerSessionBase* pSession, void* messageBuffer, size_t messageBufferSize) NN_NOEXCEPT;

protected:

    Result ReceiveRequestBase(HipcServerSessionBase* p, void* messageBuffer, size_t messageBufferSize, void* pointerBuffer, size_t pointerBufferSize) NN_NOEXCEPT;
    void CloseSessionBase(HipcServerSessionBase* pSession) NN_NOEXCEPT;
    Result RegisterBase(HipcServerSessionBase* pSession, HipcServerSessionHandle serverSessionHandle) NN_NOEXCEPT;
    Result AcceptBase(HipcServerSessionBase* pSession, HipcServerPortHandle serverPortHandle) NN_NOEXCEPT;

    virtual void DestroyServerSession(HipcServerSessionBase* p) NN_NOEXCEPT = 0;

private:

    virtual Result ProcessMessage2(HipcServerSessionBase* pSession, void* inMessageBuffer, size_t inMessageBufferSize, void* outMessageBuffer, size_t outMessageBufferSize) NN_NOEXCEPT = 0;
    virtual void RegisterServerSessionToWaitBase(HipcServerSessionBase* pSession) NN_NOEXCEPT = 0;

};

struct ClientInfo;

template <typename T>
class HipcServerSessionManagerT
    : public HipcServerSessionManagerBase
{
private:

    template <typename F>
    Result CreateServerSessionImpl(HipcServerSessionBase** pOut, const F& f, T&& x, const ClientInfo& clientInfo) NN_NOEXCEPT
    {
        auto p = this->CreateServerSession(std::forward<T>(x), clientInfo);
        NN_RESULT_THROW_UNLESS(p != nullptr, hipc::ResultOutOfServerSessionInfoMemory());
        auto success = false;
        NN_UTIL_SCOPE_EXIT
        {
            if (!success)
            {
                DestroyServerSession(p);
            }
        };
        NN_RESULT_DO(f(p));
        success = true;
        *pOut = p;
        NN_RESULT_SUCCESS;
    }

public:

    Result Register(HipcServerSessionBase** pOut, HipcServerSessionHandle serverSessionHandle, T&& x, const ClientInfo& clientInfo) NN_NOEXCEPT
    {
        auto f = [=] (HipcServerSessionBase* p) NN_NOEXCEPT
        {
            return RegisterBase(p, serverSessionHandle);
        };
        return CreateServerSessionImpl(pOut, f, std::forward<T>(x), clientInfo);
    }

    Result Accept(HipcServerSessionBase** pOut, HipcServerPortHandle serverPortHandle, T&& x, const ClientInfo& clientInfo) NN_NOEXCEPT
    {
        auto f = [=] (HipcServerSessionBase* p) NN_NOEXCEPT
        {
            return AcceptBase(p, serverPortHandle);
        };
        return CreateServerSessionImpl(pOut, f, std::forward<T>(x), clientInfo);
    }

private:

    virtual HipcServerSessionBase* CreateServerSession(T&& x, const ClientInfo& clientInfo) NN_NOEXCEPT = 0;

};

struct ClientInfo
{
    uintptr_t clientId;
    static ClientInfo NewClientInfo() NN_NOEXCEPT;
    bool IsValid() const NN_NOEXCEPT
    {
        return clientId != 0;
    }
};

class HipcServerSessionManager;
class HipcServerSessionManagerWithDomain;

class HipcServerSession
    : public HipcServerSessionBase
{
    friend class HipcServerSessionManager;
    friend class HipcServerSessionManagerWithDomain;
private:

    cmif::server::CmifServerObjectInfo m_X;
    void* m_PointerBuffer;
    size_t m_PointerBufferSize;
    ClientInfo m_ClientInfo;

};

class HipcServerSessionManager
    : private HipcServerSessionManagerT<cmif::server::CmifServerObjectInfo>
    , protected HipcServerApiModelHolder
{
private:

    typedef HipcServerSessionManagerT<cmif::server::CmifServerObjectInfo> Base;

public:

    HipcServerSessionManager()
        : HipcServerApiModelHolder(GetDefaultServerApiModel())
    {
    }

    using Base::ProcessRequest;

    Result ReceiveRequest(HipcServerSession* p, void* messageBuffer, size_t messageBufferSize) NN_NOEXCEPT
    {
        return Base::ReceiveRequestBase(p, messageBuffer, messageBufferSize, p->m_PointerBuffer, p->m_PointerBufferSize);
    }

    Result Register(HipcServerSessionHandle serverSessionHandle, cmif::server::CmifServerObjectInfo&& x, const ClientInfo& clientInfo) NN_NOEXCEPT;

    template <typename Interface>
    Result Accept(HipcServerPortHandle serverPortHandle, SharedPointer<Interface> p) NN_NOEXCEPT
    {
        return Accept(serverPortHandle, cmif::server::CmifServerObjectInfo(std::move(p)));
    }

protected:

    Result Accept(HipcServerPortHandle serverPortHandle, cmif::server::CmifServerObjectInfo&& x) NN_NOEXCEPT;

    Result Process2InvokeMethodImpl(cmif::server::CmifServerObjectInfo&& targetObject, HipcServerSession* pSession, void* inMessageBuffer, size_t inMessageBufferSize, void* outMessageBuffer, size_t outMessageBufferSize) NN_NOEXCEPT;

private:

    // 上位でオーバライドが必要な関数

    virtual HipcServerSession* AllocateServerSession() NN_NOEXCEPT = 0;
    virtual void DeallocateServerSession(HipcServerSession* pSession) NN_NOEXCEPT = 0;

    virtual void* AllocatePointerTransferBuffer(size_t* pOut, HipcServerSession* pSession) NN_NOEXCEPT
    {
        NN_UNUSED(pSession);
        *pOut = 0;
        return nullptr;
    }

    virtual void DeallocatePointerTransferBuffer(HipcServerSession* pSession, void* p, size_t size) NN_NOEXCEPT
    {
        NN_UNUSED(pSession);
        NN_UNUSED(p);
        NN_UNUSED(size);
    }

    virtual void RegisterServerSessionToWait(HipcServerSession* pSession) NN_NOEXCEPT = 0;

    virtual Result Process2ManagerInvoke(HipcServerSession* pSession, void* inMessageBuffer, size_t inMessageBufferSize, void* outMessageBuffer, size_t outMessageBufferSize) NN_NOEXCEPT;

protected:

    virtual HipcServerSessionManager* GetHipcServerSessionManagerInternal(uint32_t tag) NN_NOEXCEPT
    {
        NN_UNUSED(tag);
        return this;
    }

private:

    // 実装

    virtual Result ProcessMessage2(HipcServerSessionBase* pSession, void* inMessageBuffer, size_t inMessageBufferSize, void* outMessageBuffer, size_t outMessageBufferSize) NN_NOEXCEPT final;

    virtual HipcServerSession* CreateServerSession(cmif::server::CmifServerObjectInfo&& objectInfo, const ClientInfo& clientInfo) NN_NOEXCEPT NN_OVERRIDE final;

    virtual void DestroyServerSession(HipcServerSessionBase* pSession) NN_NOEXCEPT NN_OVERRIDE final;

    virtual void RegisterServerSessionToWaitBase(HipcServerSessionBase* pSession) NN_NOEXCEPT NN_OVERRIDE final
    {
        RegisterServerSessionToWait(static_cast<HipcServerSession*>(pSession));
    }

public:

    void SetManagerHandler(IHipcServerSessionManagerHandler* pManagerHandler) NN_NOEXCEPT;

    IHipcServerSessionManagerHandler* m_pManagerHandler = nullptr;
    IHipcServerSessionManagerHandler* GetHipcServerSessionManagerHandler() const NN_NOEXCEPT;

};

}}}}}

#elif NN_SF_DETAIL_BINARY_VERSION_EFFECTIVE == 0
    #include <nn/sf/hipc/server/v0/sf_HipcServerSessionManager-v0.h>
#else
    #error "invalid NN_SF_DETAIL_BINARY_VERSION_EFFECTIVE"
#endif
