﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

// インターフェイス生成のために、自動生成系が使用するヘッダ・マクロ群
// マクロは NN_SF_CMIF_SERVER_DETAIL_ 以下であり、フレームワーク実装外からの使用は不可

#include <nn/sf/cmif/server/detail/sf_CmifProcessFunctionTable.h>

#include <nn/nn_Common.h>
#include <nn/nn_Result.h>
#include <nn/sf/cmif/sf_CmifCommonFormat.h>
#include <nn/sf/cmif/sf_CmifMessageCommon.h>
#include <nn/sf/cmif/detail/sf_CmifMethodInfos.h>
#include <nn/sf/cmif/server/sf_CmifServerMessage.h>
#include <nn/sf/cmif/server/sf_CmifServerObjectInfo.h>
#include <nn/sf/cmif/sf_CmifResult.h>

#include <utility> // std::move
#include <memory> // std::addressof
#include <cstring> // std::memcpy
#include <array> // std::array
#include <type_traits>
#include <nn/result/result_HandlingUtility.h>
#include <nn/sf/detail/sf_CommonUtil.h> // NN_SF_DETAIL_STRIP_PAREN
#include <nn/nn_Abort.h>
#include <nn/sf/detail/sf_InternalResult.h>
#include <nn/sf/detail/sf_Config.h>

#define NN_SF_CMIF_SERVER_DETAIL_GET_POINTER_CHECKED_OR_THROW_RESULT(t, name, pointerAndSize, errorResult) \
    ::std::add_pointer<NN_SF_DETAIL_STRIP_PAREN(t)>::type name; \
    { \
        NN_SF_RESULT_THROW_UNLESS( \
            NN_STATIC_CONDITION(::std::is_empty<NN_SF_DETAIL_STRIP_PAREN(t)>::value) || (pointerAndSize).size >= sizeof(*name), \
            errorResult \
        ); \
        name = reinterpret_cast<decltype(name)>((pointerAndSize).pointer); \
        (pointerAndSize).pointer += sizeof(*name); \
        (pointerAndSize).size -= NN_STATIC_CONDITION(::std::is_empty<NN_SF_DETAIL_STRIP_PAREN(t)>::value) ? 0 : sizeof(*name); \
    }

namespace nn { namespace sf { namespace cmif { namespace server { namespace detail {

// 自動生成系は、このクラステンプレートを明示的特殊化する。
// これによって、インターフェイス Interface に対し、Interface の関数テーブル取得の実装を
// nn::sf::cmif::server::detail::CmifProcessFunctionTableGetterImpl<Interface> で得ることができ、
// 関数テーブルを nn::sf::cmif::server::detail::CmifProcessFunctionTableGetter<Interface, Tag>::s_Table として
// 得ることができる。(Tag はヘッダ上に s_Table の実体を定義するためのダミーテンプレート引数)
template <typename Interface>
struct CmifProcessFunctionTableGetterImpl;

template <int InObjectCount, int OutObjectCount>
class InOutObjectsImpl
{
private:

    static const int ArrayCount = InObjectCount > OutObjectCount ? InObjectCount : OutObjectCount;
    static_assert(InObjectCount <= ArrayCount, "[SF-Internal]");
    static_assert(OutObjectCount <= ArrayCount, "[SF-Internal]");

    CmifServerObjectInfo m_ObjectInfos[ArrayCount];

    CmifServerObjectInfo* GetInObjectInfos() NN_NOEXCEPT
    {
        return m_ObjectInfos;
    }

    CmifServerObjectInfo* GetOutObjectInfos() NN_NOEXCEPT
    {
        return m_ObjectInfos;
    }

public:

    nn::Result GetMessageObjectsInfo(const CmifServerMessage& message) NN_NOEXCEPT
    {
        if (NN_STATIC_CONDITION(InObjectCount > 0))
        {
            NN_SF_RESULT_DO(message.GetInObjects(GetInObjectInfos()));
        }
        NN_SF_RESULT_SUCCESS;
    }

    template <typename Interface>
    nn::Result GetInObjectMoved(SharedPointer<Interface>* pOut, int index) NN_NOEXCEPT
    {
        auto&& info =  GetInObjectInfos()[index];
        auto p = info.template GetServiceObject<Interface>();
        NN_SF_RESULT_THROW_UNLESS(p, sf::cmif::ResultInvalidInObject()); // オブジェクトの型が不正
        *pOut = SharedPointer<Interface>(p, true);
        NN_SF_RESULT_SUCCESS;
    }

    template <typename Interface>
    void SetOutObject(SharedPointer<Interface> p, int index) NN_NOEXCEPT
    {
        GetOutObjectInfos()[index] = CmifServerObjectInfo(std::move(p));
    }

    void WriteBack(CmifServerMessage* message) NN_NOEXCEPT
    {
        if (NN_STATIC_CONDITION(OutObjectCount > 0))
        {
            message->SetOutObjects(GetOutObjectInfos());
        }
    }

};

template <>
class InOutObjectsImpl<0, 0>
{
public:

    nn::Result GetMessageObjectsInfo(const CmifServerMessage& message) NN_NOEXCEPT
    {
        NN_UNUSED(message);
        NN_SF_RESULT_SUCCESS;
    }

    void WriteBack(CmifServerMessage* message) NN_NOEXCEPT
    {
        NN_UNUSED(message);
    }

};

template <typename MethodInfo>
class InOutObjects
    : public InOutObjectsImpl<MethodInfo::InObjectCount, MethodInfo::OutObjectCount>
{
};

// CmifProcessFunctionTableGetterImpl 実装内で、
// 出力生データ構造体の管理に使用する実装ユーティリティ。
// 該当構造体が空の場合(std::is_empty)、sizeof の結果が欲しいものにならないため、
// 特殊化をしている。
template <typename MethodInfo, typename Enabled = void>
class OutRawHolder
{
private:

    typedef typename MethodInfo::OutRawStruct OutRawStruct;

    OutRawStruct m_OutRaw;

public:

    OutRawHolder() NN_NOEXCEPT
        : m_OutRaw()
    {
    }

    OutRawStruct* operator->() NN_NOEXCEPT
    {
        return &this->m_OutRaw;
    }

    void WriteBack(OutRawStruct* pOut) const NN_NOEXCEPT
    {
        // in raw と out raw に重なりがあった場合 strict aliasing rule に反しているため、
        // *pOut = m_OutRaw だと、最適化によって誤ったコードが生成される可能性がある。
        // よって、この最適化を避けるために memcpy を使用する。
        std::memcpy(pOut, &m_OutRaw, sizeof(m_OutRaw));
    }

};

template <typename MethodInfo>
class OutRawHolder<MethodInfo, typename std::enable_if<std::is_empty<typename MethodInfo::OutRawStruct>::value>::type>
{
private:

    typedef typename MethodInfo::OutRawStruct OutRawStruct;

public:

    void WriteBack(OutRawStruct* pOut) const NN_NOEXCEPT
    {
        // nop
        NN_UNUSED(pOut);
    }

};

template <typename T, typename = void>
class ServerBufferArgumentHelper
{
public:

    typedef const T& ReturnType;

    static const T& Convert(const cmif::PointerAndSize& pas) NN_NOEXCEPT
    {
        return *reinterpret_cast<const T*>(pas.pointer);
    }

};

template <typename T>
class ServerBufferArgumentHelper<T, typename std::enable_if<std::is_base_of<sf::detail::BufferBaseTag, T>::value>::type>
{
public:

    typedef T ReturnType;

    static T Convert(const cmif::PointerAndSize& pas) NN_NOEXCEPT
    {
        return pas;
    }

};

template <typename T>
class ServerBufferArgumentHelper<Out<T>>
{
public:

    typedef Out<T> ReturnType;

    static Out<T> Convert(const cmif::PointerAndSize& pas) NN_NOEXCEPT
    {
        return Out<T>(reinterpret_cast<T*>(pas.pointer));
    }

};

template <typename T>
inline typename ServerBufferArgumentHelper<typename std::decay<T>::type>::ReturnType GetBufferArgument(const cmif::PointerAndSize& pas) NN_NOEXCEPT
{
    return ServerBufferArgumentHelper<typename std::decay<T>::type>::Convert(pas);
}

// 自動生成系で CmifProcessFunctionTableGetterImpl を実装する際の、
// 基底となる実装用ユーティリティ。
// 内部で CmifProcessFunctionTableGetterImpl<Interface>::DispatchServerMessage を呼んでおり、
// 実質的な CRTP になっている。
template <typename Interface>
struct CmifProcessFunctionTableGetterImplBase
{
    static nn::Result ProcessServerMessage(IServiceObject* p, CmifServerMessage* message, const PointerAndSize& rawData) NN_NOEXCEPT
    {
        auto inRawData = rawData;

        NN_SF_CMIF_SERVER_DETAIL_GET_POINTER_CHECKED_OR_THROW_RESULT(
            (CmifInHeader), inHeader, inRawData,
            ResultInvalidCmifHeaderSize()
        );

        NN_SF_RESULT_THROW_UNLESS(inHeader->Check(), ResultInvalidCmifInHeader());

        // outHeader にはとりあえずの値を入れておく。
        CmifOutHeader* pOutHeader = nullptr;
        auto result = CmifProcessFunctionTableGetterImpl<Interface>::DispatchServerMessage(&pOutHeader, static_cast<Interface*>(p), message, inHeader->methodId, std::move(inRawData));
        if (nn::sf::detail::ResultContextControl::Includes(result))
        {
            return result;
        }
        if (!pOutHeader)
        {
            NN_ABORT_UNLESS(!result.IsSuccess());
            return result;
        }

        // inHeader と outHeader の領域が重複している可能性があるため、
        // 直接 outHeader に Set はせず、
        // tmpOutHeader に一旦作成したものを、memcpy する。
        CmifOutHeader tmpOutHeader;
        tmpOutHeader.Set(result);
        std::memcpy(pOutHeader, &tmpOutHeader, sizeof(tmpOutHeader));

        message->EndPreparingForReply();

        NN_SF_RESULT_SUCCESS;
    }
};

inline bool CheckAppletResourceUserId(Bit64 processId, Bit64 value) NN_NOEXCEPT
{
    // TORIAEZU: 0 を通すようにしておく
    if (value == 0)
    {
        return true;
    }
    return value == processId;
}

}}}}}

// nn::sf::cmif::server::detail::CmifProcessFunctionTableGetterImpl<I> 実装の
// 自動生成系向けのマクロ群

#define NN_SF_CMIF_SERVER_DETAIL_DEFINE_DISPATCHER_BASE(type) \
    namespace nn { namespace sf { namespace cmif { namespace server { namespace detail { \
    \
    /* インテリセンスでのエラー防止のためにプライマリテンプレートを再度宣言 */ \
    template <typename Interface> \
    struct CmifProcessFunctionTableGetterImpl; \
    \
    template <> \
    struct CmifProcessFunctionTableGetterImpl<NN_SF_DETAIL_STRIP_PAREN(type)> \
        : public ::nn::sf::cmif::server::detail::CmifProcessFunctionTableGetterImplBase<NN_SF_DETAIL_STRIP_PAREN(type)> \

#define NN_SF_CMIF_SERVER_DETAIL_DEFINE_DISPATCHER_BEGIN(type) \
    { \
        typedef NN_SF_DETAIL_STRIP_PAREN(type) _tInterface;

#define NN_SF_CMIF_SERVER_DETAIL_DEFINE_DISPATCHER_TYPEDEF_BASE(baseType) \
        typedef NN_SF_DETAIL_STRIP_PAREN(baseType) _tBaseInterface;

#define NN_SF_CMIF_SERVER_DETAIL_DEFINE_DISPATCHER(type) \
    NN_SF_CMIF_SERVER_DETAIL_DEFINE_DISPATCHER_BASE(type) \
    NN_SF_CMIF_SERVER_DETAIL_DEFINE_DISPATCHER_BEGIN(type)

#define NN_SF_CMIF_SERVER_DETAIL_DEFINE_DISPATCHER_WITH_BASE(type, base) \
    NN_SF_CMIF_SERVER_DETAIL_DEFINE_DISPATCHER_BASE(type) \
    NN_SF_CMIF_SERVER_DETAIL_DEFINE_DISPATCHER_BEGIN (type)\
    NN_SF_CMIF_SERVER_DETAIL_DEFINE_DISPATCHER_TYPEDEF_BASE(base)

    #define NN_SF_CMIF_SERVER_DETAIL_DEFINE_DISPATCH_METHOD(name) \
        static ::nn::Result Process_##name(::nn::sf::cmif::CmifOutHeader** _ppOut, _tInterface* _p, ::nn::sf::cmif::server::CmifServerMessage* _message, ::nn::sf::cmif::PointerAndSize&& _inRawData) NN_NOEXCEPT \
        { \
            typedef ::nn::sf::cmif::detail::MethodInfos<_tInterface>::name _methodInfo; \

        #define NN_SF_CMIF_SERVER_DETAIL_DEFINE_DISPATCH_METHOD_PREPARE \
            NN_SF_DETAIL_CONSTEXPR auto _metaInfo = ::nn::sf::cmif::detail::MakeCmifMessageMetaInfoFromMethodInfo<_methodInfo>(); \
            ::std::array<::nn::sf::cmif::PointerAndSize, _methodInfo::BufferCount> _buffers;

            #define NN_SF_CMIF_SERVER_DETAIL_DEFINE_DISPATCH_METHOD_PREPARE_BUFFERS \
                { \

                #define NN_SF_CMIF_SERVER_DETAIL_DEFINE_DISPATCH_METHOD_PREPARE_BUFFER(name) \
                    _buffers[_methodInfo::BufferInfos::name::Index].size = _methodInfo::BufferInfos::name::FixedLength;

            #define NN_SF_CMIF_SERVER_DETAIL_DEFINE_DISPATCH_METHOD_PREPARE_BUFFERS_END \
                }

            #define NN_SF_CMIF_SERVER_DETAIL_DEFINE_DISPATCH_METHOD_PREPARE_IN_NATIVE_HANDLES \
                { \

                #define NN_SF_CMIF_SERVER_DETAIL_DEFINE_DISPATCH_METHOD_PREPARE_IN_NATIVE_HANDLE(name) \

            #define NN_SF_CMIF_SERVER_DETAIL_DEFINE_DISPATCH_METHOD_PREPARE_IN_NATIVE_HANDLES_END \
                }

            #define NN_SF_CMIF_SERVER_DETAIL_DEFINE_DISPATCH_METHOD_PREPARE_OUT_NATIVE_HANDLES \
                { \

                #define NN_SF_CMIF_SERVER_DETAIL_DEFINE_DISPATCH_METHOD_PREPARE_OUT_NATIVE_HANDLE(name) \

            #define NN_SF_CMIF_SERVER_DETAIL_DEFINE_DISPATCH_METHOD_PREPARE_OUT_NATIVE_HANDLES_END \
                }

        #define NN_SF_CMIF_SERVER_DETAIL_DEFINE_DISPATCH_METHOD_PREPARE_END \
            NN_SF_RESULT_DO(_message->PrepareForProcess(_metaInfo)); \
            \
            NN_SF_CMIF_SERVER_DETAIL_GET_POINTER_CHECKED_OR_THROW_RESULT( \
                (_methodInfo::InRawStruct), _inRawRef, _inRawData, \
                ::nn::sf::cmif::ResultInvalidInRawSize() \
            ); \
            _methodInfo::InRawStruct _inRawBuffer; \
            ::std::memcpy(&_inRawBuffer, _inRawRef, sizeof(_inRawBuffer)); \
            auto _inRaw = &_inRawBuffer; \
            NN_UNUSED(_inRaw); \
            ::nn::sf::cmif::server::detail::InOutObjects<_methodInfo> _inOutObjects; \
            NN_SF_RESULT_DO(_inOutObjects.GetMessageObjectsInfo(*_message)); \
            ::nn::sf::detail::SafeArray<::nn::sf::NativeHandle, _methodInfo::OutNativeHandleCount> _outNativeHandles; \
            NN_UNUSED(_outNativeHandles); \
            ::nn::sf::cmif::server::detail::OutRawHolder<_methodInfo> _outRaw; \
            ::nn::sf::detail::SafeArray<::nn::sf::NativeHandle, _methodInfo::InNativeHandleCount> _inNativeHandles; \
            if (NN_STATIC_CONDITION(_methodInfo::InNativeHandleCount > 0)) \
            { \
                NN_SF_RESULT_DO(_message->GetInNativeHandles(_inNativeHandles.Get())); \
            } \
            if (NN_STATIC_CONDITION(_methodInfo::BufferCount > 0)) \
            { \
                NN_SF_RESULT_DO(_message->GetBuffers(_buffers.data())); \
            } \
            ::nn::Result _result = ::nn::ResultSuccess();

        #define NN_SF_CMIF_SERVER_DETAIL_DEFINE_DISPATCH_METHOD_DEFINE_IN_OBJECT(type, name, i) \
            NN_SF_DETAIL_STRIP_PAREN(type) name; \
            NN_SF_RESULT_DO(_inOutObjects.GetInObjectMoved(&name, i));

        #define NN_SF_CMIF_SERVER_DETAIL_DEFINE_DISPATCH_METHOD_DEFINE_OUT_OBJECT(type, name) \
            NN_SF_DETAIL_STRIP_PAREN(type) name;

        #define NN_SF_CMIF_SERVER_DETAIL_DEFINE_DISPATCH_METHOD_OVERWRITE_IN_PROCESS_ID(memberName) \
            NN_SF_RESULT_DO(_message->OverwriteClientProcessId(&_inRaw->memberName));

        #define NN_SF_CMIF_SERVER_DETAIL_DEFINE_DISPATCH_METHOD_APPLY_PROCESS_ID(arg, result, target) \
            ([&_message](const ::std::decay<decltype((target))>::type& arg) -> ::nn::Result \
            { \
                ::nn::Bit64 _processId; \
                NN_SF_RESULT_DO(_message->OverwriteClientProcessId(&_processId)); \
                return ::nn::sf::detail::ToPreConditionResult(result); \
            })(target) \

        #define NN_SF_CMIF_SERVER_DETAIL_DEFINE_DISPATCH_METHOD_CHECK_PRE(c) \
            if (_result.IsSuccess()) \
            { \
                _result = ::nn::sf::detail::ToPreConditionResult(c); \
            }

        #define NN_SF_CMIF_SERVER_DETAIL_DEFINE_DISPATCH_METHOD_INVOKE(name) \
            if (_result.IsSuccess()) \
            { \
                _result = _p->NN_SF_DETAIL_SYNC_METHOD_NAME(name)(

            #define NN_SF_CMIF_SERVER_DETAIL_DEFINE_DISPATCH_METHOD_IN_MEMBER(memberName) \
                _inRaw->memberName

            #define NN_SF_CMIF_SERVER_DETAIL_DEFINE_DISPATCH_METHOD_OUT_MEMBER(memberName) \
                ::std::addressof(_outRaw->memberName)

            #define NN_SF_CMIF_SERVER_DETAIL_DEFINE_DISPATCH_METHOD_OUT_RETURN_MEMBER \
                NN_SF_CMIF_SERVER_DETAIL_DEFINE_DISPATCH_METHOD_OUT_MEMBER(_pOut)

            #define NN_SF_CMIF_SERVER_DETAIL_DEFINE_DISPATCH_METHOD_BUFFER(name) \
                ::nn::sf::cmif::server::detail::GetBufferArgument<typename _methodInfo::BufferInfos::name::Type>(_buffers[_methodInfo::BufferInfos::name::Index])

            #define NN_SF_CMIF_SERVER_DETAIL_DEFINE_DISPATCH_METHOD_IN_OBJECT(memberName) \
                ::std::move(memberName)

            #define NN_SF_CMIF_SERVER_DETAIL_DEFINE_DISPATCH_METHOD_OUT_OBJECT(memberName) \
                &memberName

            #define NN_SF_CMIF_SERVER_DETAIL_DEFINE_DISPATCH_METHOD_IN_NATIVE_HANDLE(memberName) \
                ::std::move(_inNativeHandles.Get()[_methodInfo::InNativeHandleInfos::memberName::Index])

            #define NN_SF_CMIF_SERVER_DETAIL_DEFINE_DISPATCH_METHOD_OUT_NATIVE_HANDLE(memberName) \
                (_outNativeHandles.Get() + _methodInfo::OutNativeHandleInfos::memberName::Index)

        #define NN_SF_CMIF_SERVER_DETAIL_DEFINE_DISPATCH_METHOD_INVOKE_END \
                ); \
            } \
            if (::nn::sf::detail::ResultContextControl::Includes(_result)) \
            { \
                return _result; \
            } \
            if (!_result.IsSuccess()) \
            { \
                ::nn::sf::cmif::PointerAndSize _outRawData; \
                _message->BeginPreparingForErrorReply(&_outRawData, sizeof(::nn::sf::cmif::CmifOutHeader)); \
                NN_SF_CMIF_SERVER_DETAIL_GET_POINTER_CHECKED_OR_THROW_RESULT( \
                    (::nn::sf::cmif::CmifOutHeader), _outHeader, _outRawData, \
                    ::nn::sf::cmif::ResultInvalidCmifHeaderSize() \
                ); \
                *_ppOut = _outHeader; \
                return _result; \
            } \
            ::nn::sf::cmif::PointerAndSize _outRawData; \
            _message->BeginPreparingForReply(&_outRawData); \
            NN_SF_CMIF_SERVER_DETAIL_GET_POINTER_CHECKED_OR_THROW_RESULT( \
                (::nn::sf::cmif::CmifOutHeader), _outHeader, _outRawData, \
                ::nn::sf::cmif::ResultInvalidCmifHeaderSize() \
            ); \
            *_ppOut = _outHeader; \
            NN_SF_CMIF_SERVER_DETAIL_GET_POINTER_CHECKED_OR_THROW_RESULT( \
                (_methodInfo::OutRawStruct), _outRawPointer, _outRawData, \
                ::nn::sf::cmif::ResultInvalidOutRawSize() \
            ); \

        #define NN_SF_CMIF_SERVER_DETAIL_DEFINE_DISPATCH_METHOD_SET_OUT_OBJECT(name, i) \
            _inOutObjects.SetOutObject(::std::move(name), i);

    #define NN_SF_CMIF_SERVER_DETAIL_DEFINE_DISPATCH_METHOD_END \
            if (NN_STATIC_CONDITION(_methodInfo::BufferCount > 0)) \
            { \
                _message->SetBuffers(_buffers.data()); \
            } \
            if (NN_STATIC_CONDITION(_methodInfo::OutNativeHandleCount > 0)) \
            { \
                _message->SetOutNativeHandles(_outNativeHandles.Get()); \
            } \
            _inOutObjects.WriteBack(_message); \
            _outRaw.WriteBack(_outRawPointer); \
            NN_SF_RESULT_SUCCESS; \
        }

    #define NN_SF_CMIF_SERVER_DETAIL_DEFINE_DISPATCH_MESSAGE \
        static ::nn::Result DispatchServerMessage(::nn::sf::cmif::CmifOutHeader** _ppOut, _tInterface* _p, ::nn::sf::cmif::server::CmifServerMessage* _message, ::nn::sf::cmif::MethodId methodId, ::nn::sf::cmif::PointerAndSize&& _inRawData) NN_NOEXCEPT \
        { \
            switch (methodId) \
            { \

        #define NN_SF_CMIF_SERVER_DETAIL_DEFINE_DISPATCH_MESSAGE_CASE(name) \
                case ::nn::sf::cmif::detail::MethodInfos<_tInterface>::name::MethodId: \
                { \
                    return Process_##name(_ppOut, _p, _message, ::std::move(_inRawData)); \
                }

    #define NN_SF_CMIF_SERVER_DETAIL_DEFINE_DISPATCH_MESSAGE_END_COMMON(defaultProcess) \
            case ::nn::sf::cmif::InvalidMethodId: \
            default: \
            { \
                NN_UNUSED(_ppOut); \
                NN_UNUSED(_p); \
                NN_UNUSED(_message); \
                NN_UNUSED(_inRawData); \
                defaultProcess \
            } \
        }

    #define NN_SF_CMIF_SERVER_DETAIL_DEFINE_DISPATCH_MESSAGE_END \
        NN_SF_CMIF_SERVER_DETAIL_DEFINE_DISPATCH_MESSAGE_END_COMMON(NN_SF_RESULT_THROW(::nn::sf::cmif::ResultUnknownMethodId());)

    #define NN_SF_CMIF_SERVER_DETAIL_DEFINE_DISPATCH_MESSAGE_END_WITH_BASE \
        NN_SF_CMIF_SERVER_DETAIL_DEFINE_DISPATCH_MESSAGE_END_COMMON(return ::nn::sf::cmif::server::detail::CmifProcessFunctionTableGetterImpl<_tBaseInterface>::DispatchServerMessage(_ppOut, _p, _message, methodId, std::move(_inRawData));)

#define NN_SF_CMIF_SERVER_DETAIL_DEFINE_DISPATCHER_END \
        } \
    }; \
    \
    }}}}}

#define NN_SF_CMIF_SERVER_DETAIL_DEFINE_DISPATCHER_FUNCTION_TABLE(type) \
    namespace nn { namespace sf { namespace cmif { namespace server { namespace detail { \
    \
    template <typename Tag> \
    struct CmifProcessFunctionTableGetter<NN_SF_DETAIL_STRIP_PAREN(type), Tag> \
    { \
        static const ::nn::sf::cmif::server::detail::CmifProcessFunctionTable s_Table; \
    }; \
    \
    template <typename Tag> \
    const ::nn::sf::cmif::server::detail::CmifProcessFunctionTable CmifProcessFunctionTableGetter<NN_SF_DETAIL_STRIP_PAREN(type), Tag>::s_Table = \
    { \
        &::nn::sf::cmif::server::detail::CmifProcessFunctionTableGetterImpl<NN_SF_DETAIL_STRIP_PAREN(type)>::ProcessServerMessage, \
    }; \
    \
    }}}}}
