﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <type_traits>
#include <nn/sf/detail/sf_CommonUtil.h>
#include <nn/sf/cmif/sf_CmifCommonFormat.h>
#include <nn/sf/cmif/sf_CmifMessageCommon.h>

namespace nn { namespace sf { namespace cmif { namespace detail {

#define NN_SF_CMIF_DETAIL_BUFFER_ATTRIBUTE_IN (::nn::sf::cmif::BufferAttribute_In)
#define NN_SF_CMIF_DETAIL_BUFFER_ATTRIBUTE_OUT (::nn::sf::cmif::BufferAttribute_Out)
#define NN_SF_CMIF_DETAIL_BUFFER_ATTRIBUTE_HIPC_MAP_ALIAS (::nn::sf::cmif::BufferAttribute_HipcMapAlias)
#define NN_SF_CMIF_DETAIL_BUFFER_ATTRIBUTE_HIPC_POINTER (::nn::sf::cmif::BufferAttribute_HipcPointer)
#define NN_SF_CMIF_DETAIL_BUFFER_ATTRIBUTE_HIPC_AUTO_SELECT (::nn::sf::cmif::BufferAttribute_HipcAutoSelect)
#define NN_SF_CMIF_DETAIL_BUFFER_ATTRIBUTE_HIPC_MAP_NON_SECURE (::nn::sf::cmif::BufferAttribute_HipcMapTransferAllowsNonSecure)
#define NN_SF_CMIF_DETAIL_BUFFER_ATTRIBUTE_HIPC_MAP_NON_DEVICE (::nn::sf::cmif::BufferAttribute_HipcMapTransferAllowsNonDevice)
#define NN_SF_CMIF_DETAIL_NATIVE_HANDLE_ATTRIBUTE_HIPC_COPY (::nn::sf::cmif::NativeHandleAttribute_HipcCopy)
#define NN_SF_CMIF_DETAIL_NATIVE_HANDLE_ATTRIBUTE_HIPC_MOVE (::nn::sf::cmif::NativeHandleAttribute_HipcMove)

template <int Index_, typename T, int Attribute_, size_t FixedLength_ = 0>
struct BufferInfoBase
{
    static const int Index = Index_;
    static const int Attribute = Attribute_;
    static const size_t FixedLength = FixedLength_;
    typedef T Type;
};

template <int Index_, typename T, int Attribute_, size_t FixedLength_>
const int BufferInfoBase<Index_, T, Attribute_, FixedLength_>::Index;

template <int Index_, typename T, int Attribute_, size_t FixedLength_>
const int BufferInfoBase<Index_, T, Attribute_, FixedLength_>::Attribute;

template <int Index_, typename T, int Attribute_, size_t FixedLength_>
const size_t BufferInfoBase<Index_, T, Attribute_, FixedLength_>::FixedLength;

template <int Index_, int Attribute_>
struct NativeHandleInfoBase
{
    static const int Index = Index_;
    static const int Attribute = Attribute_;
};

template <int Index_, int Attribute_>
const int NativeHandleInfoBase<Index_, Attribute_>::Index;

template <int Index_, int Attribute_>
struct ObjectInfoBase
{
    static const int Index = Index_;
};

template <int Index_, int Attribute_>
const int ObjectInfoBase<Index_, Attribute_>::Index;

// Interface に対応するメソッド情報を持つ構造体の定義用テンプレート。
// Interface ごとに明示的特殊化をして定義する。
template <typename Interface>
struct MethodInfos;

struct MethodInfoBase
{
    static const bool ReturnTypeIsVoid = false;
    static const bool ReturnTypeIsResult = false;
    static const bool InProcessIdEnable = false;
    static const int BufferCount = 0;
    static const int InObjectCount = 0;
    static const int OutObjectCount = 0;
    static const int InNativeHandleCount = 0;
    static const int OutNativeHandleCount = 0;
    struct InRawStruct
    {
    };
    static const size_t InRawSize = 0;
    struct OutRawStruct
    {
    };
    static const size_t OutRawSize = 0;
    struct BufferInfos
    {
        NN_SF_DETAIL_DEFINE_TYPE_LIST(_typelist, ());
    };
    struct InNativeHandleInfos
    {
        NN_SF_DETAIL_DEFINE_TYPE_LIST(_typelist, ());
    };
    struct OutNativeHandleInfos
    {
        NN_SF_DETAIL_DEFINE_TYPE_LIST(_typelist, ());
    };
    struct InObjectInfos
    {
        NN_SF_DETAIL_DEFINE_TYPE_LIST(_typelist, ());
    };
    struct OutObjectInfos
    {
        NN_SF_DETAIL_DEFINE_TYPE_LIST(_typelist, ());
    };
    using ArgumentInfos = void;
};

// Impl の定数を元に値の検証と定数の定義を行うためのユーティリティ
template <typename Impl>
struct MethodInfo
{
    static_assert(Impl::InRawSize == (std::is_empty<typename Impl::InRawStruct>::value ? 0 : sizeof(typename Impl::InRawStruct)), "[SF-CMIF-Unexpected]");
    static_assert(Impl::OutRawSize == (std::is_empty<typename Impl::OutRawStruct>::value ? 0 : sizeof(typename Impl::OutRawStruct)), "[SF-CMIF-Unexpected]");
    static_assert(!(Impl::ReturnTypeIsVoid && Impl::ReturnTypeIsResult), "[SF-CMIF-Unexpected]");

    static const nn::sf::cmif::MethodId MethodId = Impl::MethodId;
    static const bool InProcessIdEnable = Impl::InProcessIdEnable;
    static const std::size_t InRawSize = Impl::InRawSize;
    static const std::size_t OutRawSize = Impl::OutRawSize;
    static const int BufferCount = Impl::BufferCount;
    static const int InObjectCount = Impl::InObjectCount;
    static const int OutObjectCount = Impl::OutObjectCount;
    static const int InNativeHandleCount = Impl::InNativeHandleCount;
    static const int OutNativeHandleCount = Impl::OutNativeHandleCount;
    static const bool ReturnTypeIsVoid = Impl::ReturnTypeIsVoid;
    static const bool ReturnTypeIsResult = Impl::ReturnTypeIsResult;
    typedef typename Impl::InRawStruct InRawStruct;
    typedef typename Impl::OutRawStruct OutRawStruct;
    typedef typename Impl::BufferInfos BufferInfos;
    typedef typename Impl::InNativeHandleInfos InNativeHandleInfos;
    typedef typename Impl::OutNativeHandleInfos OutNativeHandleInfos;
    typedef typename Impl::InObjectInfos InObjectInfos;
    typedef typename Impl::OutObjectInfos OutObjectInfos;
    using ArgumentInfos = typename Impl::ArgumentInfos;
};

template <typename Impl>
const nn::sf::cmif::MethodId MethodInfo<Impl>::MethodId;

template <typename Impl>
const bool MethodInfo<Impl>::InProcessIdEnable;

template <typename Impl>
const std::size_t MethodInfo<Impl>::InRawSize;

template <typename Impl>
const std::size_t MethodInfo<Impl>::OutRawSize;

template <typename Impl>
const int MethodInfo<Impl>::BufferCount;

template <typename Impl>
const int MethodInfo<Impl>::InObjectCount;

template <typename Impl>
const int MethodInfo<Impl>::OutObjectCount;

template <typename Impl>
const int MethodInfo<Impl>::InNativeHandleCount;

template <typename Impl>
const int MethodInfo<Impl>::OutNativeHandleCount;

template <typename Impl>
const bool MethodInfo<Impl>::ReturnTypeIsVoid;

template <typename Impl>
const bool MethodInfo<Impl>::ReturnTypeIsResult;

struct AttributeSetter
{
    template <typename T, typename Target>
    NN_SF_DETAIL_CONSTEXPR static void Set(Target& x, int N) NN_NOEXCEPT
    {
        x[N] = T::Attribute;
    };
};

template <typename MethodInfo>
NN_SF_DETAIL_CONSTEXPR inline CmifMessageMetaInfo MakeCmifMessageMetaInfoFromMethodInfo() NN_NOEXCEPT
{
    CmifMessageMetaInfo ret = {};
    ret.inProcessIdEnable = MethodInfo::InProcessIdEnable;
    ret.inRawDataSize = sizeof(CmifInHeader) + MethodInfo::InRawSize;
    ret.outRawDataSize = sizeof(CmifOutHeader) + MethodInfo::OutRawSize;
    ret.bufferCount = MethodInfo::BufferCount;
    ret.inObjectCount = MethodInfo::InObjectCount;
    ret.outObjectCount = MethodInfo::OutObjectCount;
    ret.inNativeHandleCount = MethodInfo::InNativeHandleCount;
    ret.outNativeHandleCount = MethodInfo::OutNativeHandleCount;
    sf::detail::TypeListSetter<AttributeSetter, typename MethodInfo::BufferInfos::_typelist>::Set(ret.bufferAttributes);
    sf::detail::TypeListSetter<AttributeSetter, typename MethodInfo::InNativeHandleInfos::_typelist>::Set(ret.inNativeHandleAttributes);
    sf::detail::TypeListSetter<AttributeSetter, typename MethodInfo::OutNativeHandleInfos::_typelist>::Set(ret.outNativeHandleAttributes);
    return ret;
}

enum class ArgumentKind
{
    InRaw,
    OutRaw,
    Buffer,
    InHandle,
    OutHandle,
    InObject,
    OutObject,
};

template <ArgumentKind Kind_>
struct ArgumentInfoBase
{
    using Kind = NN_SF_DETAIL_INTEGRAL_CONSTANT(Kind_);
};

template <ArgumentKind Kind_, size_t Size_, size_t Alignment_, ptrdiff_t RawOffset_>
struct RawArgumentInfoBase
    : public ArgumentInfoBase<Kind_>
{
    using Size = NN_SF_DETAIL_INTEGRAL_CONSTANT(Size_);
    using Alignment = NN_SF_DETAIL_INTEGRAL_CONSTANT(Alignment_);
    using RawOffset = NN_SF_DETAIL_INTEGRAL_CONSTANT(RawOffset_);
};

template <size_t Size_, size_t Alignment_, ptrdiff_t RawOffset_>
struct InRawArgumentInfo
    : public RawArgumentInfoBase<ArgumentKind::InRaw, Size_, Alignment_, RawOffset_>
{
};

template <size_t Size_, size_t Alignment_, ptrdiff_t RawOffset_>
struct OutRawArgumentInfo
    : public RawArgumentInfoBase<ArgumentKind::OutRaw, Size_, Alignment_, RawOffset_>
{
};

template <ArgumentKind Kind_, int Index_>
struct IndexedArgumentInfoBase
    : public ArgumentInfoBase<Kind_>
{
    using Index = NN_SF_DETAIL_INTEGRAL_CONSTANT(Index_);
};

template <int Index_, int Attribute_, size_t FixedLength_ = 0>
struct BufferArgumentInfo
    : public IndexedArgumentInfoBase<ArgumentKind::Buffer, Index_>
{
    using Attribute = NN_SF_DETAIL_INTEGRAL_CONSTANT(Attribute_);
    using FixedLength = NN_SF_DETAIL_INTEGRAL_CONSTANT(FixedLength_);
};

template <int Index_, int Attribute_>
struct InHandleArgumentInfo
    : public IndexedArgumentInfoBase<ArgumentKind::InHandle, Index_>
{
    using Attribute = NN_SF_DETAIL_INTEGRAL_CONSTANT(Attribute_);
};

template <int Index_, int Attribute_>
struct OutHandleArgumentInfo
    : public IndexedArgumentInfoBase<ArgumentKind::OutHandle, Index_>
{
    using Attribute = NN_SF_DETAIL_INTEGRAL_CONSTANT(Attribute_);
};

template <int Index_, int Attribute_>
struct InObjectArgumentInfo
    : public IndexedArgumentInfoBase<ArgumentKind::InObject, Index_>
{
    using Attribute = NN_SF_DETAIL_INTEGRAL_CONSTANT(Attribute_);
};

template <int Index_, int Attribute_>
struct OutObjectArgumentInfo
    : public IndexedArgumentInfoBase<ArgumentKind::OutObject, Index_>
{
    using Attribute = NN_SF_DETAIL_INTEGRAL_CONSTANT(Attribute_);
};

template <typename T, typename ArgumentInfo_>
using ArgumentInfoWithType = ArgumentInfo_;

template <ArgumentKind Kind>
struct GetHasArgumentKind
{
    template <typename ArgumentInfo>
    struct predicate
        : public std::integral_constant<bool, ArgumentInfo::Kind::value == Kind>
    {
    };
};

template <typename MethodInfo, typename Enable = void>
struct GetCoreMethodInfo
{
    struct type {};
};

template <typename ArgumentInfos_, size_t InRawSize_, size_t OutRawSize_, bool InProcessIdEnable_>
struct CoreMethodInfo
{
    using InRawSize = NN_SF_DETAIL_INTEGRAL_CONSTANT(InRawSize_);
    using OutRawSize = NN_SF_DETAIL_INTEGRAL_CONSTANT(OutRawSize_);
    using InProcessIdEnable = NN_SF_DETAIL_INTEGRAL_CONSTANT(InProcessIdEnable_);

    using ArgumentInfos = ArgumentInfos_;

    template <ArgumentKind Kind>
    using GetArgumentCountT = typename sf::detail::TypeCountIf<GetHasArgumentKind<Kind>, ArgumentInfos>::type;

    using InRawCount = GetArgumentCountT<ArgumentKind::InRaw>;
    using OutRawCount = GetArgumentCountT<ArgumentKind::OutRaw>;
    using BufferCount = GetArgumentCountT<ArgumentKind::Buffer>;
    using InHandleCount = GetArgumentCountT<ArgumentKind::InHandle>;
    using OutHandleCount = GetArgumentCountT<ArgumentKind::OutHandle>;
    using InObjectCount = GetArgumentCountT<ArgumentKind::InObject>;
    using OutObjectCount = GetArgumentCountT<ArgumentKind::OutObject>;

    template <ArgumentKind Kind>
    using GetArgumentInfoByKind = typename sf::detail::TypeFilter<GetHasArgumentKind<Kind>, ArgumentInfos>::type;

    using InRaws = GetArgumentInfoByKind<ArgumentKind::InRaw>;
    using OutRaws = GetArgumentInfoByKind<ArgumentKind::OutRaw>;
    using Buffers = GetArgumentInfoByKind<ArgumentKind::Buffer>;
    using InHandles = GetArgumentInfoByKind<ArgumentKind::InHandle>;
    using OutHandles = GetArgumentInfoByKind<ArgumentKind::OutHandle>;
    using InObjects = GetArgumentInfoByKind<ArgumentKind::InObject>;
    using OutObjects = GetArgumentInfoByKind<ArgumentKind::OutObject>;
};

template <typename MethodInfo>
struct GetCoreMethodInfo<MethodInfo, typename std::enable_if<!std::is_void<typename MethodInfo::ArgumentInfos>::value>::type>
{
    using ArgumentInfos = typename MethodInfo::ArgumentInfos;
    using type = CoreMethodInfo<
        ArgumentInfos,
        MethodInfo::InRawSize,
        MethodInfo::OutRawSize,
        MethodInfo::InProcessIdEnable
    >;
};

}}}}
