﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

/**
 * @file
 * @brief   シェル向けの API を宣言するファイルです
 */

#include <nn/nn_Result.h>
#include <nn/os/os_Types.h>
#include <nn/os/os_SystemEvent.h>
#include <nn/ncm/ncm_ProgramId.h>
#include <nn/ncm/ncm_ProgramLocation.h>

/**
 * @namespace   nn::pm
 * @brief       プロセス管理ライブラリの名前空間です。
 */
namespace nn { namespace pm {

    namespace detail
    {
        const char PortNameForShell[] = "pm:shell";
    }

    /**
        @brief  プロセス起動時に指定できるプロセスのオプションを表します
    */
    enum LaunchProgramFlags
    {
        LaunchProgramFlags_None             = 0,            //!< フラグを指定しないことを表します
        LaunchProgramFlags_NotifyExit       = (1 << 0),     //!< 作成したプロセスが終了した時に ProcessEvent_Exit を通知することを指定します
        LaunchProgramFlags_NotifyStart      = (1 << 1),     //!< 対象のプログラムが実行開始した時に ProcessEvent_Started を通知することを指定します
        LaunchProgramFlags_NotifyException  = (1 << 2),     //!< プロセス例外が発生した時に ProcessEvent_Exception を通知することを指定します
        LaunchProgramFlags_NotifyDebug      = (1 << 3),     //!< 対象のプログラムがデバッガ操作を行われたときに ProcessEvent_Debug* を通知することを指定します
        LaunchProgramFlags_NotStart         = (1 << 4),     //!< プロセスを作成するが実行開始しないことを指定します
        LaunchProgramFlags_DisableAslr      = (1 << 5)      //!< ASLR を無効化してプログラムを起動することを指定します
    };

    /**
        @brief  実行中のプロセスで発生したプロセスにかかわるイベントを表します
    */
    enum ProcessEvent
    {
        ProcessEvent_None,              //!< 何もイベントは発生していないことを表します
        ProcessEvent_Exit,              //!< プロセスが終了したことを表します
        ProcessEvent_Started,           //!< プロセスが実行を開始したことを表します
        ProcessEvent_Exception,         //!< プロセス例外が発生したことを表します
        ProcessEvent_DebugRunning,      //!< プロセスがデバッガによって再開されたことを表します
        ProcessEvent_DebugBreaked       //!< プロセスがデバッガによって停止させられたことを表します
    };

    /**
        @brief  実行中のプロセスで発生したプロセスにかかわるイベントの情報を格納します。
    */
    struct ProcessEventInfo
    {
        ProcessEvent    event;          //!< 発生したイベント
        os::ProcessId   processId;      //!< イベントが発生したプロセスのプロセス ID
    };

    Result InitializeForShell() NN_NOEXCEPT;
    Result FinalizeForShell() NN_NOEXCEPT;



//! @name プロセス作成
//! @{

    /**
        @brief  プロセスを作成します

        @param[out] pOut        作成されたプロセスのプロセス ID を格納するバッファへのポインタ
        @param[in]  location    プロセスの作成元となるプログラムの指定
        @param[in]  flags       プロセス作成のオプション
                                LaunchProgramFlags のビットごとの OR で指定します

        @retresult
            @handleresult{nn::os::Result,}
        @endretresult

        @pre
            - pOut が有効なバッファを指していること

        @post
            - pOut が指すバッファに作成されたプロセスのプロセス ID が格納されている

        @details
            location で指定されたプログラムから、flags で指定されるフラグを使用しつつプロセスを作成します。
            プロセスの作成に成功した場合、pOut が指すバッファに作成されたプロセスのプロセス ID が格納されます。
    */
    Result LaunchProgram(nn::os::ProcessId* pOut, const ncm::ProgramLocation& location, int flags) NN_NOEXCEPT;

//! @}



//! @name プロセス操作
//! @{

    /**
        @brief  プロセスを強制終了します

        @param[in]  processId   強制終了する対象のプロセスを指すプロセス ID

        @retresult
            @handleresult{nn::os::Result,}
        @endretresult

        @post
            - processId が指定するプロセスが終了している

        @details
            processId で指定するプロセス ID のプロセスを終了させます
    */
    Result TerminateProcess(os::ProcessId processId) NN_NOEXCEPT;


    /**
        @brief  プログラムを強制終了します

        @param[in]  programId   強制終了する対象のプログラムを指すプログラム ID

        @retresult
            @handleresult{nn::os::Result,}
        @endretresult

        @post
            - 起動元のプログラムのプログラム ID が programId であるプロセスのうち未終了のものが存在しない

        @details
            programId で指定するプログラム ID のプログラムから作成されたすべてのプロセスを終了させます
    */
    Result TerminateProgram(ncm::ProgramId programId) NN_NOEXCEPT;

//! @}



//! @name プロセスイベント
//! @{

    /**
        @brief  プロセスイベントの発生を待機するための nn::os::SystemEvent を取得します

        @param[out] pOut    取得した nn::os::SystemEvent を格納するバッファへのポインタ

        @retresult
            @handleresult{nn::os::Result,}
        @endretresult

        @pre
            - pOut が有効なバッファへのポインタである

        @post
            - pOut が指すバッファに初期化済みの nn::os::SystemEvent が格納されている

        @details
            pOut には未初期化の nn::os::SystemEvent へのポインタを指定します。
            取得した nn::os::SystemEvent に対して Clear を行わないでください。

            発生したプロセスイベントは GetProcessEventInfo() で取得できます。
    */
    Result GetProcessEventEvent(os::SystemEvent* pOut) NN_NOEXCEPT;


    /**
        @brief  発生したプロセスイベントを取得します。

        @param[out] pOut    取得したプロセスイベントを格納するバッファへのポインタ

        @retresult
            @handleresult{nn::os::Result,}
        @endretresult

        @pre
            - pOut が有効なバッファへのポインタである

        @post
            - pOut が指すバッファに有効な値が格納されている

        @details
            未取得のプロセスイベントのうちの 1 つを取得します。

            pOut には未初期化の ProcessEventInfo へのポインタを指定します。
            未取得のプロセスイベントが存在しない場合は pOut->event に @ref ProcessEvent_None が格納されます。

            プロセスイベントの発生は GetProcessEventEvent() で取得できる nn::os::SystemEvent を待機することで待機できます。
    */
    Result GetProcessEventInfo(ProcessEventInfo* pOut) NN_NOEXCEPT;

//! @}


    Result GetApplicationProcessIdForShell(os::ProcessId* pOut) NN_NOEXCEPT;
    Result BoostSystemMemoryResourceLimit(int64_t boostSize) NN_NOEXCEPT;

    Result NotifyBootFinished() NN_NOEXCEPT;

}}  // namespace nn::pm
