﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * @file
 * @brief   共有メモリ機能に関する API の宣言
 */

#pragma once

#include <nn/nn_Common.h>
#include <nn/nn_Macro.h>
#include <nn/nn_Result.h>
#include <nn/os/os_Types.h>
#include <nn/os/os_NativeHandleTypes.h>
#include <nn/os/os_SharedMemoryTypes.h>


namespace nn { namespace os {

//! @name 共有メモリ関連 API
//! @{

//--------------------------------------------------------------------------
/**
 * @brief   共有メモリを作成します。
 *
 * @param[in] sharedMemory      共有メモリオブジェクトへのポインタ
 * @param[in] size              共有メモリのサイズ
 * @param[in] myPermission      自プロセスに許可されるメモリアクセス権
 * @param[in] otherPermission   他プロセスに許可されるメモリアクセス権
 *
 * @retresult
 *   @handleresult{nn::os::ResultOutOfMemory}
 *   @handleresult{nn::os::ResultOutOfResource}
 * @endretresult
 *
 * @pre
 *  - sharedMemory オブジェクトが未初期化状態もしくは初期化されていない状態である
 *  - size > 0
 *  - size が nn::os::MemoryPageSize の整数倍である
 *
 * @post
 *  - sharedMemory オブジェクトは初期化状態である
 *
 * @details
 *  共有メモリを生成し、共有メモリオブジェクトを初期化します。
 *
 *  この段階では、まだ共有メモリはメモリ空間上にマップされません。
 *  本 API で生成した共有メモリをメモリ空間上にマップするには MapSharedMemory()
 *  を使用し、アンマップするには UnmapSharedMemory() を使用します。@n
 *  共有メモリは 1 つの SharedMemoryType オブジェクトにつき、
 *  1 箇所にのみマップすることが出来ます。
 *
 *  共有メモリを生成する際、システムから共有メモリハンドルを 1 つ獲得します。
 *  このハンドルは GetSharedMemoryHandle() で取得することができ、
 *  他のプロセスに渡して AttachSharedMemory() を使用することで、
 *  同じ共有メモリを扱う SharedMemoryType オブジェクトを用意することが出来ます。
 *
 *  なお、 SharedMemoryType オブジェクトは、オブジェクトの状態とは別に、
 *  bool 型のハンドル管理フラグを保持しています。ハンドル管理フラグは、
 *  DestroySharedMemory() 時にハンドルを自動的にシステムに返却するか否かを
 *  示しており、本 API では必ず true に設定されます。
 *
 *  本 API の動作中は、対象共有メモリオブジェクトに対する操作を
 *  行なわないで下さい。
 *
 *
 *  なお、現在の共有メモリ関連 API は暫定仕様です。@n
 *  セキュリティの観点などから、将来的に互換性のない仕様変更が行なわれる
 *  可能性があります。
 *
 *  また、共有メモリ機能の使用は、システムの全体設計とも密接に関係しています。
 *  各プロセスごとのメモリアクセス権限や、メモリリソース上限などの点からも
 *  慎重かつ厳格に管理された上で使用する必要があります。@n
 *  共有メモリ機能を使用する場合は事前に関係者に相談するようにして下さい。
 *
 */
nn::Result CreateSharedMemory(SharedMemoryType* sharedMemory, size_t size, MemoryPermission myPermission, MemoryPermission otherPermission) NN_NOEXCEPT;


//--------------------------------------------------------------------------
/**
 * @brief   既存の共有メモリハンドルを SharedMemoryType オブジェクトに関連付けします。
 *
 * @param[in] sharedMemory  共有メモリオブジェクトへのポインタ
 * @param[in] size          共有メモリのサイズ
 * @param[in] handle        共有メモリのハンドル
 * @param[in] managed       共有メモリのハンドル管理フラグ
 *
 * @pre
 *  - sharedMemory オブジェクトが未初期化状態もしくは初期化されていない状態である
 *  - handle が有効なハンドル値である
 *  - handle が指す共有メモリのサイズが size と等しい
 *
 * @post
 *  - sharedMemory オブジェクトは初期化状態である
 *
 * @details
 *  指定された共有メモリハンドルを使って、共有メモリオブジェクトを初期化します。
 *  指定される size は、handle が指す共有メモリ自身のサイズと一致して
 *  いなければなりません。
 *
 *  AttachSharedMemory() で初期化した共有メモリオブジェクトは、
 *  DestroySharedMemory() にて破棄することが出来ます。
 *
 *  また、 DestroySharedMemory() 時に自動的にハンドルをシステムに
 *  返却するかどうかを managed で指定しておきます。@n
 *  managed に true を指定すると DestroySharedMemory() 時にハンドルを
 *  システムに返却し、false を指定すると DestroySharedMemory() 時にハンドルを
 *  返却しません。後者の場合、このハンドルは引き続き有効となります。
 *
 *  本 API の動作中は、対象共有メモリオブジェクトに対する操作を
 *  行なわないで下さい。
 *
 */
void AttachSharedMemory(SharedMemoryType* sharedMemory, size_t size, NativeHandle handle, bool managed) NN_NOEXCEPT;


//--------------------------------------------------------------------------
/**
 * @brief   共有メモリオブジェクトを破棄します。
 *
 * @param[in] sharedMemory  共有メモリオブジェクトへのポインタ
 *
 * @pre
 *  - sharedMemory オブジェクトは初期化状態もしくはマップ状態である
 *
 * @post
 *  - sharedMemory オブジェクトは未初期化状態である
 *
 * @details
 *  SharedMemoryType オブジェクトを破棄します。
 *
 *  同時に、 CreateSharedMemory() 時および AttachSharedMemory() 時に設定された
 *  ハンドル管理フラグが true ならば、共有メモリハンドルをシステムに返却します。
 *  ハンドル管理フラグが false の場合は、ハンドルは引き続き有効なままです。
 *
 *  sharedMemory がマップ状態だった場合には、事前にアンマップされます。
 *
 *  本 API の動作中は、対象 SharedMemoryTypeオブジェクトに対する操作を
 *  行なわないで下さい。
 *
 */
void DestroySharedMemory(SharedMemoryType* sharedMemory) NN_NOEXCEPT;


//--------------------------------------------------------------------------
/**
 * @brief   共有メモリをマップします。
 *
 * @param[in] sharedMemory  共有メモリオブジェクトへのポインタ
 * @param[in] myPermission  マップするメモリ領域に設定するアクセス権
 *
 * @return  共有メモリをマップしたアドレスを返します。
 *
 * @pre
 *  - sharedMemory オブジェクトは初期化状態である
 *  - myPermission が CreateSharedMemory() で指定されている制約に合致している
 *
 * @post
 *  - 返値が非 NULL なら sharedMemory オブジェクトはマップ状態である
 *  - 返値が NULL なら sharedMemory オブジェクトは初期化状態である
 *
 * @details
 *  指定された共有メモリを、未使用のメモリ空間を探してマップします。@n
 *  マップに成功した場合、共有メモリをマップしたアドレスを返します。
 *  共有メモリをマップする十分な大きさの空き空間が見つからなかった場合は
 *  NULL を返します。
 *
 *  マップされたメモリへのアクセス権は myPermission で指定されたものになります。
 *  アクセス権は CreateSharedMemory() 時に指定されたアクセス権の制約を
 *  満たしたものでなければなりません。
 *
 */
void* MapSharedMemory(SharedMemoryType* sharedMemory, MemoryPermission myPermission) NN_NOEXCEPT;


//--------------------------------------------------------------------------
/**
 * @brief   共有メモリをアンマップします。
 *
 * @param[in] sharedMemory  共有メモリオブジェクトへのポインタ
 *
 * @pre
 *  - sharedMemory オブジェクトは初期化状態もしくはマップ状態である
 *
 * @post
 *  - sharedMemory オブジェクトは初期化状態である
 *
 * @details
 *  MapSharedMemory() でマップした共有メモリをアンマップします。
 *  マップ状態でない場合には何も行いません。
 *
 *  マップ時に address に NULL を指定しマップ先アドレスを自動取得していた場合、
 *  本 API によって、そのアドレス空間は開放されます。
 *
 */
void UnmapSharedMemory(SharedMemoryType* sharedMemory) NN_NOEXCEPT;


//--------------------------------------------------------------------------
/**
 * @brief   共有メモリオブジェクトの マップ先アドレス を返します。
 *
 * @param[in] sharedMemory  共有メモリオブジェクトへのポインタ
 *
 * @return  共有メモリのマップ先アドレスを返します。
 *
 * @pre
 *  - sharedMemory オブジェクトは 初期化状態、マップ状態 のいずれかである
 *
 * @details
 *  共有メモリがマップされたアドレスを返します。@n
 *  共有メモリオブジェクトがマップ状態でない場合、NULL を返します。
 *
 */
void* GetSharedMemoryAddress(const SharedMemoryType* sharedMemory) NN_NOEXCEPT;

//--------------------------------------------------------------------------
/**
 * @brief   共有メモリのサイズを返します。
 *
 * @param[in] sharedMemory  共有メモリオブジェクトへのポインタ
 *
 * @return  共有メモリのサイズを返します。
 *
 * @pre
 *  - sharedMemory オブジェクトは 初期化状態、マップ状態 のいずれかである
 *
 * @details
 *  共有メモリのサイズを返します。
 *
 */
size_t GetSharedMemorySize(const SharedMemoryType* sharedMemory) NN_NOEXCEPT;


//--------------------------------------------------------------------------
/**
 * @brief   共有メモリオブジェクトの ハンドル を返します。
 *
 * @param[in] sharedMemory  共有メモリオブジェクトへのポインタ
 *
 * @return  共有メモリオブジェクトのハンドルを返します。
 *
 * @pre
 *  - sharedMemory オブジェクトは 初期化状態、マップ状態 のいずれかである
 *
 * @details
 *  共有メモリオブジェクトが管理している共有メモリハンドルを返します。
 *  ハンドル管理フラグは変更されません。
 *
 */
NativeHandle GetSharedMemoryHandle(const SharedMemoryType* sharedMemory) NN_NOEXCEPT;


//! @}

}} // namespace nn::os

