﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/*
 * @file
 * @brief   SDK 専用の省メモリなミューテックスに関する公開ヘッダファイル
 */

#pragma once

#include <nn/nn_Macro.h>
#include <nn/os/os_Config.h>
#include <nn/os/detail/os_MacroImpl.h>
#include <nn/os/detail/os_InternalCriticalSection.h>

namespace nn { namespace os {

class SdkConditionVariable;

//-----------------------------------------------------------------------------
/**
 * @brief SDK 専用の省メモリなミューテックスを扱うための構造体です。
 *
 * @details
 *  この構造体は C++ 標準の BasicLockable 要件と Lockable 要件を満たしています。
 *  オブジェクトを明示的に初期化するには Initialize() を発行して下さい。
 *  使用後はオブジェクトをそのまま破棄できます。
 *
 *  また、以下のようにマクロを使って静的にオブジェクトを初期化できます。
 *
 *  @code
 *      static nn::os::SdkMutexType  sdkMutex = NN_OS_SDK_MUTEX_INITIALIZER();
 *  @endcode
 *
 */
struct SdkMutexType
{
    union
    {
        // 静的初期化用イメージを定義するための配列
        int32_t _mutexImage[sizeof(detail::InternalCriticalSectionStorage) / sizeof(int32_t)];
        detail::InternalCriticalSectionStorage _mutex;
    };
#if defined(NN_BUILD_CONFIG_OS_WIN)
    void* _ownerThread;
#endif

    void Initialize() NN_NOEXCEPT;
    void Lock() NN_NOEXCEPT;
    bool TryLock() NN_NOEXCEPT;
    void Unlock() NN_NOEXCEPT;
    bool IsLockedByCurrentThread() const NN_NOEXCEPT;

    void lock() NN_NOEXCEPT
    {
        Lock();
    }
    bool try_lock() NN_NOEXCEPT
    {
        return TryLock();
    }
    void unlock() NN_NOEXCEPT
    {
        Unlock();
    }
};

NN_OS_DETAIL_STATIC_ASSERT_TRIVIAL(SdkMutexType);


//-----------------------------------------------------------------------------
/**
 * @brief SDK 専用の省メモリなミューテックスを扱うためのクラスです。
 *
 * @details
 *  この構造体は C++ 標準の BasicLockable 要件と Lockable 要件を満たしています。
 *  オブジェクトはコンストラクタで初期化されます。
 *
 */
class SdkMutex
{
private:
    friend class SdkConditionVariable;

    SdkMutexType m_SdkMutex;

public:
    SdkMutex() NN_NOEXCEPT
    {
        m_SdkMutex.Initialize();
    }
    void Lock() NN_NOEXCEPT
    {
        m_SdkMutex.Lock();
    }
    bool TryLock() NN_NOEXCEPT
    {
        return m_SdkMutex.TryLock();
    }
    void Unlock() NN_NOEXCEPT
    {
        m_SdkMutex.Unlock();
    }
    bool IsLockedByCurrentThread() const NN_NOEXCEPT
    {
        return m_SdkMutex.IsLockedByCurrentThread();
    }
    void lock() NN_NOEXCEPT
    {
        this->Lock();
    }
    bool try_lock() NN_NOEXCEPT
    {
        return this->TryLock();
    }
    void unlock() NN_NOEXCEPT
    {
        this->Unlock();
    }
};


//-----------------------------------------------------------------------------
/**
 * @brief SDK 専用の再帰ロック可能な省メモリなミューテックスを扱うための構造体です。
 *
 * @details
 *  この構造体は C++ 標準の BasicLockable 要件と Lockable 要件を満たしています。
 *  オブジェクトを明示的に初期化するには Initialize() を発行して下さい。
 *  使用後はオブジェクトをそのまま破棄できます。
 *
 *  再帰ロック可能な回数は UINT_MAX32 です。
 *  この回数を超えてロックしようとすると API の内部でアボートします。
 *
 *  また、以下のようにマクロを使って静的にオブジェクトを初期化できます。
 *
 *  @code
 *      static nn::os::SdkRecursiveMutexType staticSdkRecursiveMutex = NN_OS_SDK_RECURSIVE_MUTEX_INITIALIZER();
 *  @endcode
 *
 */
struct SdkRecursiveMutexType
{
    union
    {
        // 静的初期化用イメージを定義するための配列
        int32_t _mutexImage[sizeof(detail::InternalCriticalSectionStorage) / sizeof(int32_t)];
        detail::InternalCriticalSectionStorage _mutex;
    };
    uint32_t _recursiveCount;
#if defined(NN_BUILD_CONFIG_OS_WIN)
    void* _ownerThread;
#endif

    void Initialize() NN_NOEXCEPT;
    void Lock() NN_NOEXCEPT;
    bool TryLock() NN_NOEXCEPT;
    void Unlock() NN_NOEXCEPT;
    bool IsLockedByCurrentThread() const NN_NOEXCEPT;

    void lock() NN_NOEXCEPT
    {
        Lock();
    }
    bool try_lock() NN_NOEXCEPT
    {
        return TryLock();
    }
    void unlock() NN_NOEXCEPT
    {
        Unlock();
    }
};

NN_OS_DETAIL_STATIC_ASSERT_TRIVIAL(SdkRecursiveMutexType);

//-----------------------------------------------------------------------------
/**
 * @brief SDK 専用の再帰ロック可能な省メモリなミューテックスを扱うためのクラスです。
 *
 * @details
 *  この構造体は C++ 標準の BasicLockable 要件と Lockable 要件を満たしています。
 *  オブジェクトはコンストラクタで初期化されます。
 *
 */
class SdkRecursiveMutex
{
private:
    friend class SdkConditionVariable;

    SdkRecursiveMutexType m_SdkRecursiveMutex;

public:
    SdkRecursiveMutex() NN_NOEXCEPT
    {
        m_SdkRecursiveMutex.Initialize();
    }
    void Lock() NN_NOEXCEPT
    {
        m_SdkRecursiveMutex.Lock();
    }
    bool TryLock() NN_NOEXCEPT
    {
        return m_SdkRecursiveMutex.TryLock();
    }
    void Unlock() NN_NOEXCEPT
    {
        m_SdkRecursiveMutex.Unlock();
    }
    bool IsLockedByCurrentThread() const NN_NOEXCEPT
    {
        return m_SdkRecursiveMutex.IsLockedByCurrentThread();
    }
    void lock() NN_NOEXCEPT
    {
        this->Lock();
    }
    bool try_lock() NN_NOEXCEPT
    {
        return this->TryLock();
    }
    void unlock() NN_NOEXCEPT
    {
        this->Unlock();
    }
};

//-----------------------------------------------------------------------------

#if defined(NN_BUILD_CONFIG_OS_HORIZON)
    #define NN_OS_SDK_MUTEX_INITIALIZER_IMPL()                  \
        {                                                       \
            { { 0 }, },                                         \
        }
    #define NN_OS_SDK_RECURSIVE_MUTEX_INITIALIZER_IMPL()        \
        {                                                       \
            { { 0 }, }, 0,                                      \
        }
#elif defined(NN_BUILD_CONFIG_OS_WIN) && defined(NN_BUILD_CONFIG_ADDRESS_32)
    #define NN_OS_SDK_MUTEX_INITIALIZER_IMPL()                  \
        {                                                       \
            { -1, -1, 0, 0, 0, 0 },                             \
            nullptr,                                            \
        }
    #define NN_OS_SDK_RECURSIVE_MUTEX_INITIALIZER_IMPL()        \
        {                                                       \
            { -1, -1, 0, 0, 0, 0 },                             \
            0, nullptr,                                         \
        }
#elif defined(NN_BUILD_CONFIG_OS_WIN) && defined(NN_BUILD_CONFIG_ADDRESS_64)
    #define NN_OS_SDK_MUTEX_INITIALIZER_IMPL()                  \
        {                                                       \
            { -1, -1, -1, 0, 0, 0, 0, 0, 0, 0 },                \
            nullptr,                                            \
        }
    #define NN_OS_SDK_RECURSIVE_MUTEX_INITIALIZER_IMPL()        \
        {                                                       \
            { -1, -1, -1, 0, 0, 0, 0, 0, 0, 0 },                \
            0, nullptr,                                         \
        }
#endif  // NN_BUILD_CONFIG_OS

//--------------------------------------------------------------------------
/**
 * @brief   nn::os::SdkMutexType オブジェクトを静的に初期化します。
 *
 * @details
 *  nn::os::SdkMutexType オブジェクトを静的に初期化するためのマクロです。
 *
 *  使用例：
 *      @code
 *      nn::os::SdkMutexType g_StaticSdkMutex = NN_OS_SDK_MUTEX_INITIALIZER();
 *      @endcode
 *
 */
#define NN_OS_SDK_MUTEX_INITIALIZER() NN_OS_SDK_MUTEX_INITIALIZER_IMPL()

//--------------------------------------------------------------------------
/**
 * @brief   nn::os::SdkRecursiveMutexType オブジェクトを静的に初期化します。
 *
 * @details
 *  nn::os::SdkRecursiveMutexType オブジェクトを静的に初期化するためのマクロです。
 *
 *  使用例：
 *      @code
 *      nn::os::SdkRecursiveMutexType g_StaticSdkRecursiveMutex = NN_OS_SDK_RECURSIVE_MUTEX_INITIALIZER();
 *      @endcode
 *
 */
#define NN_OS_SDK_RECURSIVE_MUTEX_INITIALIZER() NN_OS_SDK_RECURSIVE_MUTEX_INITIALIZER_IMPL()

//-----------------------------------------------------------------------------

}}  // namespace nn::os

