﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/*
 * @file
 * @brief   SDK 専用のスタック使用量計測ユーティリティ
 *
 * @details
 *  本ヘッダは SDK 開発者専用のため公開禁止です。
 *
 */

#pragma once

#include <nn/nn_Common.h>
#include <nn/nn_SdkAssert.h>
#include <nn/os/os_Config.h>
#include <nn/os/os_ThreadTypes.h>

#if defined(NN_BUILD_CONFIG_OS_HORIZON)
#include <nn/svc/svc_ThreadLocalRegion.h>

namespace nn { namespace os { namespace detail {
// カナリヤコード（適当）
const uint64_t SdkCanaryValue = 0xA58E7FF6CD9EF2CBull;
}}}

#if defined(NN_BUILD_CONFIG_CPU_ARM64)
    #define NN_OS_DETAIL_SDK_GET_SVC_THREAD_LOCAL_REGION(p) \
        asm volatile("mrs  %0, tpidrro_el0": "=&r"(p)::"memory")
#else
    #define NN_OS_DETAIL_SDK_GET_SVC_THREAD_LOCAL_REGION(p) \
        asm volatile("mrc  p15, 0, %0, c13, c0, 3": "=&r"(p)::"memory")
#endif

#define NN_OS_DETAIL_GET_CURRENT_STACK_TOP_ADDRESS(stackTop) \
        do \
        { \
            NN_PRAGMA_PUSH_WARNINGS \
            NN_DISABLE_WARNING_SHADOW \
            ::nn::svc::ThreadLocalRegion* p; \
            NN_OS_DETAIL_SDK_GET_SVC_THREAD_LOCAL_REGION(p); \
            auto pThread = reinterpret_cast<::nn::os::ThreadType*>(p->pThreadType); \
            (stackTop) = reinterpret_cast<void*>(pThread->_currentFiber ? pThread->_currentFiber->_stack : pThread->_stack); \
            NN_PRAGMA_POP_WARNINGS \
        } while (NN_STATIC_CONDITION(0))

//--------------------------------------------------------------------------
/**
 * @brief   スタック使用量計測のための事前準備を行なう内部実装マクロです。
 *
 * @param[in]  stackTop   現コンテキストで使用しているスタック領域の先頭アドレス
 *
 * @details
 *  通常は本マクロではなく、
 *  NN_OS_SDK_PREPARE_TO_MEASURE_STACK_USAGE() を使用して下さい。
 *
 *  本マクロは NN_OS_SDK_PREPARE_TO_MEASURE_STACK_USAGE() の内部実装として
 *  使用されます。本マクロは、独自スタックを指定したユーザ例外ハンドラなど、
 *  上記マクロで対応できない特殊ケース向けに用意されています。
 *
 */
#define NN_OS_SDK_PREPARE_TO_MEASURE_STACK_USAGE_IMPL(stackTop) \
        do \
        { \
            NN_PRAGMA_PUSH_WARNINGS \
            NN_DISABLE_WARNING_SHADOW \
            auto pStackTop = reinterpret_cast<uint64_t*>(stackTop); \
            uint64_t* pCurrentSp; \
            asm volatile ("mov %0, sp" : "=&r"(pCurrentSp) :: "memory"); \
            NN_SDK_ASSERT(reinterpret_cast<uintptr_t>(pCurrentSp) % 8 == 0); \
            NN_SDK_ASSERT(pStackTop <= pCurrentSp); \
            auto p = pStackTop; \
            while (p < pCurrentSp) \
            { \
                *p++ = ::nn::os::detail::SdkCanaryValue; \
            } \
            NN_PRAGMA_POP_WARNINGS \
        } while (NN_STATIC_CONDITION(0))

//--------------------------------------------------------------------------
/**
 * @brief   スタック使用量の計測結果を返す内部実装マクロです。
 *
 * @param[out] stackUsage 結果の代入先変数（size_t 型）
 * @param[in]  stackTop   現コンテキストで使用しているスタック領域の先頭アドレス
 *
 * @return  計測したスタック使用量を size_t 型で返します。
 *
 * @details
 *  通常は本マクロではなく、
 *  NN_OS_SDK_GET_STACK_USAGE() を使用して下さい。
 *
 *  本マクロは NN_OS_SDK_PREPARE_TO_MEASURE_STACK_USAGE_IMPL() マクロと
 *  セットで使用すべきものです。詳細はそちらのリファレンスを参照して下さい。
 *
 */
#define NN_OS_SDK_GET_STACK_USAGE_IMPL(stackUsage, stackTop) \
        do \
        { \
            NN_PRAGMA_PUSH_WARNINGS \
            NN_DISABLE_WARNING_SHADOW \
            auto pStackTop = reinterpret_cast<uint64_t*>(stackTop); \
            uint64_t* pCurrentSp; \
            asm volatile ("mov %0, sp" : "=&r"(pCurrentSp) :: "memory"); \
            NN_SDK_ASSERT(reinterpret_cast<uintptr_t>(pCurrentSp) % 8 == 0); \
            NN_SDK_ASSERT(pStackTop <= pCurrentSp); \
            auto p = pStackTop; \
            while (p < pCurrentSp) \
            { \
                if (*p != ::nn::os::detail::SdkCanaryValue) \
                { \
                    break; \
                } \
                ++p; \
            } \
            (stackUsage) = (pCurrentSp - p) * sizeof(*pCurrentSp); \
            NN_PRAGMA_POP_WARNINGS \
        } while (NN_STATIC_CONDITION(0))

//--------------------------------------------------------------------------
/**
 * @brief   スタック使用量計測のための事前準備を行ないます。
 *
 * @details
 *  本マクロは NN_OS_SDK_GET_STACK_USAGE() マクロとセットで使用します。
 *  以下のように関数内に設置することで、2 つのマクロの間に挟まれた処理の
 *  スタック使用量を計測します。
 *
 *  @code
 *      #include <nn/os/os_SdkMeasureStackUsage.h>
 *      #include <nn/nn_Log.h>
 *
 *      void Func()
 *      {
 *          Hoge1();
 *          size_t stackUsage;
 *          NN_OS_SDK_PREPARE_TO_MEASURE_STACK_USAGE(); // 計測の準備
 *          Hoge2();
 *          NN_OS_SDK_GET_STACK_USAGE(stackUsage);      // 結果の取得
 *          NN_LOG("stack usage = %zu\n", stackUsage);
 *          Hoge3();
 *      }
 *  @endcode
 *
 *  スタック使用量の計測はカナリヤを使った単純なアルゴリズムのため、
 *  得られた結果が正確でない可能性もありますのでご注意下さい。
 *
 *  本機能は通常のスレッドスタックもしくはファイバスタック上で動作中の
 *  コンテキスト上で呼ばれた場合にのみ正しく動作します。
 *  ユーザ例外ハンドラなどで独自スタック上で動作している場合は、
 *  NN_OS_SDK_PREPARE_TO_MEASURE_STACK_USAGE_IMPL() および
 *  NN_OS_SDK_GET_STACK_USAGE_IMPL() を直接利用して下さい。
 *
 */
#define NN_OS_SDK_PREPARE_TO_MEASURE_STACK_USAGE() \
        do \
        { \
            NN_PRAGMA_PUSH_WARNINGS \
            NN_DISABLE_WARNING_SHADOW \
            void* _nn_os_detail_stacktop_temp; \
            NN_OS_DETAIL_GET_CURRENT_STACK_TOP_ADDRESS(_nn_os_detail_stacktop_temp); \
            NN_OS_SDK_PREPARE_TO_MEASURE_STACK_USAGE_IMPL(_nn_os_detail_stacktop_temp); \
            NN_PRAGMA_POP_WARNINGS \
        } while (NN_STATIC_CONDITION(0))

//--------------------------------------------------------------------------
/**
 * @brief   スタック使用量を計測し、その結果を返します。
 *
 * @param[out] stackUsage 結果の代入先変数（size_t 型）
 *
 * @details
 *  本マクロは NN_OS_SDK_PREPARE_TO_MEASURE_STACK_USAGE() マクロとセットで
 *  使用することで、スタック使用量を計測するためのものです。
 *  詳細は NN_OS_SDK_PREPARE_TO_MEASURE_STACK_USAGE() を参照して下さい。
 *
 */
#define NN_OS_SDK_GET_STACK_USAGE(stackUsage) \
        do \
        { \
            NN_PRAGMA_PUSH_WARNINGS \
            NN_DISABLE_WARNING_SHADOW \
            void* _nn_os_detail_stacktop_temp; \
            NN_OS_DETAIL_GET_CURRENT_STACK_TOP_ADDRESS(_nn_os_detail_stacktop_temp); \
            NN_OS_SDK_GET_STACK_USAGE_IMPL((stackUsage), _nn_os_detail_stacktop_temp); \
            NN_PRAGMA_POP_WARNINGS \
        } while (NN_STATIC_CONDITION(0))


#else // defined(NN_BUILD_CONFIG_OS_HORIZON)
#define NN_OS_SDK_PREPARE_TO_MEASURE_STACK_USAGE() do { } while (NN_STATIC_CONDITION(0))
#define NN_OS_SDK_GET_STACK_USAGE(stackUsage)      do { } while (NN_STATIC_CONDITION(0))
#endif
