﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * @file
 * @brief   割込みイベントに関する API の宣言
 */

#pragma once

#include <nn/nn_Macro.h>
#include <nn/os/os_EventCommon.h>
#include <nn/os/os_InterruptEventCommon.h>


//--------------------------------------------------------------------------
//  C++ 向けの宣言
//--------------------------------------------------------------------------

namespace nn {

class   TimeSpan;

namespace os {

struct  InterruptEventType;
struct  MultiWaitHolderType;

//! @name 割込みイベント関連 API
//! @{

//--------------------------------------------------------------------------
/**
 * @brief   InterruptEventType オブジェクトを初期化します。
 *
 * @param[in] event         InterruptEventType オブジェクトへのポインタ
 * @param[in] interruptName 割込みを一意に特定するための識別子
 * @param[in] clearMode     InterruptEventType オブジェクトの自動クリアモード
 *
 * @pre
 *  - event が未初期化状態もしくは初期化されていない状態である
 *
 * @post
 *  - event は初期化状態である。
 *
 */
void    InitializeInterruptEvent(InterruptEventType* event, InterruptName interruptName, EventClearMode clearMode) NN_NOEXCEPT;


//--------------------------------------------------------------------------
/**
 * @brief   InterruptEventType オブジェクトを Finalize します。
 *
 * @param[in] event     InterruptEventType オブジェクトへのポインタ
 *
 * @pre
 *  - event が初期化状態である
 *  - event を待機しているスレッドがいない
 *
 * @post
 *  - event が未初期化状態である
 *
 * @details
 *  割込みイベントオブジェクトをファイナライズし使用不可にします。@n
 *  event に対して待機中のスレッドがある場合にはアサートに失敗します。
 *
 *  本 API の動作中に対象 event オブジェクトに対する操作は行なわないで下さい。
 *
 */
void    FinalizeInterruptEvent(InterruptEventType* event) NN_NOEXCEPT;


//--------------------------------------------------------------------------
/**
 * @brief   InterruptEventType オブジェクトがシグナル状態になるまで待機します。
 *
 * @param[in] event     InterruptEventType オブジェクトへのポインタ
 *
 * @pre
 *  - event が初期化状態である
 *
 * @post
 *  - event が初期化状態である
 *
 */
void    WaitInterruptEvent(InterruptEventType* event) NN_NOEXCEPT;

//--------------------------------------------------------------------------
/**
 * @brief   InterruptEventType オブジェクトがシグナル状態かをポーリングします。
 *
 * @param[in] event     InterruptEventType オブジェクトへのポインタ
 *
 * @return  割込みイベントオブジェクト待機の成否を示します。
 *
 * @pre
 *  - event が初期化状態である
 *
 * @post
 *  - event が初期化状態である
 *
 */
bool    TryWaitInterruptEvent(InterruptEventType* event)  NN_NOEXCEPT;

//--------------------------------------------------------------------------
/**
 * @brief   InterruptEventType オブジェクトがシグナル状態になるまで待機します。
 *
 * @param[in] event     InterruptEventType オブジェクトへのポインタ
 * @param[in] timeout   最大待ち時間
 *
 * @return  割込みイベントオブジェクト待機の成否を示します。
 *
 * @pre
 *  - event が初期化状態である
 *  - timeout >= 0
 *
 * @post
 *  - event が初期化状態である
 *
 */
bool    TimedWaitInterruptEvent(InterruptEventType* event, TimeSpan timeout)  NN_NOEXCEPT;


//--------------------------------------------------------------------------
/**
 * @brief   InterruptEventType オブジェクトを非シグナル状態にクリアします。
 *
 * @param[in] event     InterruptEventType オブジェクトへのポインタ
 *
 * @pre
 *  - event が初期化状態である
 *
 * @post
 *  - event が非シグナル状態である
 *
 * @details
 *  割込みイベントオブジェクトを 非シグナル状態 にクリアします。
 *
 */
void    ClearInterruptEvent(InterruptEventType* event)    NN_NOEXCEPT;

//! @}

//! @name 多重待ち関連 API
//! @{

//--------------------------------------------------------------------------
/**
 * @brief   multiWaitHolder を初期化し、interruptEvent と関連付けます。
 *
 * @overloadlist{nn_os_InitializeMultiWaitHolder}
 *
 * @param[in] multiWaitHolder   多重待ちオブジェクトへのポインタ
 * @param[in] interruptEvent    紐付ける先の割込みイベントへのポインタ
 *
 * @pre
 *  - multiWaitHolder が未初期化状態である
 *  - interruptEvent が初期化状態である
 *
 * @post
 *  - multiWaitHolder が初期化状態である（interruptEvent に関連付けられる）
 *  - @ref nn::os::GetMultiWaitHolderUserData "nn::os::GetMultiWaitHolderUserData(multiWaitHolder)" == 0
 *
 */
void    InitializeMultiWaitHolder(MultiWaitHolderType* multiWaitHolder, InterruptEventType* interruptEvent) NN_NOEXCEPT;

//! @}


}} // namespace nn::os

