﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * @file
 * @brief   割込みイベントに関する公開ヘッダファイル
 */

#pragma once

#include <nn/nn_Macro.h>
#include <nn/nn_TimeSpan.h>
#include <nn/os/os_EventCommon.h>
#include <nn/os/os_InterruptEventCommon.h>
#include <nn/os/os_InterruptEventTypes.h>
#include <nn/os/os_InterruptEventApi.h>


namespace nn { namespace os {

//--------------------------------------------------------------------------
/**
 * @brief   割込みイベントを扱うためのクラスです。
 *
 * @details
 *  割込みイベントは、割込みコントローラからの通知を契機としてスレッドを起床
 *  させるための同期機能です。
 *
 *  nn::os::InterruptEvent クラスは、nn::os::InterruptEventType オブジェクトを
 *  使用する同期制御 API の呼び出しをラッピングしたユーティリティクラスです。@n
 *  本クラスが提供するメンバ関数は、@ref nn::os::InterruptEventType 型を引数に
 *  とる nn::os::WaitInterruptEvent() 等の API を発行する形で実装されています。
 */
class InterruptEvent
{
    NN_DISALLOW_COPY( InterruptEvent );
    NN_DISALLOW_MOVE( InterruptEvent );

public:
    /**
     * @brief   割込みイベントオブジェクトを構築し初期化します。
     *
     * @param[in] interruptName 割込みを一意に特定するための識別子
     * @param[in] clearMode     割込みイベントのクリアモード
     *
     * @details
     *  コンストラクタです。@n
     *  割込みイベントオブジェクトを構築し、指定されたパラメータで初期化します。@n
     *  詳細は nn::os::InitializeInterruptEvent() を参照して下さい。
     */
    explicit InterruptEvent(InterruptName interruptName, EventClearMode clearMode) NN_NOEXCEPT
    {
        nn::os::InitializeInterruptEvent(&m_InterruptEvent, interruptName, clearMode);
    }

    /**
     * @brief  デストラクタです。
     *
     * @details
     *  割込みイベントオブジェクトを破棄します。@n
     *  詳細は nn::os::FinalizeInterruptEvent() を参照して下さい。
     */
    ~InterruptEvent() NN_NOEXCEPT
    {
        nn::os::FinalizeInterruptEvent(&m_InterruptEvent);
    }

    /**
     * @brief   割込みイベントがシグナル状態になるまで待機します。
     *
     * @details
     *  割込みイベントがシグナル状態になるまで待機します。@n
     *  割込みイベントが nn::os::EventClearMode_AutoClear の場合、
     *  シグナル状態を検知すると同時に割込みイベントは非シグナル状態に
     *  クリアされます。@n
     *  詳細は nn::os::WaitInterruptEvent() を参照して下さい。
     */
    void Wait() NN_NOEXCEPT
    {
        nn::os::WaitInterruptEvent(&m_InterruptEvent);
    }

    /**
     * @brief   割込みイベントがシグナル状態かをポーリングします。
     *
     * @return  割込みイベントのポーリング成否を返します。
     *
     * @details
     *  割込みイベントがシグナル状態かどうかをポーリングします。@n
     *  割込みイベントがシグナル状態なら true を、そうでなければ false を
     *  返します。@n
     *  割込みイベントが nn::os::EventClearMode_AutoClear の場合、
     *  シグナル状態を検知すると同時に割込みイベントは非シグナル状態に
     *  クリアされます。@n
     *  詳細は nn::os::TryWaitInterruptEvent() を参照して下さい。
     */
    bool TryWait() NN_NOEXCEPT
    {
        return nn::os::TryWaitInterruptEvent(&m_InterruptEvent);
    }

    /**
     * @brief   割込みイベントがシグナル状態になるまで時限付きで待機します。
     *
     * @param[in] timeout   最大待ち時間
     *
     * @return  割込みイベント待機の成否を返します。
     *
     * @details
     *  割込みイベントをシグナル状態になるまで時限付きで待機します。@n
     *  指定された timeout 時間までに割込みイベントがシグナル状態になれば
     *  true を、そうでなければ false を返します。@n
     *  割込みイベントが nn::os::EventClearMode_AutoClear の場合、
     *  シグナル状態を検知すると同時に割込みイベントは非シグナル状態に
     *  クリアされます。@n
     *  詳細は nn::os::TimedWaitInterruptEvent() を参照して下さい。
     */
    bool TimedWait(nn::TimeSpan timeout) NN_NOEXCEPT
    {
        return nn::os::TimedWaitInterruptEvent(&m_InterruptEvent, timeout);
    }

    /**
     * @brief   割込みイベントを非シグナル状態にクリアします。
     *
     * @details
     *  割込みイベントを非シグナル状態にクリアします。@n
     *  詳細は nn::os::ClearInterruptEvent() を参照して下さい。
     */
    void Clear() NN_NOEXCEPT
    {
        nn::os::ClearInterruptEvent(&m_InterruptEvent);
    }


    /**
     * @brief   InterruptEventType オブジェクトへの参照を返します。
     *
     * @return  InterruptEventType オブジェクトへの参照
     *
     * @details
     *  自インスタンスが持つ InterruptEventType オブジェクトへの参照を返します。
     *  この変換演算子があるため、InterruptEventType& を引数にとる関数に対して、
     *  InterruptEvent& を指定することが可能です。
     */
    NN_IMPLICIT operator InterruptEventType&() NN_NOEXCEPT
    {
        return m_InterruptEvent;
    }

    /**
     * @brief   InterruptEventType オブジェクトへの const 参照を返します。
     *
     * @return  InterruptEventType オブジェクトへの const 参照
     *
     * @details
     *  自インスタンスが持つ InterruptEventType オブジェクトへの const 参照を返します。
     *  この変換演算子があるため、const InterruptEventType& を引数にとる関数に対して、
     *  InterruptEvent& を指定することが可能です。
     */
    NN_IMPLICIT operator const InterruptEventType&() const NN_NOEXCEPT
    {
        return m_InterruptEvent;
    }

    /**
     * @brief   自インスタンスの InterruptEventType オブジェクトへのポインタを返します。
     *
     * @return  InterruptEventType オブジェクトへのポインタ
     *
     * @details
     *  自インスタンスが持つ InterruptEventType オブジェクトへのポインタを返します。
     */
    InterruptEventType* GetBase() NN_NOEXCEPT
    {
        return &m_InterruptEvent;
    }

private:
    InterruptEventType   m_InterruptEvent;
};

//--------------------------------------------------------------------------

}} // namespace nn::os

