﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#ifdef __cplusplus

#include <atomic>
#include <nn/nn_Common.h>
#include <nn/os/os_Event.h>
#include <nn/os/os_EventTypes.h>
#include <nn/net/osl/osl_Mbuf.h>
#include <nn/net/osl/osl_MbufIndexed.h>
#include <nn/net/osl/osl_MbufPool.h>
#include <nn/net/osl/osl_Singleton.h>

namespace nn {
namespace net {
namespace osl {

class MbufSystem : public Singleton<MbufSystem>
{
public:
    typedef nnnetOslMbufIndex Index;
    static const int32_t MAX_POOL = 16;
    struct Config
    {
        size_t  unitSize;
        int32_t count;
    };

protected:
    static const uint32_t MAGIC = 0x9e3fbaba;
    struct HeaderBlock
    {
        uint32_t magic;
        uint32_t poolCount;
        Config  configs[MAX_POOL];
    };

public:
    static const size_t HEADER_SIZE = sizeof(HeaderBlock);

    MbufSystem();
    ~MbufSystem();

    void Initialize(const Config pConfigs[], int32_t count, void* workMemory, size_t size);
    void Finalize();
    void WaitFree();
    void NotifyFree();

    static size_t GetRequiredMemorySize(const Config pConfigs[], int32_t count);
    static Mbuf* Allocate(size_t size);
    static Mbuf* TryAllocate(size_t size);
    static void Free(Mbuf* pMbuf);
    static MbufIndexed* GetPtrFromIndex(Index index);
    static Index GetIndexFromPtr(const Mbuf* pMbuf);

protected:
    int32_t GetPoolCount() const;

    MbufPool                m_Pools[MAX_POOL];
    os::EventType           m_Event;
    HeaderBlock*            m_pHeaderBlock;
};

inline int32_t MbufSystem::GetPoolCount() const
{
    return m_pHeaderBlock->poolCount;
}

inline void MbufSystem::WaitFree()
{
    nn::os::WaitEvent(&m_Event);
}

inline void MbufSystem::NotifyFree()
{
    nn::os::SignalEvent(&m_Event);
}

}
}
}

#endif
