﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * NSOCモジュール 接続管理プロセス用定義
 */

#ifndef NN_NET_COMPATIBLE_NSOC_NSOC_ADMIN_H_
#define NN_NET_COMPATIBLE_NSOC_NSOC_ADMIN_H_

#ifndef NSOC_NO_DEPEND_HEADER
#include <nn/net/nos/nos_compatible.h>
#include <nn/net/nos/nos.h>			/* for NOSxxxx */
#include <nn/net/nnet/nnet.h>			/* for NNETxxxx */
#include <nn/net/nsoc/nsoc_user.h>		/* for NSOCInAddr, NSOCDNSServerInfo */
#endif

#ifdef  __cplusplus
extern "C" {
#endif

/*---------------------------------------------------------------------------*
 * Definitions
 *---------------------------------------------------------------------------*/

#define NSOC_VENDOR_NINTENDO        0x0000  /**< 本実装が対応している設定のベンダーコード */
#define NSOC_COMMON_CONF_VERSION    0x3100  /**< 本実装が対応している全体設定のバージョン */
#define NSOC_INS_CONF_VERSION       0x4100  /**< 本実装が対応しているインスタンス設定のバージョン */

/**
 * NSOCモジュールのエラーコード
 */
#define NSOC_ERROR_BASE                          0xffff8000
#define NSOC_ERROR_CODE(code)                    ((s32)(NSOC_ERROR_BASE|(code)))

enum NSOCErrorCode {
    NSOC_ERR_PARAM          = NSOC_ERROR_CODE(1),       /**< パラメータエラー */
    NSOC_ERR_INIT           = NSOC_ERROR_CODE(2),       /**< 初期状態のため実行不可 */
    NSOC_ERR_FINISH         = NSOC_ERROR_CODE(3),       /**< 終了状態のため実行不可 */
    NSOC_ERR_CONFVER        = NSOC_ERROR_CODE(4),       /**< 設定のバージョンエラー */
    NSOC_ERR_STATE          = NSOC_ERROR_CODE(5),       /**< ステートエラー */
    NSOC_ERR_EXIST          = NSOC_ERROR_CODE(6),       /**< 既に作成済み */
    NSOC_ERR_NOMEM          = NSOC_ERROR_CODE(7),       /**< メモリ不足 */
    NSOC_ERR_NOS            = NSOC_ERROR_CODE(8),       /**< OSサービス関数がエラーを返した */
    NSOC_ERR_LOWER          = NSOC_ERROR_CODE(9),       /**< 下位レイヤからのエラー */
    NSOC_ERR_TIMEOUT        = NSOC_ERROR_CODE(10),      /**< タイムアウト */
    NSOC_ERR_INS_RESOURCES  = NSOC_ERROR_CODE(11),      /**< OSリソース確保失敗 */
    NSOC_ERR_MAXSOCKET      = NSOC_ERROR_CODE(12),      /**< ソケット数が最大に達した */
    NSOC_ERR_ABORT          = NSOC_ERROR_CODE(13),      /**< アボートされた */

    NSOC_ERR_IP_UNREACHABLE = NSOC_ERROR_CODE(21),      /**< 到達できなかった */
    NSOC_ERR_IP_DATASIZE    = NSOC_ERROR_CODE(22),      /**< データサイズ不正 */
    NSOC_ERR_IP_NETDOWN     = NSOC_ERROR_CODE(23),      /**< ネットワークダウン */
    NSOC_ERR_IP_INVALID     = NSOC_ERROR_CODE(24),      /**< 不正パラメータ */
    NSOC_ERR_IP_REFUSED     = NSOC_ERROR_CODE(25),      /**< 処理の拒否 */
    NSOC_ERR_IP_TIMEOUT     = NSOC_ERROR_CODE(26),      /**< タイムアウト */
    NSOC_ERR_IP_CANCELED    = NSOC_ERROR_CODE(27),      /**< キャンセル */
    NSOC_ERR_IP_BUSY        = NSOC_ERROR_CODE(28),      /**< ビジー */
    NSOC_ERR_IP_CLOSING     = NSOC_ERROR_CODE(29),      /**< ソケットを閉じた */
    NSOC_ERR_IP_EXIST       = NSOC_ERROR_CODE(30),      /**< 既に存在する */
    NSOC_ERR_IP_RESET       = NSOC_ERROR_CODE(31),      /**< リセットを受けた */
    NSOC_ERR_IP_NOT_EXIST   = NSOC_ERROR_CODE(32),      /**< 最早存在しない */
    NSOC_ERR_IP_INV_SOCKET  = NSOC_ERROR_CODE(33),      /**< 無効なソケット */
    NSOC_ERR_IP_AGAIN       = NSOC_ERROR_CODE(34),      /**< 実行中 */
    NSOC_ERR_IP_SOCKET_UNSPECIFIED = NSOC_ERROR_CODE(35),   /**< ソケットが指定されていない */
    NSOC_ERR_IP_SOURCE_QUENCH = NSOC_ERROR_CODE(36),    /**< 送信抑制 */
    NSOC_ERR_IP_INV_OPTION  = NSOC_ERROR_CODE(37),      /**< 不正なオプション */
    NSOC_ERR_IP_ADDR_COLLISION = NSOC_ERROR_CODE(38),   /**< IPアドレス重複 */
    NSOC_ERR_IP_SHUTDOWN    = NSOC_ERROR_CODE(39),      /**< シャットダウン状態 */
    NSOC_ERR_IP_INV_HEADER  = NSOC_ERROR_CODE(40),      /**< 不正なIPヘッダ */
    NSOC_ERR_IP_INV_ADDR    = NSOC_ERROR_CODE(41),      /**< 不正なIPアドレス */
    NSOC_ERR_IP_INV_DATA    = NSOC_ERROR_CODE(42),      /**< 不正なIPデータ */

    NSOC_ERR_DHCP_EXPIRED   = NSOC_ERROR_CODE(51),      /**< DHCPリース時間満了 */
    NSOC_ERR_DHCP_TIMEOUT   = NSOC_ERROR_CODE(52),      /**< DHCPタイムアウト */
    NSOC_ERR_DHCP_NAK       = NSOC_ERROR_CODE(53),      /**< DHCPのNAKを受信 */

    NSOC_ERR_IP6_ADDR_COLLISION = NSOC_ERROR_CODE(60),  /**< IPv6アドレス重複 */

    NSOC_ERR_NONE           = 0                         /**< 正常終了 */
};

/*---------------------------------------------------------------------------*
 * Types/Declarations
 *---------------------------------------------------------------------------*/
/**
 * NSOC_set_common_config()で使用するモジュール全体の設定構造体
 */
typedef struct NSOCCommonConfig {
    u16         vendor;             /**< ベンダーコード。NSOC_VENDOR_NINTENDOを指定すること */
    u16         version;            /**< バージョンコード。NSOC_COMMON_CONF_VERSIONを指定すること */

    s32         timeWaitBuffer;     /**< TCPのTIME-WAITソケットを保持するためのバッファサイズ */

    s32         reassembleMtu;      /**< 再構築後の最大IPパケットサイズ */

    // TCP
    s32         rwin;               /**< TCP受信ウインドウサイズ(default 2 x MSS) */
    s32         r2;                 /**< TCP再送のトータル時間(default 100 sec) */

    // UDP
    s32         udpRecvBuff;        /**< UDP受信バッファサイズ(default 4416 byte) */

    // DHCP
    const char* hostName;           /**< DHCPで使用するホスト名へのポインタ */
    s32         rdhcp;              /**< DHCP再送回数(default 4) */

} NSOCCommonConfig;

/*
 * NSOCInstanceConfigのflagにセットするフラグ
 */
enum NSOCInstanceConfigFlag {
    NSOC_FLAG_DHCP         =  0x0001, /**< DHCPによるIPアドレス取得 */
    NSOC_FLAG_PPP          =  0x0002, /**< PPPによるIPアドレス取得 */
    NSOC_FLAG_IP4          =  0x0004, /**< IPv4を有効にする */
    NSOC_FLAG_IP6          =  0x0008, /**< IPv6を有効にする */
    NSOC_FLAG_DHCP6        =  0x0010, /**< DHCPv6によるIPv6アドレス取得 */
    NSOC_FLAG_NORA_DHCP6   =  0x0020, /**< NDP RAが受信されないとき、DHCPv6を起動する */
    NSOC_FLAG_DHCP_REBOOT  =  0x0100, /**< DHCPでのIPアドレス取得失敗時に再度実行する */
    NSOC_FLAG_DHCP_RELEASE =  0x0200, /**< NSOC_CleanupIns()実行時に、DHCP Releaseを送信する */
    NSOC_FLAG_DNS_AUTO     =  0x1000, /**< DHCPやPPPで取得したDNSサーバを使用する */
    NSOC_FLAG_ZEROCONF     =  0x8000  /**< リンクローカルアドレスを取得する */
};

/**
 * NSOC_SetInstanceConfig()で使用するインスタンスごとの設定
 */
typedef struct NSOCInstanceConfig {
    u16         vendor;             /**< ベンダーコード。NSOC_VENDOR_NINTENDOを指定すること */
    u16         version;            /**< バージョンコード。NSOC_INS_CONF_VERSIONを指定すること */

    u32         flag;               /**< フラグ。NSOC_FLAG_XXXのORをとったもの */
    NSOCInAddr  addr;               /**< 自IPアドレス */
    NSOCInAddr  netmask;            /**< サブネットマスク */
    NSOCInAddr  router;             /**< デフォルトゲートウエイIPアドレス */
    s32         mtu;                /**< MTUサイズ */

    NSOCDNSServerInfo dnsserver;    /**< DNSサーバアドレスリスト */

#ifdef NSOC_IP6_ENABLE
    u8          curHopLimit;     /* ホップリミット */
    u32         retransTimer;    /* NS再送間隔(msec) */
    u32         dadNSTransmits;  /* アドアレス重複検査用NS送信回数 */
    u8          interfaceId[8];  /* インタフェースID */
#endif /* NSOC_IP6_ENABLE */
} NSOCInstanceConfig;

/**
 * NSOCインスタンスで発生したエラーを通知するコールバック関数
 */
typedef void (* NSOCConfigErrorCallback )(NSOCInstancePtr insP, s32 err);

#ifndef NSOC_NO_DEPEND_HEADER
/**
 * NSOC_GetInstanceOpt(), NSOC_SetInstanceOpt()でNSOC_CONFIG_FILTER_INPUT/NSOC_CONFIG_FILTER_OUTPUT使用時の型
 */
typedef BOOL (* NSOCFilter)(NSOCInstancePtr insP, NOSMessageBuf *mbuf);
#endif

/**
 * NSOCインスタンスからNNETInstanceを取得する
 */
NN_INLINE NNETInstance *NSOC_GetNNETInstance(NSOCInstancePtr insP)
{
    return (NNETInstance *)insP;
}

/**
 * NNETInstanceからNSOCインスタンスを取得する
 */
NN_INLINE NSOCInstancePtr NSOC_GetNSOCInstance(NNETInstance *ins)
{
    return (NSOCInstancePtr)ins;
}

/*---------------------------------------------------------------------------*
 * Function Prototypes
 *---------------------------------------------------------------------------*/
/* nsoc_main.c */
extern s32 NSOC_Init(const NNETAllocator *allocator);
extern s32 NSOC_Finish(void);
extern s32 NSOC_SetCommonConfig(const NSOCCommonConfig *common_conf);
extern s32 NSOC_SetInstanceConfig(NSOCInstancePtr ins, const NSOCInstanceConfig *ins_conf);
extern s32 NSOC_StartupIns(NNETInstance *ins, s32 timeout);
extern s32 NSOC_CleanupIns(NNETInstance *ins, s32 timeout);
extern s32 NSOC_AbortIns(NNETInstance *ins);
extern s32 NSOC_CreateIns(NSOCInstancePtr *insP);
extern s32 NSOC_DestroyIns(NSOCInstancePtr insP);
extern s32 NSOC_AttachIF(NSOCInstancePtr insP, NNETEthernetAPI *low_api);
extern s32 NSOC_DetachIF(NSOCInstancePtr insP);
extern s32 NSOC_SetConfigErrorCallback(NSOCInstancePtr nsoc_insP, NSOCConfigErrorCallback callback);
extern s32 NSOC_SetUserInsParam(NSOCInstancePtr insP, void* param);
extern s32 NSOC_GetUserInsParam(NSOCInstancePtr insP, void** param);

/* nsoc_socket.c */
extern int NSOC_GetUserInfo(int s, void** userInfo);
extern void NSOC_FreeUserInfo(void* userInfo);
extern void* NSOC_DuplicateUserInfo(void* userInfo, int socket);



#ifdef  __cplusplus
}
#endif

/* NN_NET_COMPATIBLE_NSOC_NSOC_ADMIN_H_ */
#endif


