﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <random>
#include <nn/sf/sf_Buffers.h>
#include <nn/ncm/ncm_ContentStorage.h>
#include <nn/ncm/ncm_MakeContentPathFunction.h>

namespace nn { namespace ncm {

    class ReadOnlyContentStorageImpl
    {
        NN_DISALLOW_COPY(ReadOnlyContentStorageImpl);
        NN_DISALLOW_MOVE(ReadOnlyContentStorageImpl);

    public:
        /**
        * @brief    コンストラクタです。
        */
        ReadOnlyContentStorageImpl() NN_NOEXCEPT;

        Result Initialize(const char* rootPath, MakeContentPathFunction func) NN_NOEXCEPT;

        Result GeneratePlaceHolderId(sf::Out<PlaceHolderId> outValue) NN_NOEXCEPT;

        Result CreatePlaceHolder(PlaceHolderId placeHolderId, ContentId contentId, int64_t size) NN_NOEXCEPT;

        Result SetPlaceHolderSize(PlaceHolderId placeHolderId, int64_t size) NN_NOEXCEPT;

        Result DeletePlaceHolder(PlaceHolderId id) NN_NOEXCEPT;

        Result HasPlaceHolder(sf::Out<bool> outValue, PlaceHolderId id) const NN_NOEXCEPT;

        Result WritePlaceHolder(PlaceHolderId id, int64_t offset, sf::InBuffer buffer) NN_NOEXCEPT;

        Result Register(PlaceHolderId placeHolderId, ContentId contentId) NN_NOEXCEPT;

        Result RevertToPlaceHolder(PlaceHolderId placeHolderId, ContentId contentId, ContentId postContentId) NN_NOEXCEPT;

        Result Delete(ContentId id) NN_NOEXCEPT;

        /**
        * @brief    指定された ID のコンテンツの存在を確認します。
        *
        * @param[out]   outValue    コンテンツが存在すれば true が返ります。
        * @param[in]    id          存在を確認するコンテンツの ID
        *
        * @return   処理の結果が返ります。想定外のファイルシステムエラーが無ければ常に成功します。
        *
        * @pre
        *           - Initialize の成功
        * @post
        *           - コンテンツが存在するとき *outValue == true、そうでないとき *outValue == false
        */
        Result Has(sf::Out<bool> outValue, ContentId id) const NN_NOEXCEPT;

        /**
        * @brief    指定された ID のコンテンツのファイルパスを取得します。
        *
        * @details  ストレージにコンテンツが存在するかどうかは検知しません。
        *           存在を確認したい場合は Has() を利用してください。
        *
        * @param[in]    id          ファイルパスを取得するコンテンツの ID
        *
        * @return   指定された ID のコンテンツのファイルパスを示す文字列が返ります。
        *
        * @pre
        *           - Initialize の成功
        */
        Result GetPath(sf::Out<Path> outValue, ContentId id) const NN_NOEXCEPT;

        /**
        * @brief    指定された ID のコンテンツのファイルサイズを取得します。
        *
        * @param[out]   outValue    コンテンツのファイルサイズ
        * @param[in]    id          ファイルパスを取得するコンテンツの ID
        *
        * @return   処理の結果が返ります。想定外のファイルシステムエラーが無ければ常に成功します。
        *
        * @pre
        *           - Initialize の成功
        */
        Result GetSizeFromContentId(sf::Out<std::int64_t> outValue, ContentId id) const NN_NOEXCEPT;

        Result GetSizeFromPlaceHolderId(sf::Out<std::int64_t> outValue, PlaceHolderId id) const NN_NOEXCEPT;

        Result GetFreeSpaceSize(sf::Out<std::int64_t> outValue) const NN_NOEXCEPT;

        Result GetTotalSpaceSize(sf::Out<std::int64_t> outValue) const NN_NOEXCEPT;

        Result GetPlaceHolderPath(sf::Out<Path> outValue, PlaceHolderId id) const NN_NOEXCEPT;

        Result CleanupAllPlaceHolder() NN_NOEXCEPT;

        Result ListPlaceHolder(sf::Out<std::int32_t> outCount, const sf::OutArray<ncm::PlaceHolderId>& outList) const NN_NOEXCEPT;

        Result GetContentCount(nn::sf::Out<std::int32_t> outCount) const NN_NOEXCEPT;

        Result ListContentId(nn::sf::Out<std::int32_t> outCount, const nn::sf::OutArray<nn::ncm::ContentId>& outList, std::int32_t offset) const NN_NOEXCEPT;

        Result DisableForcibly() NN_NOEXCEPT;

        Result ReadContentIdFile(sf::OutBuffer buffer, ContentId id, int64_t offset) NN_NOEXCEPT;

        Result GetRightsIdFromPlaceHolderId(sf::Out<RightsId> outValue, PlaceHolderId id) NN_NOEXCEPT;

        Result GetRightsIdFromContentId(sf::Out<RightsId> outValue, ContentId id) const NN_NOEXCEPT;

        Result WriteContentForDebug(ContentId id, int64_t offset, sf::InBuffer buffer) NN_NOEXCEPT;

        Result FlushPlaceHolder() NN_NOEXCEPT;

        Result RepairInvalidFileAttribute() NN_NOEXCEPT;

    private:
        PathString                  m_RootPath;
        MakeContentPathFunction     m_MakeContentPathFunction;
        bool                        m_IsDisabled;
    };

}}
