﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/ncm/ncm_ContentId.h>
#include <nn/ncm/ncm_PathString.h>

namespace nn { namespace ncm {

    /**
    * @brief コンテンツのファイルパスを作成する関数です。
    *
    * @param[out]   outValue        作成したコンテンツのファイルパスです。
    * @param[in]    id              パスを作成するコンテンツの ID です。
    * @param[in]    contentRootPath コンテンツを配置するルートディレクトリのパスです。
    *
    */
    typedef void (*MakeContentPathFunction)(PathString* outValue, ContentId id, const char* contentRootPath);

    /**
    * @brief フラットにコンテンツを配置するファイルパス作成関数です。
    *
    * @param[out]   outValue        作成したコンテンツのファイルパスです。
    * @param[in]    id              パスを作成するコンテンツの ID です。
    * @param[in]    contentRootPath コンテンツを配置するルートディレクトリのパスです。
    *
    */
    void MakeFlatContentFilePath(PathString* outValue, ContentId id, const char* contentRootPath);

    /**
    * @brief SHA256 ハッシュで階層化したディレクトリにコンテンツを配置するファイルパス作成関数です。4K クラスタの FAT に最適化されています。
    *
    * @param[out]   outValue        作成したコンテンツのファイルパスです。
    * @param[in]    id              パスを作成するコンテンツの ID です。
    * @param[in]    contentRootPath コンテンツを配置するルートディレクトリのパスです。
    *
    */
    void MakeSha256HierarchicalContentFilePath_ForFat4KCluster(PathString* outValue, ContentId id, const char* contentRootPath);

    /**
    * @brief SHA256 ハッシュで階層化したディレクトリにコンテンツを配置するファイルパス作成関数です。32K クラスタの FAT に最適化されています。
    *
    * @param[out]   outValue        作成したコンテンツのファイルパスです。
    * @param[in]    id              パスを作成するコンテンツの ID です。
    * @param[in]    contentRootPath コンテンツを配置するルートディレクトリのパスです。
    *
    */
    void MakeSha256HierarchicalContentFilePath_ForFat32KCluster(PathString* outValue, ContentId id, const char* contentRootPath);

    /**
    * @brief SHA256 ハッシュで階層化したディレクトリにコンテンツを配置するファイルパス作成関数です。16K クラスタの FAT に最適化されています。
    *
    * @param[out]   outValue        作成したコンテンツのファイルパスです。
    * @param[in]    id              パスを作成するコンテンツの ID です。
    * @param[in]    contentRootPath コンテンツを配置するルートディレクトリのパスです。
    *
    */
    void MakeSha256HierarchicalContentFilePath_ForFat16KCluster(PathString* outValue, ContentId id, const char* contentRootPath);

    /**
    * @brief コンテントファイルパス作成関数を使ったときに、どの深さまでトラバースするべきかを取得する API です。
    */
    int GetHierarchicalContentDirectoryDepth(MakeContentPathFunction func) NN_NOEXCEPT;
}}
