﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <random>
#include <nn/sf/sf_Buffers.h>
#include <nn/fs.h>
#include <nn/ncm/ncm_ContentStorage.h>
#include <nn/ncm/ncm_PathString.h>
#include <nn/ncm/ncm_MakeContentPathFunction.h>
#include <nn/ncm/ncm_MakePlaceHolderPathFunction.h>
#include <nn/ncm/ncm_PlaceHolderAccessor.h>

namespace nn { namespace ncm {

    /**
    * @brief    コンテンツを書き込み可能なファイルシステム上でインポート・削除・状態取得・参照を行うためのクラスです。
    *
    * @details  コンテンツを配置するルート毎にインスタンスが必要です。
    *           活線挿抜などで配置するメディアが無効になった場合は、再度インスタンスの作成と初期化が必要です。
    *           Intialize 以外の関数は複数のスレッドから同時に呼び出すことが可能です。
    */
    class ContentStorageImpl
    {
        NN_DISALLOW_COPY(ContentStorageImpl);
        NN_DISALLOW_MOVE(ContentStorageImpl);

    public:

        /**
        * @brief    指定されたルートパス以下にコンテンツストレージの基礎ディレクトリ・ファイルが正しく存在するか検証します。
        *
        * @param[in]    rootPath    コンテンツおよびプレースホルダを配置するルートパス。
        *
        * @return   処理の結果が返ります。
        * @retval   ResultContentStorageBaseNotFound    コンテンツストレージの基礎ディレクトリが全て存在しません
        * @retval   ResultInvalidContentStorageBase     コンテンツストレージの基礎ディレクトリ・ファイルが不正です。
        *
        */
        static Result VerifyBase(const char* rootPath) NN_NOEXCEPT;

        /**
        * @brief    指定されたルートパス以下にコンテンツストレージの基礎ディレクトリ・ファイルを初期化します。
        *
        * @detail   プレースホルダのコンテキストは初期化され、全てのプレースホルダが削除されます。
        *
        * @param[in]    rootPath    コンテンツおよびプレースホルダを配置するルートパス。
        *
        * @return   処理の結果が返ります。
        * @retval   fs::ResultUsableSpaceNotEnough  ファイルシステムの空き容量が不足しています。
        *
        * @post
        *           - 配置の基礎となるディレクトリが作成されている
        */
        static Result InitializeBase(const char* rootPath) NN_NOEXCEPT;

        /**
        * @brief    指定されたルートパス以下のコンテンツストレージの基礎ディレクトリ・ファイルを削除します。
        *
        * @detail   コンテンツおよびプレースホルは全て削除されます。
        *
        * @param[in]    rootPath    コンテンツおよびプレースホルダを配置するルートパス。
        *
        * @return   処理の結果が返ります。想定外のファイルシステムエラーが無ければ常に成功します。
        */
        static Result CleanupBase(const char* rootPath) NN_NOEXCEPT;

        /**
        * @brief    コンストラクタです。
        */
        ContentStorageImpl() NN_NOEXCEPT;

        /**
        * @brief    デストラクタです。
        */
        ~ContentStorageImpl() NN_NOEXCEPT;

        /**
        * @brief    他の関数を呼び出す前に呼び出す必要がある初期化関数です。
        *
        * @detail   配置の基礎となるディレクトリの存在をチェックします。
        *
        * @param[in]    rootPath    コンテンツおよびプレースホルダを配置するルートパス。
        * @param[in]    func        コンテンツのファイルパスを作成する関数
        *
        * @param[in]    delayFlush  placeholder への書き込みフラッシュを遅延させるかどうか
        *
        * @return   処理の結果が返ります。
        * @retval   ResultInvalidContentStorageBase     コンテンツストレージの基礎ディレクトリ・ファイルが不正です。
        *
        * @pre
        *           - Initialize が成功していない
        *           - func != NULL
        * @post
        *           - 他の関数が呼び出せる
        */
        Result Initialize(const char* rootPath, MakeContentPathFunction func, MakePlaceHolderPathFunction placeHolderFunc, bool delayFlush) NN_NOEXCEPT;

        /**
        * @brief    ユニークなプレースホルダ ID を生成します。
        *
        * @details  上位レイヤから強制中断時のクリーンアップをするために、生成されたプレースホルダ ID を
        *           上位レイヤで永続化してからプレースホルダを作成してください。
        *
        * @return   処理の結果が返ります。想定外のファイルシステムエラーが無ければ常に成功します。
        *
        * @pre
        *           - Initialize の成功
        * @post
        *           - outValue にユニークなプレースホルダ ID が書き込まれている
        */
        Result GeneratePlaceHolderId(sf::Out<PlaceHolderId> outValue) NN_NOEXCEPT;

        /**
        * @brief    指定された ID とサイズのコンテンツのプレースホルダを作成します。
        *
        * @details  コンテンツの配置に必要なディレクトリは自動的に作成します。
        *
        * @param[in]   placeHolderId   作成するプレースホルダの ID
        * @param[in]    contentId       作成するコンテンツの ID
        * @param[in]    size            作成するプレースホルダのサイズ
        *
        * @return   処理の結果が返ります。
        * @retval   ResultPlaceHolderAlreadyExists      指定された ID のプレースホルダが既に存在しています。
        * @retval   ResultContentAlreadyExists          指定された ID のコンテンツが既に存在します。
        * @retval   fs::ResultUsableSpaceNotEnough  ファイルシステムの空き容量が不足しています。
        *
        * @pre
        *           - Initialize の成功
        * @post
        *           - contentId で指定されたコンテンツの配置に必要なディレクトリが存在する
        *           - placeHolderId で指定されたプレースホルダが size の大きさで存在する
        */
        Result CreatePlaceHolder(PlaceHolderId placeHolderId, ContentId contentId, int64_t size) NN_NOEXCEPT;

        /**
        * @brief    指定されたプレースホルダのサイズを変更します。
        *
        * @param[in]   placeHolderId   作成するプレースホルダの ID
        * @param[in]    size            作成するプレースホルダのサイズ
        *
        * @return   処理の結果が返ります。
        * @retval   ResultPlaceHolderNotFound           指定された ID のプレースホルダは存在しません。
        * @retval   fs::ResultUsableSpaceNotEnough  ファイルシステムの空き容量が不足しています。
        *
        * @pre
        *           - Initialize の成功
        * @post
        *           - placeHolderId で指定されたプレースホルダが size の大きさで存在する
        */
        Result SetPlaceHolderSize(PlaceHolderId placeHolderId, int64_t size) NN_NOEXCEPT;

        /**
        * @brief    指定された ID のプレースホルダを削除します。
        *
        * @details  コンテンツの配置に必要なディレクトリは削除されません。
        *
        * @param[in]    id      削除するプレースホルダの ID
        *
        * @return   処理の結果が返ります。
        * @retval   ResultPlaceHolderNotFound           指定された ID のプレースホルダは存在しません。
        *
        * @pre
        *           - Initialize の成功
        * @post
        *           - id で指定されたプレースホルダが削除されている
        */
        Result DeletePlaceHolder(PlaceHolderId id) NN_NOEXCEPT;

        /**
        * @brief    指定された ID のプレースホルダの存在を確認します。
        *
        * @param[out]   outValue    プレースホルダが存在すれば true が返ります。
        * @param[in]    id          存在を確認するプレースホルダの ID
        *
        * @return   処理の結果が返ります。想定外のファイルシステムエラーが無ければ常に成功します。
        *
        * @pre
        *           - Initialize の成功
        * @post
        *           - プレースホルダが存在するとき *outValue == true、そうでないとき *outValue == false
        */
        Result HasPlaceHolder(sf::Out<bool> outValue, PlaceHolderId id) const NN_NOEXCEPT;

        /**
        * @brief    プレースホルダにデータを書き込みます。
        *
        * @param[in]    id          データを書き込むプレースホルダの ID
        *
        * @return   処理の結果が返ります。
        * @retval   ResultPlaceHolderNotFound           指定された ID のプレースホルダは存在しません。
        *
        * @pre
        *           - Initialize の成功
        *           - offset >= 0
        *           - offset + size <= プレースホルダのサイズ
        *           - data != nullptr
        * @post
        *           - id で指定されたプレースホルダの offset に buffer から size 分のデータが書き込まれている
        */
        Result WritePlaceHolder(PlaceHolderId id, int64_t offset, sf::InBuffer buffer) NN_NOEXCEPT;

        /**
        * @brief    プレースホルダを登録して、コンテンツとして利用できるようにします。
        *
        * @param[in]    placeHolderId   登録するプレースホルダ ID
        * @param[in]    contentId       登録するコンテンツ ID
        *
        * @return   処理の結果が返ります。
        * @retval   ResultPlaceHolderNotFound           指定された ID のプレースホルダは存在しません。
        * @retval   ResultContentAlreadyExists          指定された ID のコンテンツが既に存在します。
        *
        * @pre
        *           - Initialize の成功
        * @post
        *           - contentId で指定されたコンテンツが存在する
        *           - placeHolderId で指定されたプレースホルダが存在しない
        */
        Result Register(PlaceHolderId placeHolderId, ContentId contentId) NN_NOEXCEPT;

        /**
        * @brief    登録済みのコンテンツを、再度更新するためにプレースホルダに移動します。
        *
        * @param[in]    placeHolderId   登録するプレースホルダ ID
        * @param[in]    contentId       更新対象のコンテンツ ID
        * @param[in]    postContentId   更新後のコンテンツ ID
        *
        * @return   処理の結果が返ります。
        * @retval   ResultPlaceHolderAlreadyExists  指定された ID のプレースホルダがすでに存在します。
        * @retval   ResultContentNotFound           指定された ID のコンテンツが存在しません。
        *
        * @pre
        *           - Initialize の成功
        * @post
        *           - contentId で指定されたコンテンツが存在しない
        *           - placeHolderId で指定されたプレースホルダが存在する
        */
        Result RevertToPlaceHolder(PlaceHolderId placeHolderId, ContentId contentId, ContentId postContentId) NN_NOEXCEPT;

        /**
        * @brief    指定された ID のコンテンツを削除します。
        *
        * @details  コンテンツが配置されているディレクトリは削除されません。
        *
        * @param[in]    id      削除するコンテンツの ID
        *
        * @return   処理の結果が返ります。
        * @retval   ResultContentInUse                  指定された ID のコンテンツは利用中です。
        * @retval   ResultContentNotFound               指定された ID のコンテンツは存在しません。
        *
        * @pre
        *           - Initialize の成功
        * @post
        *           - id で指定されたコンテンツが削除されている
        */
        Result Delete(ContentId id) NN_NOEXCEPT;

        /**
        * @brief    指定された ID のコンテンツの存在を確認します。
        *
        * @param[out]   outValue    コンテンツが存在すれば true が返ります。
        * @param[in]    id          存在を確認するコンテンツの ID
        *
        * @return   処理の結果が返ります。想定外のファイルシステムエラーが無ければ常に成功します。
        *
        * @pre
        *           - Initialize の成功
        * @post
        *           - コンテンツが存在するとき *outValue == true、そうでないとき *outValue == false
        */
        Result Has(sf::Out<bool> outValue, ContentId id) const NN_NOEXCEPT;

        /**
        * @brief    指定された ID のコンテンツのファイルパスを取得します。
        *
        * @details  ストレージにコンテンツが存在するかどうかは検知しません。
        *           存在を確認したい場合は Has() を利用してください。
        *
        * @param[in]    id          ファイルパスを取得するコンテンツの ID
        *
        * @return   指定された ID のコンテンツのファイルパスを示す文字列が返ります。
        *
        * @pre
        *           - Initialize の成功
        */
        Result GetPath(sf::Out<Path> outValue, ContentId id) const NN_NOEXCEPT;

        /**
        * @brief    指定された ID のコンテンツのファイルサイズを取得します。
        *
        * @param[out]   outValue    コンテンツのファイルサイズ
        * @param[in]    id          ファイルパスを取得するコンテンツの ID
        *
        * @return   処理の結果が返ります。想定外のファイルシステムエラーが無ければ常に成功します。
        *
        * @pre
        *           - Initialize の成功
        */
        Result GetSizeFromContentId(sf::Out<std::int64_t> outValue, ContentId id) const NN_NOEXCEPT;

        // キャッシュの制御があるため、const にはならない
        Result GetSizeFromPlaceHolderId(sf::Out<std::int64_t> outValue, PlaceHolderId id) NN_NOEXCEPT;

        /**
        * @brief    コンテントストレージの空き容量を取得します
        *
        * @param[out]    outValue   空き容量
        */
        Result GetFreeSpaceSize(sf::Out<std::int64_t> outValue) const NN_NOEXCEPT;

        /**
        * @brief    コンテントストレージの総容量を取得します
        *
        * @param[out]    outValue   総容量
        */
        Result GetTotalSpaceSize(sf::Out<std::int64_t> outValue) const NN_NOEXCEPT;

        Result GetPlaceHolderPath(sf::Out<Path> outValue, PlaceHolderId id) NN_NOEXCEPT;

        Result CleanupAllPlaceHolder() NN_NOEXCEPT;

        Result ListPlaceHolder(sf::Out<std::int32_t> outCount, const sf::OutArray<ncm::PlaceHolderId>& outList) const NN_NOEXCEPT;

        Result GetContentCount(nn::sf::Out<std::int32_t> outCount) const NN_NOEXCEPT;

        Result ListContentId(nn::sf::Out<std::int32_t> outCount, const nn::sf::OutArray<nn::ncm::ContentId>& outList, std::int32_t offset) const NN_NOEXCEPT;

        Result DisableForcibly() NN_NOEXCEPT;

        Result ReadContentIdFile(sf::OutBuffer buffer, ContentId id, int64_t offset) NN_NOEXCEPT;

        Result GetRightsIdFromPlaceHolderId(sf::Out<RightsId> outValue, PlaceHolderId id) NN_NOEXCEPT;

        Result GetRightsIdFromContentId(sf::Out<RightsId> outValue, ContentId id) const NN_NOEXCEPT;

        Result WriteContentForDebug(ContentId id, int64_t offset, sf::InBuffer buffer) NN_NOEXCEPT;

        Result FlushPlaceHolder() NN_NOEXCEPT;

        Result RepairInvalidFileAttribute() NN_NOEXCEPT;

    private:
        Result OpenContentIdFile(ContentId contentId) NN_NOEXCEPT;
        void InvalidateFileCache() NN_NOEXCEPT;

    private:
        PathString                  m_RootPath;
        MakeContentPathFunction     m_MakeContentPathFunction;
        bool                        m_IsDisabled;
        PlaceHolderAccessor         m_PlaceHolderAccessor;
        ContentId                   m_CacheContentId{};
        fs::FileHandle              m_CacheFileHandle{};
    };

}}
