﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Common.h>
#include <nn/nn_Result.h>
#include <nn/sf/sf_Types.h>
#include <nn/ncm/ncm_ProgramId.h>
#include <nn/ncm/ncm_ContentMetaId.h>
#include <nn/ncm/ncm_StorageId.h>

#include <nn/lr/lr_Path.h>

namespace nn { namespace lr {

    class IAddOnContentLocationResolver;
    class AddOnContentLocationResolver
    {
        NN_DISALLOW_COPY(AddOnContentLocationResolver);

    public:

        /**
        * @brief    コンストラクタです。
        */
        AddOnContentLocationResolver() NN_NOEXCEPT{}

        /**
        * @brief    コンストラクタです。
        */
        explicit AddOnContentLocationResolver(sf::SharedPointer<IAddOnContentLocationResolver> interface) NN_NOEXCEPT : m_Interface(interface) {}

        AddOnContentLocationResolver(AddOnContentLocationResolver&& rvalue) NN_NOEXCEPT
        {
            m_Interface = std::move(rvalue.m_Interface);
        }

        AddOnContentLocationResolver& operator=(AddOnContentLocationResolver&& rvalue) NN_NOEXCEPT
        {
            AddOnContentLocationResolver(std::move(rvalue)).swap(*this);

            return *this;
        }

        void swap(AddOnContentLocationResolver& other) NN_NOEXCEPT
        {
            std::swap(m_Interface, other.m_Interface);

        }
        /**
         * @brief   あらかじめ RegisterAddOnContentPath で登録したコンテンツのパスを取得します。
         *
         * @detail  該当するプログラム ID が登録されていない場合、ResultAddOnContentNotFound を返します。
         *
         * @param[out]   outValue    プログラムパスを取得するバッファ
         * @param[in]    id          プログラムパスを取得するプログラムの ID
         *
         * @return   処理の結果が返ります。
         * @retval   ResultAddOnContentNotFound   指定されたプログラムが登録されていません。
         */
        Result ResolveAddOnContentPath(Path* outValue, ncm::DataId id) NN_NOEXCEPT;

        /**
         * @brief   データ ID と、それに対応するコンテンツのパスを lr に登録します。
         *
         * @detail  登録したパスは、ResolveAddOnContentPath で取得できます。
         *          同時に登録可能なコンテンツの数は 2048 つです。
         *          2049個以上登録しようとした場合は ResultTooManyPathRegistered が返ります。
         *
         * @param[in]    id    プログラムの ID
         * @param[in]    path  対応するパス
         *
         * @return   処理の結果が返ります。
         */
        Result RegisterAddOnContentStorage(ncm::DataId id, ncm::StorageId storageId) NN_NOEXCEPT;

        /**
         * @brief   登録済みの全ての ID と、それに対応するコンテンツのパスを削除します。
         *
         * @return   なし。
         */
        void UnregisterAllAddOnContentPath() NN_NOEXCEPT;

    private:
        sf::SharedPointer<IAddOnContentLocationResolver> m_Interface;
    };
}}
