﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <algorithm>
#include <cstring>
#include <map>
#include <memory>
#include <nn/nn_Common.h>
#include <nn/nn_Result.h>
#include <nn/nn_SdkAssert.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/fs/fs_FileSystem.h>
#include <nn/fs/fs_File.h>
#include <nn/fs/fs_Result.h>
#include <nn/util/util_ScopeExit.h>
#include <nn/os/os_Tick.h>
#include <nn/kvdb/kvdb_AutoBuffer.h>
#include <nn/kvdb/kvdb_BoundedString.h>
#include <nn/kvdb/kvdb_KeyValueArchive.h>
#include <nn/kvdb/kvdb_Result.h>
#include <nn/kvdb/kvdb_Log.h>

namespace nn { namespace kvdb {

    template<class Key>
    class InMemoryKeyValueStore
    {
        NN_DISALLOW_COPY(InMemoryKeyValueStore);
        NN_DISALLOW_MOVE(InMemoryKeyValueStore);

    private:
        typedef nn::kvdb::AutoBuffer Value;
        typedef std::map<Key, Value> MapType;

    public:
        class Iterator
        {
            typedef typename MapType::iterator ImplType;

        public:

            Iterator() NN_NOEXCEPT{}
            Iterator(ImplType begin, ImplType end) NN_NOEXCEPT : m_Begin(begin), m_End(end){}

            Key Get() const
            {
                return (*m_Begin).first;
            }

            const void* GetPointer() const
            {
                return (*m_Begin).second.Get();
            }

            size_t GetSize() const
            {
                return (*m_Begin).second.GetSize();
            }

            void Next()
            {
                ++m_Begin;
            }

            bool IsEnd() const
            {
                return m_Begin == m_End;
            }

        private:
            ImplType m_Begin;
            ImplType m_End;
        };

        InMemoryKeyValueStore() NN_NOEXCEPT {}

        /**
        * @brief    キーバリューストアが永続化に利用するディレクトリを設定します。
        *
        */
        Result Initialize(const char directoryPath[]) NN_NOEXCEPT
        {
            fs::DirectoryEntryType entryType;
            NN_RESULT_DO(fs::GetEntryType(&entryType, directoryPath));
            NN_RESULT_THROW_UNLESS(entryType == fs::DirectoryEntryType_Directory, fs::ResultPathNotFound());
            NN_RESULT_THROW_UNLESS(strnlen(directoryPath, 769) <= 768, fs::ResultPathNotFound());

            m_ArchivePath.AssignFormat("%s/imkvdb.arc", directoryPath);
            m_ArchiveTemporaryPath.AssignFormat("%s/imkvdb.tmp", directoryPath);

            NN_RESULT_SUCCESS;
        }

        /**
        * @brief    ファイルシステムからキーバリューをロードします。
        *
        * @detail   ファイルシステムに何も永続化していない時はキーバリューがクリアされて成功します。
        *
        */
        Result Load() NN_NOEXCEPT
        {
            NN_SDK_ASSERT( m_ArchivePath.GetLength() > 0 );
            NN_SDK_ASSERT( m_ArchiveTemporaryPath.GetLength() > 0 );

            m_Impl.clear();

            nn::kvdb::AutoBuffer buffer;
            NN_RESULT_TRY(ReadArchive(&buffer))
                NN_RESULT_CATCH(fs::ResultPathNotFound)
                {
                    NN_RESULT_SUCCESS;
                }
            NN_RESULT_END_TRY

            NN_RESULT_DO(LoadFrom(buffer.Get(), buffer.GetSize()));

            NN_RESULT_SUCCESS;
        }

        /**
        * @brief    ファイルシステムにキーバリューを永続化します。
        * @details  ファイルシステムがトランザクション機能を有している場合、destructive を true にすることで更新に必要な空き容量を小さくできます。
        * @params[in] destructive 前回永続化されていたデータを破壊的に更新します。
        */
        Result Save() NN_NOEXCEPT
        {
            return Save(false);
        }

        Result Save(bool destructive) NN_NOEXCEPT
        {
            NN_SDK_ASSERT( m_ArchivePath.GetLength() > 0 );
            NN_SDK_ASSERT( m_ArchiveTemporaryPath.GetLength() > 0 );

            auto archiveSize = CalculateArchiveSize();
            nn::kvdb::AutoBuffer buffer;
            NN_RESULT_DO(buffer.Initialize(archiveSize));

            nn::kvdb::KeyValueArchiveBufferWriter writer(buffer.Get(), buffer.GetSize());
            SaveTo(&writer);

            NN_RESULT_DO(CommitArchive(buffer.Get(), buffer.GetSize(), destructive));

            NN_RESULT_SUCCESS;
        }

        /**
        * @brief    キーに対応する値を配置します。既に同じキーが存在する場合は上書きします。
        *
        * @return   処理の結果が返ります。想定外のエラーが無ければ常に成功します。
        *
        * @post
        *           - Get で当該キーの値が取得できる
        */
        Result Put(const Key& key, const void* data, size_t size) NN_NOEXCEPT
        {
            Value value;
            NN_RESULT_DO(value.Initialize(data, size));
            m_Impl[key] = std::move(value);

            NN_RESULT_SUCCESS;
        }

        /**
        * @brief    キーに対応する値を取得します。
        *
        * @return   処理の結果が返ります。
        * @retval   ResultKeyNotFound   キーが存在しません。
        *
        * @post
        *           - outValue にキーに対応する値が書き込まれている
        */
        Result Get(size_t* outValue, const Key& key, void* buffer, size_t size) NN_NOEXCEPT
        {
            auto iter = m_Impl.find(key);
            NN_RESULT_THROW_UNLESS(iter != m_Impl.end(), nn::kvdb::ResultKeyNotFound());

            Value& value = (*iter).second;
            auto readSize = std::min(size, value.GetSize());
            std::memcpy(buffer, value.Get(), readSize);

            *outValue = readSize;

            NN_RESULT_SUCCESS;
        }

        /**
        * @brief    キーに対応する値のポインタを取得します。
        *
        * @return   処理の結果が返ります。
        * @retval   ResultKeyNotFound   キーが存在しません。
        *
        * @post
        *           - outValue にキーに対応する値のポインタが書き込まれている。
        */
        Result GetPointer(void** outValue, const Key& key) NN_NOEXCEPT
        {
            auto iter = m_Impl.find(key);
            NN_RESULT_THROW_UNLESS(iter != m_Impl.end(), nn::kvdb::ResultKeyNotFound());

            Value& value = (*iter).second;
            *outValue = value.Get();

            NN_RESULT_SUCCESS;
        }

        Result GetSize(size_t* outValue, const Key& key) NN_NOEXCEPT
        {
            auto iter = m_Impl.find(key);
            NN_RESULT_THROW_UNLESS(iter != m_Impl.end(), nn::kvdb::ResultKeyNotFound());

            *outValue = (*iter).second.GetSize();

            NN_RESULT_SUCCESS;
        }

        /**
        * @brief    キーに対応する値を削除します。
        *
        * @return   処理の結果が返ります。
        * @retval   ResultKeyNotFound   キーが存在しません。
        *
        * @post
        *           - Get で当該キーの値が取得できない
        */
        Result Delete(const Key& key) NN_NOEXCEPT
        {
            auto erasedCount = m_Impl.erase(key);
            NN_RESULT_THROW_UNLESS(erasedCount != 0, nn::kvdb::ResultKeyNotFound());

            NN_RESULT_SUCCESS;
        }

        /**
        * @brief    イテレータを作成します。
        *
        * @return   処理の結果が返ります。想定外のエラーが無ければ常に成功します。
        *
        */
        Iterator GetBeginIterator() NN_NOEXCEPT
        {
            return Iterator(m_Impl.begin(), m_Impl.end());
        }

        Iterator GetLowerBoundIterator(const Key& key) NN_NOEXCEPT
        {
            return Iterator(m_Impl.lower_bound(key), m_Impl.end());
        }

        int Count() const NN_NOEXCEPT
        {
            return m_Impl.size();
        }

        static const size_t MaxPathLength = 788;
    private:
        MapType m_Impl;
        typedef BoundedString<MaxPathLength> Path;
        Path m_ArchivePath;
        Path m_ArchiveTemporaryPath;

        size_t CalculateArchiveSize() NN_NOEXCEPT
        {
            nn::kvdb::KeyValueArchiveSizeCalculator calculator;

            for (auto& kv : m_Impl)
            {
                calculator.AddEntry(sizeof(Key), kv.second.GetSize());
            }

            return calculator.GetSize();
        }

        Result LoadFrom(const char* buffer, size_t size) NN_NOEXCEPT
        {
            NN_KVDB_PERF_LOG("Begin %lld ms\n", nn::os::GetSystemTick().ToTimeSpan().GetMilliSeconds());

            nn::kvdb::KeyValueArchiveBufferReader reader(buffer, size);

            uint32_t entryCount;
            NN_RESULT_DO(reader.ReadEntryCount(&entryCount));

            for (uint32_t i = 0; i < entryCount; i++)
            {
                nn::kvdb::KeyValueArchiveBufferReader::KeyValueSize keyValueSize;
                NN_RESULT_DO(reader.GetKeyValueSize(&keyValueSize));

                Key key;
                Value value;
                NN_RESULT_DO(value.Initialize(keyValueSize.valueSize));
                NN_RESULT_DO(reader.ReadKeyValue(&key, sizeof(key), value.Get(), value.GetSize()));

                m_Impl[key] = std::move(value);
            }

            NN_KVDB_PERF_LOG("End %lld ms\n", nn::os::GetSystemTick().ToTimeSpan().GetMilliSeconds());
            NN_KVDB_PERF_LOG("entryCount %u\n", entryCount);

            NN_RESULT_SUCCESS;
        }

        Result ReadArchive(nn::kvdb::AutoBuffer* buffer) NN_NOEXCEPT
        {
            fs::FileHandle file;
            NN_RESULT_DO(fs::OpenFile(&file, m_ArchivePath, fs::OpenMode_Read));
            NN_UTIL_SCOPE_EXIT{ fs::CloseFile(file); };

            int64_t archiveSize;
            NN_RESULT_DO(fs::GetFileSize(&archiveSize, file));

            NN_RESULT_DO(buffer->Initialize(static_cast<size_t>(archiveSize)));
            NN_RESULT_DO(fs::ReadFile(file, 0, buffer->Get(), buffer->GetSize()));

            NN_RESULT_SUCCESS;
        }

        void SaveTo(nn::kvdb::KeyValueArchiveBufferWriter* writer) NN_NOEXCEPT
        {
            NN_KVDB_PERF_LOG("m_Impl.size() %u\n", m_Impl.size());
            NN_KVDB_PERF_LOG("Begin %lld ms\n", nn::os::GetSystemTick().ToTimeSpan().GetMilliSeconds());

            writer->WriteHeader(static_cast<uint32_t>(m_Impl.size()));
            for (auto& kv : m_Impl)
            {
                writer->WriteEntry(&kv.first, sizeof(kv.first), kv.second.Get(), kv.second.GetSize());
            }

            NN_KVDB_PERF_LOG("End %lld ms\n", nn::os::GetSystemTick().ToTimeSpan().GetMilliSeconds());
        }

        Result CommitArchive(const void* buffer, size_t size, bool destructive) NN_NOEXCEPT
        {
            if (destructive)
            {
                fs::DeleteFile(m_ArchivePath);
                NN_RESULT_DO(fs::CreateFile(m_ArchivePath, size));
                {
                    fs::FileHandle file;
                    NN_RESULT_DO(fs::OpenFile(&file, m_ArchivePath, fs::OpenMode_Write));
                    NN_UTIL_SCOPE_EXIT{ fs::CloseFile(file); };
                    NN_RESULT_DO(fs::WriteFile(file, 0, buffer, size, fs::WriteOption::MakeValue(fs::WriteOptionFlag_Flush)));
                }
            }
            else
            {
                // TODO: fs が強制 Create を実装したら対応する
                fs::DeleteFile(m_ArchiveTemporaryPath);
                NN_RESULT_DO(fs::CreateFile(m_ArchiveTemporaryPath, size));

                {
                    fs::FileHandle file;
                    NN_RESULT_DO(fs::OpenFile(&file, m_ArchiveTemporaryPath, fs::OpenMode_Write));
                    NN_UTIL_SCOPE_EXIT{ fs::CloseFile(file); };
                    NN_RESULT_DO(fs::WriteFile(file, 0, buffer, size, fs::WriteOption::MakeValue(fs::WriteOptionFlag_Flush)));
                }

                // TODO: fs が atomic rename を実装したら対応する
                fs::DeleteFile(m_ArchivePath);
                NN_RESULT_DO(fs::RenameFile(m_ArchiveTemporaryPath, m_ArchivePath));
            }

            NN_RESULT_SUCCESS;
        }
    };

}}
