﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
*   @file
*   @brief  ホストファイルシステムを実現するライブラリの API
*   @detail Htcfs プロトコルに基づいてホスト・ターゲット通信を行い、
*           ホストファイルシステムを実現するライブラリの API です。
*/

#pragma once

#include <nn/nn_Common.h>
#include <nn/nn_Result.h>

#include <nn/fs/fs_Directory.h>
#include <nn/fs/fs_File.h>
#include <nn/fs/fs_FileSystem.h>

#include <nn/htcfs/htcfs_Types.h>

namespace nn { namespace htcfs {

void Initialize() NN_NOEXCEPT;
void Finalize() NN_NOEXCEPT;

nn::Result GetEntryType(nn::fs::DirectoryEntryType* pOutType, const char* pPath) NN_NOEXCEPT;
nn::Result OpenFile(FileHandle* pOutHandle, const char* pPath, nn::fs::OpenMode mode) NN_NOEXCEPT;
void CloseFile(FileHandle handle) NN_NOEXCEPT;
nn::Result GetPriorityForFile(int32_t* pOutPriority, FileHandle handle) NN_NOEXCEPT;
nn::Result SetPriorityForFile(int32_t priority, FileHandle handle) NN_NOEXCEPT;
nn::Result CreateFile(const char* pPath, int64_t size) NN_NOEXCEPT;
nn::Result DeleteFile(const char* pPath) NN_NOEXCEPT;
nn::Result RenameFile(const char* pFromName, const char* pToName) NN_NOEXCEPT;
nn::Result FileExists(bool* pOutExists, const char* pPath) NN_NOEXCEPT;
nn::Result ReadFile(size_t* pOutSize, void* pOutBuffer, FileHandle handle, int64_t offset, size_t size, const nn::fs::ReadOption& option) NN_NOEXCEPT;
nn::Result WriteFile(const void* pBuffer, FileHandle handle, int64_t offset, size_t size, const nn::fs::WriteOption& option) NN_NOEXCEPT;
nn::Result FlushFile(FileHandle handle) NN_NOEXCEPT;
nn::Result GetFileTimeStamp(uint64_t* pOutCreateTime, uint64_t* pOutAccessTime, uint64_t* pOutModifyTime, const char* pPath) NN_NOEXCEPT;
nn::Result GetFileSize(int64_t* pOutSize, FileHandle handle) NN_NOEXCEPT;
nn::Result SetFileSize(int64_t size, FileHandle handle) NN_NOEXCEPT;
nn::Result OpenDirectory(DirectoryHandle* pOutHandle, const char* pPath, nn::fs::OpenDirectoryMode mode) NN_NOEXCEPT;
void CloseDirectory(DirectoryHandle handle) NN_NOEXCEPT;
nn::Result CreateDirectory(const char* pPath) NN_NOEXCEPT;
nn::Result DeleteDirectory(const char* pPath, bool recursively) NN_NOEXCEPT;
nn::Result RenameDirectory(const char* pFromName, const char* pToName) NN_NOEXCEPT;
nn::Result DirectoryExists(bool* pOutExists, const char* pPath) NN_NOEXCEPT;
nn::Result ReadDirectory(int64_t* pOutCount, nn::fs::DirectoryEntry* pOutBuffer, int64_t count, DirectoryHandle handle) NN_NOEXCEPT;
nn::Result GetEntryCount(int64_t* pOutCount, DirectoryHandle handle) NN_NOEXCEPT;

}}

