﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Common.h>

#include <nn/crypto/crypto_Aes128CbcDecryptor.h>
#include <nn/crypto/crypto_Aes128CbcEncryptor.h>
#include <nn/crypto/crypto_Aes128CtrDecryptor.h>
#include <nn/crypto/crypto_Aes128CtrEncryptor.h>
#include <nn/crypto/crypto_HmacSha256Generator.h>
#include <nn/crypto/crypto_Sha256Generator.h>

#include <nn/gc/gc.h>
#include <nn/gc/detail/gc_Counter.h>
#include <nn/gc/detail/gc_Util.h>

namespace nn { namespace gc {
namespace detail {

enum LastAesCtrOperation
{
    LastAesCtrOperation_None,
    LastAesCtrOperation_Decryption,
    LastAesCtrOperation_Encryption
};



// 暗号関連の処理を行うクラス
class GcCrypto
{
    NN_DISALLOW_COPY(GcCrypto);

public:
    static const size_t GcSha256Length = 32;
    static const size_t GcHmacLength = GcSha256Length;

    static const size_t GcAesKeyLength = 16;
    static const size_t GcAsicSeriralNumberLength = 16;
    static const size_t GcRsaKeyLength = 256;
    static const size_t GcExponentLength = 3;
    static const size_t GcRsaBlockSize = GcRsaKeyLength;
    static const size_t GcRsaOaepLabelMaxLength = 256;
    static const size_t GcHmacKeyLength = 32;
    static const size_t GcCvConstLength = 16;
    static const size_t GcCvLength = 32;
    static const size_t GcCommonKeyLength = 16;
    static const size_t GcAesCbcIvLength = 16;
    static const size_t GcAes128BlockSize = 16;
    static const size_t GcCalibrationPartitionDecryptionKeyLength = 16;
    static const size_t GcCa1PublicModulusLength = 256;
    static const size_t GcCa1PublicExponentLength = 3;

    // *** セッション情報（セッション構築時に ASIC から受信したり、その場で生成する）ここから ***

    // ブリッジASIC証明書
    static char g_AsicCertBuffer[GcCertificateSize];

NN_DETAIL_GC_PRIVATE:
    // ブリッジASIC公開鍵
    char m_AsicSerialNumber[GcAsicSeriralNumberLength];
    char m_AsicPublicKeyModulus[GcRsaKeyLength];
    char m_AsicPublicKeyExponent[GcExponentLength];

    // セッションの共通鍵（メモリに浮くことは妥協：HWSEC-127）
    char m_CommonKey[GcCrypto::GcAesKeyLength];

    // 鍵更新に使用するための乱数RND6
    char m_RandomValueForKeyUpdateAsic[GcRandomValueSize];

    // 鍵更新に使用するための乱数RND5M
    char m_HashStockForKeyUpdate[GcSha256Length];
    bool m_IsHashStockForKeyUpdateReady;
    u64 m_KeyUpdateCount;

    // AES-CBC, AES-CTR用初期ベクトル（IV）
    char m_AesInitialVector[GcAesCbcIvLength];

    // AES CTR カウンタは 64bit 周り切りを検知するか、手前でリセットして避ける必要がある
    Counter128Bit m_SessionAesCtrCounter;
    // CV カウンタは特定 bit の周り切り検知は不要
    Counter128Bit m_SessionCvCounter;

    // *** セッション情報ここまで ***

    // 暗号機能
    nn::crypto::Sha256Generator m_Sha256Generator;
    nn::crypto::HmacSha256Generator m_HmacSha256Generator;
    LastAesCtrOperation m_LastAesCtrOperation;

public:
    // Singleton パターン
    static GcCrypto& GetInstance() NN_NOEXCEPT;
    bool IsProd() NN_NOEXCEPT;

    // GcCryptoクラス向け内部処理
    // CVカウンタはAES-CTRでの暗号化の前に使用される
    void SetRandomValuesForKeyUpdate(const char* randomValue5Buffer, const size_t randomValue5BufferLength, const char* randomValue6Buffer, const size_t randomValue6BufferLength) NN_NOEXCEPT;
    void UpdateSessionKey() NN_NOEXCEPT;

    bool IsResetSessionKeyNeeded() NN_NOEXCEPT
    {
        return m_SessionAesCtrCounter.IsCounterValueOver64bit();
    }

    // 外部向け関数
    void ClearKeys() NN_NOEXCEPT;
    nn::Result VerifyCertificateAndGetPublicKey(const char* certBuffer, const size_t bufferLength) NN_NOEXCEPT;
    // 共通鍵、AES-CBC/CTRのIV（CTRカウンタ初期値）, CVカウンタ初期値を生成・代入
    void GenerateCommonKeyFromRandomValues(const char* randomValue1Buffer, const size_t randomValue1BufferLength,
        const char* randomValue2Buffer, const size_t randomValue2BufferLength) NN_NOEXCEPT;

    // コマンド認証値をつける（outWorkBuffer はオペレーション 64byte 転送の値が格納されている必要がある）
    void AddCommandVerificationValue(char* outWorkBuffer, const size_t bufferLength) NN_NOEXCEPT;

    // 乱数生成、MAC、ハッシュ
    nn::Result GenerateRandomValue(char* outBuffer, const size_t bufferLength) NN_NOEXCEPT;
    void GenerateFixedValue(char* outBuffer, const size_t bufferLength) NN_NOEXCEPT;
    void CalcSha256(char* outBuffer, const size_t outBufferLength, const char* inBuffer, const size_t inBufferLength) NN_NOEXCEPT
    {
        m_Sha256Generator.Initialize();
        m_Sha256Generator.Update(inBuffer, inBufferLength);
        m_Sha256Generator.GetHash(outBuffer, outBufferLength);
    }

    // outBuffer == buffer を許容する
    // RSA-OAEP
    nn::Result DecryptWithRsaOaep(size_t* outLength, char* outBuffer, const size_t outBufferLength, const char* inBuffer, const size_t inBufferLength) NN_NOEXCEPT;
    void EncryptWithRsaOaep(char* outBuffer, const size_t outBufferLength, const char* inBuffer, const size_t inBufferLength) NN_NOEXCEPT;

    // AES-CBC
    void DecryptWithAesCbc(char* outBuffer, const size_t outBufferLength, const char* inBuffer, const size_t inBufferLength) NN_NOEXCEPT;
    void EncryptWithAesCbc(char* outBuffer, const size_t outBufferLength, const char* inBuffer, const size_t inBufferLength) NN_NOEXCEPT;
    // AES-CTR
    void DecryptWithAesCtr(char* outBuffer, const size_t outBufferLength, const char* inBuffer, const size_t inBufferLength) NN_NOEXCEPT;
    void DecryptWithAesCtrWithCounter(char* outBuffer, const size_t outBufferLength, const char* inBuffer, const size_t inBufferLength, const char* counterBuffer, const size_t counterLength) NN_NOEXCEPT;
    void EncryptWithAesCtr(char* outBuffer, const size_t outBufferLength, const char* inBuffer, const size_t inBufferLength) NN_NOEXCEPT;

    void GetCounter(char* pOutCounter, size_t outCounterLength) NN_NOEXCEPT
    {
        NN_DETAIL_GC_ABORT_UNLESS_SDK_REQUIRES( outCounterLength >= m_SessionAesCtrCounter.counterLength );
        memcpy(pOutCounter, m_SessionAesCtrCounter.valueCounterArray, m_SessionAesCtrCounter.counterLength);
    }
    void IncrementCounter(size_t length) NN_NOEXCEPT
    {
        // カウントアップ
        m_SessionAesCtrCounter.CountUp( static_cast<GcCounter128bitContainerType>(length / (size_t)GcAes128BlockSize) );
    }

    // ** spl 関連のラッパ
public:
    static bool CheckDevelopmentSpl() NN_NOEXCEPT;
    static nn::Result DecryptAesKeySpl(void* pOutBuffer, size_t outBufferSize,
        const void* pSource, size_t sourceSize, int generation, Bit32 option) NN_NOEXCEPT;
    static nn::Result DecryptAndStoreGcKeySpl(
        const void* pData,
        size_t      dataSize  ) NN_NOEXCEPT;
    static nn::Result DecryptGcMessageSpl(
        size_t*     pOutResultSize,
        void*       pResultBuffer,
        size_t      resultBufferSize,
        const void* pCipher,
        size_t      cipherSize,
        const void* pModulus,
        size_t      modulusSize,
        const void* pLabelDigest,
        size_t      labelDigestSize  ) NN_NOEXCEPT;
private:
    static nn::Result GenerateRandomBytesSpl(void* pOutBuffer, size_t bufferSize) NN_NOEXCEPT;

private:
    GcCrypto() NN_NOEXCEPT;
    virtual ~GcCrypto() NN_NOEXCEPT;
    void SetCvCounter(const char* randomValue1Buffer, const size_t randomValue1BufferLength, const char* randomValue2Buffer, const size_t randomValue2BufferLength) NN_NOEXCEPT;
    void UpdateCommonKeyFromRandomValues(const char* randomValue6Buffer, const size_t randomValue6BufferLength,
                                         const char* randomValue5mBuffer, const size_t randomValue5mBufferLength) NN_NOEXCEPT
    {
        GenerateCommonKeyFromRandomValues(randomValue6Buffer, randomValue6BufferLength, randomValue5mBuffer, randomValue5mBufferLength);
    }
};



} } }
