﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/dd.h>
#include <nn/nn_Result.h>
#include <nn/sdmmc/sdmmc_GcAsic.h>
#include <nn/gc/detail/gc_Types.h>
#include <nn/util/util_BitPack.h>

namespace nn { namespace gc {
namespace detail {

typedef nn::util::BitPack32 BitPack32;

// データの送受信周りを扱うクラス
class DataIo
{
    NN_DISALLOW_COPY(DataIo);

public:
    static const size_t SectorSize = nn::sdmmc::SectorSize;

public:
    // Singleton パターン
    static DataIo& GetInstance() NN_NOEXCEPT;
    static void Initialize() NN_NOEXCEPT;
    nn::Result Finalize() NN_NOEXCEPT;

    void RegisterDeviceVirtualAddress(uintptr_t bufferAddress, size_t bufferSize, nn::dd::DeviceVirtualAddress bufferDeviceVirtualAddress) NN_NOEXCEPT;
    void UnregisterDeviceVirtualAddress(uintptr_t bufferAddress, size_t bufferSize, nn::dd::DeviceVirtualAddress bufferDeviceVirtualAddress) NN_NOEXCEPT;

    nn::Result Activate() NN_NOEXCEPT;
    void Deactivate() NN_NOEXCEPT;

    // MMCコマンド：オペレーション
    // workAlignedOperationBuffer は、GcMmcCmd60DataSize の長さが必要で、中身は変更されうる
    nn::Result SendOperationStart(char* workAlignedOperationBuffer, const size_t bufferLength) NN_NOEXCEPT;

    nn::Result AbortGcAsicOperation() NN_NOEXCEPT;
    nn::Result FinishOperation() NN_NOEXCEPT;

    // bufferLength はセクタサイズ切り捨て
    nn::Result SendCommandDataRead(char* outAlignedBuffer, const size_t bufferLength) NN_NOEXCEPT;
    // bufferLength はセクタサイズ切り捨て
    // workAlignedDataBuffer には送信データが入っており、このバッファは暗号化データで上書きされる
    nn::Result SendCommandDataWrite(char* workAlignedDataBuffer, const size_t bufferLength) NN_NOEXCEPT;

    // ステータス取得
    nn::Result GetDeviceStatus(BitPack32 *outStatus) NN_NOEXCEPT;
    void PrintDeviceStatus() NN_NOEXCEPT;

    // MMC特殊コマンド
    nn::Result PutGcAsicToSleep() NN_NOEXCEPT;
    nn::Result AwakenGcAsic() NN_NOEXCEPT;
    nn::Result UpdateKey() NN_NOEXCEPT;

    // MMC 転送処理の中断
    void SignalRemovedEvent() NN_NOEXCEPT;
    void ClearRemovedEvent() NN_NOEXCEPT;

    // MMC スリープ処理
    void PutSdmmcToSleep() NN_NOEXCEPT;
    nn::Result AwakenSdmmc() NN_NOEXCEPT;

    // MMC Transfer Timeout 設定
    void SetTransferTimeout(uint32_t milliSeconds);
    void SetDefaultTransferTimeout();

private:
    DataIo() NN_NOEXCEPT;

    // sdmmc 関数のポインタの定義
    typedef nn::Result (*sdmmcCommandFunctionPointer)(nn::sdmmc::Port);
    typedef nn::Result (*sdmmcCommandFunctionWithBufferPointer)(nn::sdmmc::Port, const void*, size_t);
    nn::Result SendSdmmcCommandWithRetry(sdmmcCommandFunctionPointer pFunction) NN_NOEXCEPT;
    nn::Result SendSdmmcCommandWithRetry(sdmmcCommandFunctionWithBufferPointer pFunction, const void* operationBuffer, const size_t operationBufferSize) NN_NOEXCEPT;

private:
    int m_MaxRetryCount;
    bool m_IsDeviceVirtualAddressRegistered;
    static char* g_SdmmcWorkAlignedBuffer;
    static size_t g_SdmmcWorkAlignedBufferLength;
    static const nn::sdmmc::Port m_GcPort = nn::sdmmc::Port_GcAsic0;
};

struct AsicDeviceStatus
{
    typedef BitPack32::Field<2, 2, u8> asicError;
};

} } }
