﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Result.h>
#include <nn/dd.h>
#include <nn/os.h>
#include <nn/os/os_SystemEvent.h>

#include <nn/gc/gc.h>

#include <nn/gc/detail/gc_ThreadInterface.h>
#include <nn/gc/detail/gc_Util.h>

namespace nn { namespace gc {
namespace detail {

// この構造体の変数は起動時からの通算の値
struct TotalAsicInfo
{
    // ASIC の初期化回数
    uint32_t initializeCount;
    // Awaken が呼ばれた回数
    uint32_t awakenCount;
    // Awaken に失敗した回数
    uint16_t awakenFailureNum;
    char reserved[2];
};

// カードアクセスに関するカウントのうち、特にカード挿入 or スリープでリセットされるもの
struct CardAccessInternalInfo
{
    uint16_t retryLimitOutNum;
    uint16_t asicReinitializeNum;
    uint16_t asicReinitializeFailureNum;
    uint16_t refreshSeccessNum;
    uint32_t lastReadErrorPageAddress;
    uint32_t lastReadErrorPageCount;
    // 挿入以降 Read が呼ばれた回数
    uint32_t readCountFromInsert;
    // Awaken 以降 Read が呼ばれた回数
    uint32_t readCountFromAwaken;
    nn::Result lastReinitializeFailureResult;
};

class AsicHandler;
// このクラスを呼んでいいのは基本的に AsicHandler だけ：例外は gc.cpp と Writer のみ
class AsicHandlerCore
{
    NN_DISALLOW_COPY(AsicHandlerCore);
    friend class AsicHandler;

private:
    static const int BusWaitTimeMsec = 250;
    static const size_t CardHeaderAllAreaSize = 0x200;
    static const int InitializationFailureMaxCount = 3;

    // 挿入されているカードの ID
    CardId1* const m_pCardId1;  // CardId1 は CardInitReceiveData を参照する
    CardUid* const m_pCardUid;  // CardUid は CardSecurityInformation を参照する
    TotalAsicInfo m_TotalAsicInfo;
    CardAccessInternalInfo m_CardAccessInternalInfo;
    // スリープ前の情報
    struct {
        bool isInserted;
        GameCardMode cardMode;
        char cardHeader[sizeof(CardHeader)];
        char deviceId[CardDeviceIdLength];
    } m_CardInfoBeforeSleep;


    // セッション構築失敗フラグ（初回を除き、常にセッション構築の成功・失敗の結果が入っている）
    nn::Result m_InitializationResult;
    // セッション構築連続失敗回数
    int m_InitializationFailureCount;
    // セッション単位の抜去フラグ（抜去されたら true っぱなし, Reset, Activate 時にクリアする）
    bool m_IsRemovedAfterActivate;

    // Reset で消去する状態保持メンバ
    struct {
        union
        {
            CardInitReceiveData m_CardInitReceiveData;
            char m_CardInitReceiveDataArray[ sizeof(CardInitReceiveData) ];
        };
        union
        {
            CardHeader m_CardHeader;
            char m_CardHeaderArray[ sizeof(CardHeader) ];
        };
        union
        {
            CardSecurityInformation m_CardSecurityInformation;
            char m_CardSecurityInformationAray[ sizeof(CardSecurityInformation) ];
        };

        char m_CardImageHash[GcCardImageHashSize];

        CardId2 m_CardId2;          // CardId2 は CardSecurityInformation を参照できるが、一応 Normal モードで参照できるように明示的に取得する
        CardId3 m_CardId3;          // CardId3 は何にもついてこないので明示的に取得

        // カードバス有効化フラグ（sdmmc 転送中断の判定に利用）
        bool m_IsCardBusEnabled;
        // スリープフラグ
        bool m_IsAsleep;

        // リード直後のデバイス状態取得結果
        nn::Result m_DeviceStatusAfterRead;
    } m_SessionInfo;

    // 抜去通知に関する排他（DeviceDetector と AsicHandler 間の排他）
    mutable nn::os::Mutex m_AsicMutex;

protected:
    // AsicHandler スレッドのメインループから呼んでもらう関数
    void Reset() NN_NOEXCEPT;
    nn::Result ProcessInsert() NN_NOEXCEPT;
    nn::Result ProcessRemove() NN_NOEXCEPT;
    nn::Result GetInitializationResult() NN_NOEXCEPT
    {
        return m_InitializationResult;
    }

private:
    // 外部から依頼される仕事の実体となる API
    nn::Result ActivateCard() NN_NOEXCEPT;
    nn::Result ActivateCard(bool isRetryNeeded) NN_NOEXCEPT;
    void DeactivateCard() NN_NOEXCEPT;
    nn::Result SetCardToSecureMode() NN_NOEXCEPT;
    nn::Result PutToSleep() NN_NOEXCEPT;
    nn::Result Awaken() NN_NOEXCEPT;
    nn::Result Read(ReadInfo* pReadInfo) NN_NOEXCEPT;
    nn::Result GetCardStatus(ReadInfo* pReadInfo) NN_NOEXCEPT;
    nn::Result GetCardDeviceId(ReadInfo* pReadInfo) NN_NOEXCEPT;
    nn::Result GetCardDeviceCertificate(ReadInfo* pReadInfo) NN_NOEXCEPT;
    nn::Result GetCardImageHash(ReadInfo* pReadInfo) NN_NOEXCEPT;
    nn::Result GetCardIdSet(ReadInfo* pReadInfo) NN_NOEXCEPT;
    nn::Result GetCardHeader(ReadInfo* pReadInfo) NN_NOEXCEPT;
    nn::Result GetErrorInfo(ReadInfo* pReadInfo) NN_NOEXCEPT;

private:
    // 雑多な内部 API
    bool IsRemoved() NN_NOEXCEPT;
    bool IsCardBusEnabled() NN_NOEXCEPT
    {
        return m_SessionInfo.m_IsCardBusEnabled;
    }
    bool IsAsleep() NN_NOEXCEPT
    {
        return m_SessionInfo.m_IsAsleep;
    }
    uint64_t GetCardSize() NN_NOEXCEPT;
    uint64_t GetNormalAreaSize() NN_NOEXCEPT;
    nn::Result GetCardDeviceId(char* outBuffer, const size_t bufferLength) NN_NOEXCEPT;
    nn::Result GetLastDeviceStatusAfterRead() NN_NOEXCEPT
    {
        return m_SessionInfo.m_DeviceStatusAfterRead;
    }

    // エラーハンドリングまわり
    nn::Result InitializeAsicIfNeeded() NN_NOEXCEPT;
    nn::Result CheckCardInsertionToUpdateKey() NN_NOEXCEPT;
    nn::Result ExecuteOperationWithRetry(nn::Result (AsicHandlerCore::*operationFunc)()) NN_NOEXCEPT;

    // 低レイヤの ASIC 通信 API：ExecuteOperationWithRetry などを通して呼ぶ
    nn::Result EnableCardBus() NN_NOEXCEPT;
    nn::Result GetCardHeader() NN_NOEXCEPT;
    nn::Result ChangeGcModeToSecure() NN_NOEXCEPT;
    nn::Result ReadWithErrorHandling(char* outDataBuffer, const size_t outDataBufferLength, const u32 pageAddress, const u32 pageCount) NN_NOEXCEPT;
    nn::Result HandleReadError() NN_NOEXCEPT;
    nn::Result ReinitializeAsicWhenReadFailed() NN_NOEXCEPT;
    void       EvacuateCardInfoAndInitializeAsicIfNeeded() NN_NOEXCEPT;
    nn::Result ResumeCardStateAndCheckCardInfo(bool isRetryNeeded) NN_NOEXCEPT;

public:
    static AsicHandlerCore& GetInstance() NN_NOEXCEPT;

    // spl のエラーが起きたとき
    void SetSplFatalErrorOccurredFlag(nn::Result result) NN_NOEXCEPT;

    // Activate が有効な状態かチェック（gc.cpp 向け）
    bool IsActivationValid() NN_NOEXCEPT;

    // 挿抜状態の確認
    nn::Result CheckCardReady() NN_NOEXCEPT;

    // 転送中断用コールバック関数
    static void ProcessCardRemovalForCallback(void *pParameter) NN_NOEXCEPT;
    void AbortCommunication() NN_NOEXCEPT;

    // writer からも呼ばれるもの
    nn::Result InvalidateAsicIfResultFailure(nn::Result result) NN_NOEXCEPT;
    nn::Result EnableCardBusForWriter() NN_NOEXCEPT;

    // エラー情報
    void IncrementAwakenFailureCount() NN_NOEXCEPT
    {
        m_TotalAsicInfo.awakenFailureNum++;
    }

    // 内部情報のクリア
    void ClearSessionInfo() NN_NOEXCEPT;
    void ClearSessionInfoForErrorReport() NN_NOEXCEPT;

private:
    AsicHandlerCore() NN_NOEXCEPT;
};

// マクロ
#define NN_DETAIL_GC_RESULT_THROW_UNLESS_AWAKENED  if(IsAsleep()) { return nn::fs::ResultGameCardNotAwakened(); }
#define NN_DETAIL_GC_RESULT_THROW_IF_REMOVED if(IsRemoved()) { return nn::fs::ResultGameCardCardNotInserted(); }
#define NN_DETAIL_GC_RESULT_THROW_UNLESS_ACTIVATED  if(StateMachine::GetInstance().IsCardActivated() == false) { return nn::fs::ResultGameCardCardNotActivated(); }

#define NN_DETAIL_GC_CANCEL_SDMMC_LOCK_GUARD   std::lock_guard<nn::os::Mutex> lock(m_AsicMutex);



// 通常シーケンスにおいて通信エラーを想定しないもの
class AsicHandlerProcedure
{
private:
    static const int GcPostReleaseResetWaitTimeUnitNsec = 10;
    static const int GcPostReleaseResetWaitTimeMaxNsec = 1000;

public:
    static nn::Result InitializeAsicImpl() NN_NOEXCEPT;

    // ASIC の電源制御込の API
    static nn::Result ActivateAsicIo() NN_NOEXCEPT;
    static nn::Result DeactivateAsicIo() NN_NOEXCEPT;

    static nn::Result UpdateKey() NN_NOEXCEPT;
    static nn::Result VerifyCertificateMutually() NN_NOEXCEPT;
    static nn::Result GenerateCommonKey() NN_NOEXCEPT;
    static nn::Result AuthenticateMutually() NN_NOEXCEPT;
    static nn::Result ExchangeRandomValues() NN_NOEXCEPT;
};

} } }
