﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <atomic>
#include <iterator>
#include <vector>
#include <nn/nn_StaticAssert.h>
#include <nn/os/os_Mutex.h>
#include <nn/fs/fs_SubStorage.h>

namespace nn { namespace fssystem { namespace utilTool {

/**
 * @brief   データの比較結果の詳細です。
 */
enum BinaryMatchDetail
{
    BinaryMatchDetail_Unknown,      // 未設定
    BinaryMatchDetail_Match,        // 旧パッチと同じデータの領域
    BinaryMatchDetail_Padding,      // 新パッチに追加された無駄な領域
    BinaryMatchDetail_Anchor,       // Match の後続領域として移動
    BinaryMatchDetail_SmallFit,     // 小さい領域をファーストフィット方式で移動
    BinaryMatchDetail_LargeSplit,   // 大きな領域を分割して移動
    BinaryMatchDetail_LeftJustify,  // 左詰めで移動
    BinaryMatchDetail_SoftSplit,    // 分割数の上限を決めて前詰め
    BinaryMatchDetail_ForceSplit,   // 強制的に分割して前詰め
    BinaryMatchDetail_RightJustify, // ストレージの末尾に後詰め
    BinaryMatchDetail_Max
};

/**
 * @brief   データの比較結果です。
 */
struct BinaryMatchResult
{
    static const int64_t InvalidOffset = -1;

    int64_t newOffset;
    int64_t oldOffset;
    int64_t size;
    int32_t storageIndex;
    int16_t detail;
    int16_t detailIndex;

    bool operator<(const BinaryMatchResult& other) const NN_NOEXCEPT
    {
        return this->newOffset < other.newOffset;
    }

    bool CanMerge(const BinaryMatchResult& next) const NN_NOEXCEPT
    {
        // this,next 両方未設定 or 両方設定済み
        if( ((this->detail == BinaryMatchDetail_Unknown) == (next.detail == BinaryMatchDetail_Unknown))
            && (this->newOffset + this->size == next.newOffset) )
        {
            // 新しいデータが連続
            if( this->oldOffset == InvalidOffset && next.oldOffset == InvalidOffset )
            {
                return true;
            }
            // 古いデータまたはセルフマッチが連続
            if( this->oldOffset != InvalidOffset && next.oldOffset != InvalidOffset )
            {
                return this->oldOffset + this->size == next.oldOffset
                    && this->storageIndex == next.storageIndex;
            }
        }
        return false;
    }

    int64_t GetNewEndOffset() const NN_NOEXCEPT
    {
        return this->newOffset + this->size;
    }

    int64_t GetOldEndOffset() const NN_NOEXCEPT
    {
        return this->oldOffset + this->size;
    }

    static const BinaryMatchResult MakeMatch(
        int64_t newOffset, int64_t oldOffset, int64_t size) NN_NOEXCEPT
    {
        BinaryMatchResult result =
        {
            newOffset,
            oldOffset,
            size,
            0,
            BinaryMatchDetail_Match,
            0
        };
        return result;
    }

    static const BinaryMatchResult MakeUnknown(
        int64_t newOffset, int64_t size) NN_NOEXCEPT
    {
        BinaryMatchResult result =
        {
            newOffset,
            InvalidOffset,
            size,
            1,
            BinaryMatchDetail_Unknown,
            0
        };
        return result;
    }

    static const BinaryMatchResult MakeUnknown(
        int64_t newOffset, int64_t oldOffset, int64_t size) NN_NOEXCEPT
    {
        BinaryMatchResult result =
        {
            newOffset,
            oldOffset,
            size,
            1,
            BinaryMatchDetail_Unknown,
            0
        };
        return result;
    }

    static const BinaryMatchResult MakeAnchor(
        int64_t newOffset, int64_t oldOffset, int64_t size) NN_NOEXCEPT
    {
        BinaryMatchResult result =
        {
            newOffset,
            oldOffset,
            size,
            1,
            BinaryMatchDetail_Anchor,
            0
        };
        return result;
    }

    static const BinaryMatchResult MakeSmallFit(
        int64_t newOffset, int64_t oldOffset, int64_t size) NN_NOEXCEPT
    {
        BinaryMatchResult result =
        {
            newOffset,
            oldOffset,
            size,
            1,
            BinaryMatchDetail_SmallFit,
            0
        };
        return result;
    }

    static const BinaryMatchResult MakeLeftJustify(
        int64_t newOffset, int64_t oldOffset, int64_t size) NN_NOEXCEPT
    {
        BinaryMatchResult result =
        {
            newOffset,
            oldOffset,
            size,
            1,
            BinaryMatchDetail_LeftJustify,
            0
        };
        return result;
    }

    static const BinaryMatchResult MakeSoftSplit(
        int64_t newOffset, int64_t oldOffset, int64_t size,
        BinaryMatchDetail detail, int16_t detailIndex) NN_NOEXCEPT
    {
        BinaryMatchResult result =
        {
            newOffset,
            oldOffset,
            size,
            1,
            static_cast<int16_t>(detail),
            detailIndex
        };
        return result;
    }

    static const BinaryMatchResult MakeForceSplit(
        int64_t newOffset, int64_t oldOffset, int64_t size) NN_NOEXCEPT
    {
        BinaryMatchResult result =
        {
            newOffset,
            oldOffset,
            size,
            1,
            BinaryMatchDetail_ForceSplit,
            0
        };
        return result;
    }

    static const BinaryMatchResult MakeRightJustify(
        int64_t newOffset, int64_t oldOffset, int64_t size) NN_NOEXCEPT
    {
        BinaryMatchResult result =
        {
            newOffset,
            oldOffset,
            size,
            1,
            BinaryMatchDetail_RightJustify,
            0
        };
        return result;
    }

    static void Merge(std::vector<BinaryMatchResult>* pResults) NN_NOEXCEPT;
    static nn::Result Verify(const std::vector<BinaryMatchResult>& results) NN_NOEXCEPT;
};
NN_STATIC_ASSERT(std::is_pod<BinaryMatchResult>::value);

/**
 * @brief   データの比較対象外の領域です。
 */
struct BinaryExcludeRange
{
    int64_t offset;
    int64_t size;
};
NN_STATIC_ASSERT(std::is_pod<BinaryExcludeRange>::value);

/**
 * @brief   データを比較する際のヒント情報です。
 */
struct BinaryMatchHint
{
    int64_t oldOffset;
    int64_t oldSize;
    int64_t newOffset;
    int64_t newSize;
};
NN_STATIC_ASSERT(std::is_pod<BinaryMatchHint>::value);

/**
 * @brief   ブロックハッシュです。
 *
 * ブロックハッシュまたはリージョンハッシュの計算方法を変更した場合は、
 * BinaryRegionFileHeader::Version をインクリメントしてください。
 * さもないと、不正なリージョンハッシュのキャッシュを使用することになります。
 */
struct BinaryBlockHash
{
    static const int BlockSize = 16;
    static const int ValueCount = 4;
    static const int RotateCount = 128;

    uint32_t value[ValueCount];

    static uint32_t RotateLeft(uint32_t value, int shift) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_RANGE(shift, 0, NN_BITSIZEOF(uint32_t));

        // これはアセンブラの1命令 (ror or rol) に変換される
        return (value << shift) | (value >> (32 - shift));
    }

    void Set(const char* data, size_t dataSize) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_NOT_NULL(data);
        NN_SDK_REQUIRES_EQUAL(dataSize, static_cast<size_t>(BlockSize));
        NN_UNUSED(dataSize);

        uint32_t num[BlockSize / sizeof(uint32_t)];
        std::memcpy(num, data, sizeof(num));

        num[0] -= RotateLeft(num[1], 1);
        num[1] -= RotateLeft(num[2], 3);
        num[2] -= RotateLeft(num[3], 5);
        num[3] -= RotateLeft(num[0], 7);

        num[0] -= RotateLeft(num[1], 11);
        num[1] -= RotateLeft(num[2], 13);
        num[2] -= RotateLeft(num[3], 17);
        num[3] -= RotateLeft(num[0], 19);

        num[0] -= RotateLeft(num[1], 23);
        num[1] -= RotateLeft(num[2], 27);
        num[2] -= RotateLeft(num[3], 29);
        num[3] -= RotateLeft(num[0], 31);

        this->value[0] = num[0];
        this->value[1] = num[1];
        this->value[2] = num[2];
        this->value[3] = num[3];
    }
};
NN_STATIC_ASSERT(std::is_pod<BinaryBlockHash>::value);

/**
 * @brief   リージョンハッシュです。
 *
 * ブロックハッシュまたはリージョンハッシュの計算方法を変更した場合は、
 * BinaryRegionFileHeader::Version をインクリメントしてください。
 * さもないと、不正なリージョンハッシュのキャッシュを使用することになります。
 */
struct BinaryRegionHash
{
    typedef BinaryBlockHash BlockHash;

    static const int BinCount = 8;
    static const int HashSize = BinCount * sizeof(uint32_t);

    uint32_t bins[BinCount];

    void RotateBins() NN_NOEXCEPT
    {
        // bins の後半を連結してシフトする
        const auto RotateBinCount = HashSize / sizeof(uint64_t) / 2;
        NN_STATIC_ASSERT((RotateBinCount & (RotateBinCount - 1)) == 0);

        // ワークバッファは SSE のレジスタ上に確保される
        uint64_t oldBins[RotateBinCount];
        std::memcpy(oldBins, &this->bins[BinCount / 2], sizeof(oldBins));

        // このあたりは SSE 命令に変換される
        uint64_t newBins[RotateBinCount];
        for( int i = 0; i < RotateBinCount; ++i )
        {
            newBins[i] = (oldBins[i] >> 1) | (oldBins[(i + 1) & (RotateBinCount - 1)] << 63);
        }

        std::memcpy(&this->bins[BinCount / 2], newBins, sizeof(newBins));
    }

    void Make(const char* data, int hashCount, size_t hashSize) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_NOT_NULL(data);
        NN_SDK_REQUIRES_GREATER(hashCount, 0);
        NN_SDK_REQUIRES_GREATER(hashSize, static_cast<size_t>(0));

        for( int hashIndex = 0; hashIndex < hashCount; ++hashIndex )
        {
            BlockHash hash;
            hash.Set(data + hashIndex * hashSize, hashSize);
            Add(hash);
        }
    }

    void Add(const BlockHash& hash) NN_NOEXCEPT
    {
        RotateBins();

        // 前半は add-sub
        this->bins[0] += hash.value[0];
        this->bins[1] += hash.value[1];
        this->bins[2] += hash.value[2];
        this->bins[3] += hash.value[3];
        // 後半は xor-shift
        this->bins[4] ^= hash.value[0];
        this->bins[5] ^= hash.value[1];
        this->bins[6] ^= hash.value[2];
        this->bins[7] ^= hash.value[3];
    }

    void Remove(const BlockHash& hash) NN_NOEXCEPT
    {
        // 前半は add-sub
        this->bins[0] -= hash.value[0];
        this->bins[1] -= hash.value[1];
        this->bins[2] -= hash.value[2];
        this->bins[3] -= hash.value[3];
        // 後半は xor-shift
        this->bins[4] ^= hash.value[0];
        this->bins[5] ^= hash.value[1];
        this->bins[6] ^= hash.value[2];
        this->bins[7] ^= hash.value[3];
    }

    uint16_t GetIndex() const NN_NOEXCEPT
    {
        // TODO: ビッグエンディアンでは別途対応が必要
        return static_cast<uint16_t>(this->bins[0] >> 16);
    }

    bool IsEqual(const char* data, int hashCount, size_t hashSize) const NN_NOEXCEPT
    {
        // 事前検証は Make() に任せる
        BinaryRegionHash other = {};
        other.Make(data, hashCount, hashSize);

        return *this == other;
    }

    int Compare(const BinaryRegionHash& other) const NN_NOEXCEPT
    {
        for( int i = 0; i < BinCount; ++i )
        {
            if( this->bins[i] < other.bins[i] )
            {
                return -1;
            }
            else if( other.bins[i] < this->bins[i] )
            {
                return 1;
            }
        }
        return 0;
    }

    bool operator==(const BinaryRegionHash& other) const NN_NOEXCEPT
    {
        return Compare(other) == 0;
    }
};
NN_STATIC_ASSERT(std::is_pod<BinaryRegionHash>::value);

/**
 * @brief   領域を表します。
 */
struct BinaryRegion
{
    // リージョンハッシュです
    BinaryRegionHash hash;

    // 比較元データのオフセットとサイズです
    // 同じリージョンハッシュが連続する場合は size がリージョンサイズの整数倍になります
    int64_t offset;
    int64_t size;

    // この領域と同じリージョンハッシュを持つ領域の最初および最後の次のインデックスです
    int beginIndex;
    int endIndex;

    bool operator<(const BinaryRegion& other) const NN_NOEXCEPT
    {
        int result = this->hash.Compare(other.hash);
        return result == 0 ? this->offset < other.offset : result < 0;
    }

    bool IsInside(int64_t position) const NN_NOEXCEPT
    {
        return (this->offset <= position) && (position < this->offset + this->size);
    }
};
NN_STATIC_ASSERT(std::is_pod<BinaryRegion>::value);

/**
 * @brief   BinaryRegion の配列を表します。
 */
class BinaryRegionArray
{
public:
    typedef BinaryRegion Region;
    typedef Region value_type;
    typedef int difference_type;
    typedef value_type* pointer;
    typedef const value_type* const_pointer;
    typedef value_type& reference;
    typedef const value_type& const_reference;
    typedef Region* iterator;
    typedef const Region* const_iterator;

public:
    BinaryRegionArray(const BinaryRegionArray&) = default;
    BinaryRegionArray& operator=(const BinaryRegionArray&) = default;

    BinaryRegionArray() NN_NOEXCEPT
        : m_Count(0)
        , m_Capacity(0)
        , m_pRegion(nullptr)
    {
    }

    BinaryRegionArray(Region* regionArray, int regionCount) NN_NOEXCEPT
        : m_Count(regionCount)
        , m_Capacity(regionCount)
        , m_pRegion(regionArray)
    {
    }

    void Reserve(Region* regionArray, int regionCount) NN_NOEXCEPT
    {
        m_Count = 0;
        m_Capacity = regionCount;
        m_pRegion = regionArray;
    }

    void clear() NN_NOEXCEPT
    {
        m_Count = 0;
    }

    void push_back(const_reference value) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES(IsValid());
        NN_SDK_REQUIRES_LESS(m_Count, m_Capacity);

        m_pRegion[m_Count] = value;
        ++m_Count;
    }

    void shrink_to_fit() NN_NOEXCEPT
    {
        m_Capacity = m_Count;
    }

    reference front() NN_NOEXCEPT
    {
        NN_SDK_REQUIRES(IsValid());
        NN_SDK_REQUIRES_GREATER(m_Count, 0);

        return m_pRegion[0];
    }

    const_reference front() const NN_NOEXCEPT
    {
        NN_SDK_REQUIRES(IsValid());
        NN_SDK_REQUIRES_GREATER(m_Count, 0);

        return m_pRegion[0];
    }

    reference back() NN_NOEXCEPT
    {
        NN_SDK_REQUIRES(IsValid());
        NN_SDK_REQUIRES_GREATER(m_Count, 0);

        return m_pRegion[m_Count - 1];
    }

    const_reference back() const NN_NOEXCEPT
    {
        NN_SDK_REQUIRES(IsValid());
        NN_SDK_REQUIRES_GREATER(m_Count, 0);

        return m_pRegion[m_Count - 1];
    }

    iterator begin() NN_NOEXCEPT
    {
        return m_pRegion;
    }

    const_iterator begin() const NN_NOEXCEPT
    {
        return m_pRegion;
    }

    const_iterator cbegin() const NN_NOEXCEPT
    {
        return begin();
    }

    iterator end() NN_NOEXCEPT
    {
        return m_pRegion + m_Count;
    }

    const_iterator end() const NN_NOEXCEPT
    {
        return m_pRegion + m_Count;
    }

    const_iterator cend() const NN_NOEXCEPT
    {
        return end();
    }

    pointer data() const NN_NOEXCEPT
    {
        return m_pRegion;
    }

    bool empty() const NN_NOEXCEPT
    {
        return m_Count == 0;
    }

    int size() const NN_NOEXCEPT
    {
        return m_Count;
    }

    bool IsValid() const NN_NOEXCEPT
    {
        return m_pRegion != nullptr;
    }

    size_t GetBytes() const NN_NOEXCEPT
    {
        return m_Count * sizeof(Region);
    }

private:
    int m_Count;
    int m_Capacity;
    Region* m_pRegion;
};

/**
* @brief    BinaryMatch の進捗状況です。
*/
enum class BinaryMatchPhase : int
{
    MakeRegionHash,
    FileBaseMatch,
    BinaryMatch1stPass,
    BinaryMatch2ndPass,
    Optimize,
    Count
};

/**
 * @brief   バイナリマッチの進捗を表すクラス
 */
class BinaryMatchProgress
{
    NN_DISALLOW_COPY(BinaryMatchProgress);
    NN_DISALLOW_MOVE(BinaryMatchProgress);

public:
    typedef BinaryMatchPhase Phase;

    struct Info
    {
        int phase;
        int64_t total;
        int64_t value;

        double GetRate() const NN_NOEXCEPT
        {
            if( this->total <= 0 )
            {
                return 0.0;
            }
            if( this->total <= this->value )
            {
                return 1.0;
            }
            return this->value / double(this->total);
        }
    };
    NN_STATIC_ASSERT(std::is_pod<Info>::value);

public:
    BinaryMatchProgress() NN_NOEXCEPT
        : m_Mutex(false)
        , m_Phase(0)
        , m_Total(0)
        , m_Value(0)
    {
    }

    void SetPhase(Phase phase, int64_t total) NN_NOEXCEPT;

    void SetValue(int64_t value) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_GREATER_EQUAL(value, 0);

        m_Value = value;
    }

    // NOTE: os::Mutex::lock() を呼ぶので const 関数にはしない
    const Info GetInfo() NN_NOEXCEPT;

private:
    os::Mutex m_Mutex;
    int m_Phase;
    int64_t m_Total;
    std::atomic<int64_t> m_Value;
};

/**
 * @brief   データの比較を行うクラスです。
 */
class BinaryMatch
{
public:
    typedef BinaryRegion Region;
    typedef BinaryRegionArray RegionArray;
    typedef std::unique_ptr<Region[]> RegionBuffer;
    typedef std::pair<RegionArray, RegionBuffer> RegionInfo;
    typedef BinaryRegionHash RegionHash;
    typedef BinaryMatchResult Result;
    typedef BinaryMatchHint Hint;
    typedef BinaryExcludeRange ExcludeRange;
    typedef BinaryMatchProgress Progress;

public:
    static const size_t RegionSizeMin = BinaryBlockHash::BlockSize * BinaryBlockHash::RotateCount;

public:
    // コンストラクタです。
    BinaryMatch(size_t blockSize, size_t regionSize, int64_t matchSize) NN_NOEXCEPT;

    // デストラクタです。
    ~BinaryMatch() NN_NOEXCEPT;

    // マッチングを実行します。
    nn::Result Run(fs::SubStorage oldStorage, fs::SubStorage newStorage, size_t shiftSize) NN_NOEXCEPT;

    // ウインドウサイズを設定します。
    void SetWindowSize(int64_t windowSize) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_GREATER(windowSize, 0);
        m_WindowSize = windowSize;
    }

    // ヒント情報を設定します。
    void SetHint(const Hint* hintArray, int hintCount) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES((hintArray == nullptr) ? (hintCount == 0) : (0 <= hintCount));

        if( 0 < hintCount )
        {
            m_Hints.assign(hintArray, hintArray + hintCount);
        }
    }

    // ヒント情報を設定します。
    void SetHint(std::vector<Hint>&& hints) NN_NOEXCEPT
    {
        m_Hints = std::move(hints);
    }

    // リージョンハッシュを設定します。
    void SetRegion(const RegionArray& region) NN_NOEXCEPT
    {
        m_Regions = region;
    }

    // リージョンハッシュを取得します。
    RegionInfo GetRegionInfo() NN_NOEXCEPT
    {
        return std::pair<RegionArray, RegionBuffer>(m_Regions, std::move(m_RegionBuffer));
    }

    // 古いストレージの比較しない領域を追加します。
    nn::Result AddExcludeRangeForOldStorage(const ExcludeRange& range) NN_NOEXCEPT
    {
        return AddExcludeRange(&m_ExcludeRangeOld, range);
    }

    // 新しいストレージの比較しない領域を追加します。
    nn::Result AddExcludeRangeForNewStorage(const ExcludeRange& range) NN_NOEXCEPT
    {
        return AddExcludeRange(&m_ExcludeRangeNew, range);
    }

    // マッチング情報を取得します。
    //
    // 結果は newOffset の昇順に隙間なく並べられたものが得られます。
    // 各要素は [newOffset, newOffset + size) が古いストレージまたは新しいストレージに含まれるいずれの領域に対応するかを示します。
    // storageIndex == 0 である場合、古いストレージの [oldOffset, oldOffset + size) に対応します。
    // storageIndex =! 0 である場合、新しいストレージに含まれるデータに対応します。
    // このとき、oldOffset == Result::InvalidOffset であれば、新しいストレージの [newOffset, newOffset + size) に対応します。
    // oldOffset != Result::InvalidOffset であれば、新しいストレージの [oldOffset, oldOffset + size) にセルフマッチします。
    std::vector<Result> GetResult() NN_NOEXCEPT
    {
        return std::move(m_Result);
    }

    // 進捗の取得
    std::shared_ptr<Progress> GetProgress() const NN_NOEXCEPT
    {
        return m_pProgress;
    }

private:
    class RegionSet;
    class RegionRange;
    class HashQueue;
    class HashQueueArray;
    class Candidate;
    class CandidateHolder;
    class CompareRange;

private:
    // 比較しない領域を追加します。
    nn::Result AddExcludeRange(
        std::vector<ExcludeRange>* pRanges, const ExcludeRange& range) NN_NOEXCEPT;

    // リージョンハッシュを生成します。
    nn::Result MakeRegionHash(char* workBuffer) NN_NOEXCEPT;

    // ヒントを元に新旧データを比較します。
    nn::Result CompareDataWithHint(char* workBuffer) NN_NOEXCEPT;

    // 新旧データを比較します。
    nn::Result CompareData(char* workBuffer, size_t shiftSize, Progress::Phase phase) NN_NOEXCEPT;

    // ハッシュとデータを同時に比較します。
    nn::Result CompareTogether(
                   std::vector<Candidate>* pCandidates,
                   int64_t dataOffset,
                   const char* textBuffer,
                   char* workBuffer,
                   int64_t mismatchSize,
                   int64_t compareSize
               ) NN_NOEXCEPT;

    // 一致する領域の前方を比較します。
    nn::Result ComparePreviousData(
                   std::vector<Candidate>* pCandidates,
                   int64_t dataOffset,
                   char* workBuffer,
                   int64_t mismatchSize
               ) NN_NOEXCEPT;

    // 一致する領域の後方を比較します。
    nn::Result CompareNextData(
                   std::vector<Candidate>* pCandidates,
                   int64_t dataOffset,
                   const char* textBuffer,
                   char* workBuffer,
                   int64_t compareSize
               ) NN_NOEXCEPT;

    // ハッシュ比較後にデータを比較します。
    nn::Result CompareSeparate(
                   std::vector<Candidate>* pCandidates,
                   int64_t dataOffset,
                   char* workBuffer,
                   int64_t mismatchSize,
                   int64_t compareSize
               ) NN_NOEXCEPT;

    // 一致した領域より後方部分のリージョンハッシュを比較します。
    nn::Result CompareNextHash(
                   std::vector<Candidate>* pCandidates,
                   int64_t dataOffset,
                   char* workBuffer,
                   int64_t compareSize
               ) NN_NOEXCEPT;

    // リージョンハッシュが一致した部分の厳密比較をします。
    nn::Result CompareDataStrict(
                   std::vector<Candidate>* pCandidates,
                   int64_t dataOffset,
                   char* workBuffer,
                   int64_t mismatchSize,
                   int64_t compareSize
               ) NN_NOEXCEPT;

    // 比較結果を生成します。
    void MakeResult(int64_t storageSize) NN_NOEXCEPT;

    // 比較結果から新しいストレージの比較しない領域を排除します。
    void ExcludeNewStorage() NN_NOEXCEPT;

    // 比較結果から古いストレージの比較しない領域を排除します。
    void ExcludeOldStorage() NN_NOEXCEPT;

private:
    const size_t m_BlockSize;
    const size_t m_RegionSize;
    const int64_t m_MatchSize;
    int64_t m_WindowSize;
    fs::SubStorage m_OldStorage;
    int64_t m_OldStorageSize;
    fs::SubStorage m_NewStorage;
    int64_t m_NewStorageSize;
    RegionBuffer m_RegionBuffer;
    RegionArray m_Regions;
    std::unique_ptr<RegionSet> m_pRegionSet;
    std::vector<Hint> m_Hints;
    std::vector<ExcludeRange> m_ExcludeRangeOld;
    std::vector<ExcludeRange> m_ExcludeRangeNew;
    std::vector<Result> m_Result;
    size_t m_ResultCountCache;
    std::shared_ptr<Progress> m_pProgress;

    friend class BinaryMatchTest;
};

}}}
