﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/fssystem/fs_AesCtrCounterExtendedStorage.h>
#include <nn/fssystem/utilTool/fs_RelocatedIndirectStorageBuilder.h>

namespace nn { namespace fssystem { namespace utilTool {

/**
 * @brief   AesCtrCounterExtendedStorage で扱うデータを構築するクラスです。
 */
class AesCtrCounterExtendedStorageBuilder
{
    NN_DISALLOW_COPY(AesCtrCounterExtendedStorageBuilder);

public:
    typedef RelocatedIndirectStorageBuilder::PatchInfo PatchInfo;

    /**
     * @brief   テーブルの各種情報です。
     */
    class TableInfo
    {
    public:
        TableInfo() NN_NOEXCEPT
            : m_EntryCount(0)
            , m_NodeStorage()
            , m_EntryStorage()
        {
        }

        TableInfo(
            int entryCount,
            fs::SubStorage nodeStorage,
            fs::SubStorage entryStorage
        ) NN_NOEXCEPT
            : m_EntryCount(entryCount)
            , m_NodeStorage(nodeStorage)
            , m_EntryStorage(entryStorage)
        {
        }

        template< typename T >
        Result Initialize(const void* pHeader, size_t headerSize, T pStorage) NN_NOEXCEPT;

        template< typename T >
        Result Initialize(const NcaPatchInfo& patchInfo, T pStorage) NN_NOEXCEPT;

        int GetEntryCount() const NN_NOEXCEPT
        {
            return m_EntryCount;
        }

        fs::SubStorage& GetNodeStorage() NN_NOEXCEPT
        {
            return m_NodeStorage;
        }

        const fs::SubStorage& GetNodeStorage() const NN_NOEXCEPT
        {
            return m_NodeStorage;
        }

        fs::SubStorage& GetEntryStorage() NN_NOEXCEPT
        {
            return m_EntryStorage;
        }

        const fs::SubStorage& GetEntryStorage() const NN_NOEXCEPT
        {
            return m_EntryStorage;
        }

    private:
        int m_EntryCount;
        fs::SubStorage m_NodeStorage;
        fs::SubStorage m_EntryStorage;
    };

    typedef MemoryResource IAllocator;

public:
    /**
     * @brief   コンストラクタです。
     */
    AesCtrCounterExtendedStorageBuilder() NN_NOEXCEPT;

    /**
     * @brief   デストラクタです。
     */
    ~AesCtrCounterExtendedStorageBuilder() NN_NOEXCEPT;

    /**
     * @brief   初期化をします。
     *
     * @param[in]   dataSize    データのサイズ
     * @param[in]   generation  全ての領域に設定する世代
     *
     * @return  実行結果を返します。
     *
     * @pre
     *      - IsInitialized() == false
     *      - IsBuilt() == false
     *      - 0 < dataSize
     *
     * 古いストレージがない状態として、テーブルの作成も同時に行います。
     */
    void Initialize(int64_t dataSize, uint32_t generation) NN_NOEXCEPT;

    /**
     * @brief   初期化をします。
     *
     * @param[in]   oldTableInfo        古いストレージのテーブル関連の情報
     * @param[in]   oldStorageOffset    古いストレージまでのオフセット
     * @param[in]   oldDataStorageSize  古いデータを指すストレージのサイズ
     * @param[in]   newStorageOffset    新しいストレージまでのオフセット
     * @param[in]   newDataStorageSize  新しいデータを指すストレージのサイズ
     *
     * @pre
     *      - IsInitialized() == false
     *      - IsBuilt() == false
     *      - 0 <= oldStorageOffset
     *      - 0 <= oldDataStorageSize
     *      - 0 <= newStorageOffset
     *      - 0 <= newDataStorageSize
     */
    void Initialize(
             const TableInfo& oldTableInfo,
             int64_t oldStorageOffset,
             int64_t oldDataStorageSize,
             int64_t newStorageOffset,
             int64_t newDataStorageSize
         ) NN_NOEXCEPT;

    /**
     * @brief   終了処理をします。
     */
    void Finalize() NN_NOEXCEPT;

    /**
     * @brief   初期化済みかどうかを取得します。
     *
     * @return  初期化済みかどうかを返します。
     */
    bool IsInitialized() const NN_NOEXCEPT
    {
        return m_IsInitialized;
    }

    /**
     * @brief   テーブル作成済みかどうかを取得します。
     *
     * @return  テーブル作成済みかどうかを返します。
     */
    bool IsBuilt() const NN_NOEXCEPT
    {
        return m_IsBuilt;
    }

    /**
     * @brief   RelocationTable を解析してテーブルを作成します。
     *
     * @param[in]   pAllocator              アロケータのポインタ
     * @param[in]   invertedRelocationTable 逆リロケーションテーブル
     * @param[in]   indirectTableOffset     IndirectStorage テーブルのオフセット
     * @param[in]   indirectTableSize       IndirectStorage テーブルのサイズ
     * @param[in]   generation              更新された領域に設定する世代
     *
     * @return  実行結果を返します。
     *
     * @pre
     *      - IsInitialized() != false
     *      - IsBuilt() == false
     *      - pAllocator != nullptr
     *      - 0 <= indirectTableOffset
     *      - 0 <= indirectTableSize
     */
    Result Build(
               IAllocator* pAllocator,
               const RelocationTable& invertedRelocationTable,
               int64_t indirectTableOffset,
               int64_t indirectTableSize,
               uint32_t generation
           ) NN_NOEXCEPT;

    /**
     * @brief   テーブルを書き出します。
     *
     * @param[in]   pAllocator      アロケータのポインタ
     * @param[in]   headerStorage   テーブルヘッダのストレージ
     * @param[in]   nodeStorage     テーブルノードのストレージ
     * @param[in]   entryStorage    テーブルエントリのストレージ
     *
     * @return  実行結果を返します。
     *
     * @pre
     *      - IsInitialized() != false
     *      - IsBuilt() != false
     *      - pAllocator != nullptr
     */
    Result WriteTable(
               IAllocator* pAllocator,
               fs::SubStorage headerStorage,
               fs::SubStorage nodeStorage,
               fs::SubStorage entryStorage
           ) const NN_NOEXCEPT;

    /**
     * @brief   テーブルでヘッダ部分のストレージのサイズを取得します。
     *
     * @return  ヘッダ部分のストレージサイズを返します。
     */
    int64_t QueryTableHeaderStorageSize() const NN_NOEXCEPT
    {
        return AesCtrCounterExtendedStorage::QueryHeaderStorageSize();
    }

    /**
     * @brief   テーブルでノード部分のストレージのサイズを取得します。
     *
     * @return  ノード部分のストレージサイズを返します。
     *
     * @pre
     *      - IsBuilt() != false
     */
    int64_t QueryTableNodeStorageSize() const NN_NOEXCEPT
    {
        NN_SDK_REQUIRES(IsBuilt());

        return AesCtrCounterExtendedStorage::QueryNodeStorageSize(GetEntryCount());
    }

    /**
     * @brief   テーブルでエントリ部分のストレージサイズを取得します。
     *
     * @return  エントリ部分のストレージサイズを返します。
     *
     * @pre
     *      - IsBuilt() != false
     */
    int64_t QueryTableEntryStorageSize() const NN_NOEXCEPT
    {
        NN_SDK_REQUIRES(IsBuilt());

        return AesCtrCounterExtendedStorage::QueryEntryStorageSize(GetEntryCount());
    }

    /**
     * @brief   テーブルのストレージサイズを取得します。
     *
     * @return  テーブルのストレージサイズを返します。
     *
     * @pre
     *      - IsBuilt() != false
     */
    int64_t QueryTableStorageSize() const NN_NOEXCEPT
    {
        return QueryTableHeaderStorageSize() +
               QueryTableNodeStorageSize() +
               QueryTableEntryStorageSize();
    }

    /**
     * @brief   AesCtrCounterExtendedStorageBuilder のビルド情報を出力します。
     *
     * @param[in]   pAllocator              アロケータのポインタ
     * @param[in]   pTableStorage           テーブル情報を書き出すストレージのポインタ
     * @param[in]   pNodeStorage            ノード情報を書き出すストレージのポインタ
     * @param[in]   pEntryStorage           エントリ情報を書き出すストレージのポインタ
     * @param[in]   startOffset             IndirectStorage の開始オフセット
     * @param[in]   invertedRelocationTable 逆リロケーションテーブル
     *
     * @pre
     *      - IsBuilt() != false
     *      - pAllocator != nullptr
     *      - pTableStorage != nullptr
     *      - pNodeStorage != nullptr
     *      - pEntryStorage != nullptr
     *      - 0 <= startOffset
     */
    Result OutputBuildLog(
               IAllocator* pAllocator,
               fs::IStorage* pTableFileStorage,
               fs::IStorage* pNodeFileStorage,
               fs::IStorage* pEntryFileStorage,
               int64_t startOffset,
               const RelocationTable& invertedRelocationTable
           ) const NN_NOEXCEPT;

private:
    // AesCtrCounterExtendedStorage のエントリ数を取得します。
    int GetEntryCount() const NN_NOEXCEPT
    {
        NN_SDK_ASSERT(util::IsIntValueRepresentable<int>(m_Entries.size()));
        return static_cast<int>(m_Entries.size());
    }

    // データ領域のテーブルを作成します。
    Result BuildImpl(
               IAllocator* pAllocator,
               const RelocationTable& relocationTable,
               int64_t tableEndOffset,
               uint32_t generation
           ) NN_NOEXCEPT;

private:
    bool m_IsInitialized;
    bool m_IsBuilt;
    TableInfo m_OldTableInfo;
    int64_t m_OldStorageOffset;
    int64_t m_OldDataStorageSize;
    int64_t m_NewStorageOffset;
    int64_t m_NewDataStorageSize;
    std::vector<AesCtrCounterExtendedStorage::Entry> m_Entries;
    int64_t m_EndOffset;
};

template< typename T >
Result AesCtrCounterExtendedStorageBuilder::TableInfo::Initialize(const void* pHeader, size_t headerSize, T pStorage) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pHeader);
    NN_SDK_REQUIRES_NOT_NULL(pStorage);

    NN_RESULT_THROW_UNLESS(sizeof(BucketTree::Header) <= headerSize, fs::ResultInvalidArgument());

    BucketTree::Header header;
    std::memcpy(&header, pHeader, sizeof(header));
    NN_RESULT_DO(header.Verify());

    const auto nodeSize = AesCtrCounterExtendedStorage::QueryNodeStorageSize(header.entryCount);
    const auto entrySize = AesCtrCounterExtendedStorage::QueryEntryStorageSize(header.entryCount);

    const auto nodeOffset = 0;
    const auto entryOffset = nodeOffset + nodeSize;

    m_EntryCount = header.entryCount;
    m_NodeStorage = fs::SubStorage(pStorage, nodeOffset, nodeSize);
    m_EntryStorage = fs::SubStorage(pStorage, entryOffset, entrySize);

    NN_RESULT_SUCCESS;
}

template< typename T >
Result AesCtrCounterExtendedStorageBuilder::TableInfo::Initialize(const NcaPatchInfo& patchInfo, T pStorage) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pStorage);

    NN_RESULT_THROW_UNLESS(patchInfo.HasAesCtrExTable(), fs::ResultInvalidArgument());

    BucketTree::Header header;
    std::memcpy(&header, patchInfo.aesCtrExHeader, sizeof(header));
    NN_RESULT_DO(header.Verify());

    const auto nodeSize = AesCtrCounterExtendedStorage::QueryNodeStorageSize(header.entryCount);
    const auto entrySize = AesCtrCounterExtendedStorage::QueryEntryStorageSize(header.entryCount);
    NN_RESULT_THROW_UNLESS(nodeSize + entrySize <= patchInfo.aesCtrExSize, fs::ResultInvalidAesCtrCounterExtendedTableSize());

    const auto nodeOffset = 0;
    const auto entryOffset = nodeOffset + nodeSize;

    m_EntryCount = header.entryCount;
    m_NodeStorage = fs::SubStorage(pStorage, nodeOffset, nodeSize);
    m_EntryStorage = fs::SubStorage(pStorage, entryOffset, entrySize);

    NN_RESULT_SUCCESS;
}

}}}
