﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/os.h>
#include <nn/fs/fs_IStorage.h>
#include <nn/fs/fs_SubStorage.h>

class RemapStoragePrivateTest;

namespace nn { namespace fssystem { namespace save {

/**
* @brief        リサイズを可能にするため論物変換を行うストレージ
*
* @details      このストレージには論理オフセットを用いてアクセスし、
*               内部でそれを物理オフセットに変換して下位ストレージにアクセスします。
*               リサイズが必要になるデータをこのストレージの上に置きます。
*/
class RemapStorage : public fs::IStorage
{
public:
    //! 管理領域
    struct ControlArea
    {
        uint32_t magic;          //!< マジックコード
        uint32_t version;        //!< 管理領域のバージョン
        int32_t countMapEntries; //!< 使用済みのマッピングエントリ数。登録回数より多い可能性があります
        int32_t countMaps;       //!< マッピングの登録回数。マッピングエントリ数より少ない可能性があります
        int32_t countHigherBits; //!< 論理インデックスを新規登録するとき上位何ビットをカウントアップするか
        uint8_t _reserved[64 - 4 * 5];
    };

    NN_STATIC_ASSERT(sizeof(ControlArea) == 64);
    NN_STATIC_ASSERT(std::is_pod<ControlArea>::value);

public:
    //! マッピングの登録時に指定できるアライメントのうち小さいほうです
    static const int64_t AlignmentSmall   = 512;

    //! マッピングの登録時に指定できるアライメントのうち大きいほうです
    static const int64_t AlignmentLarge   = 16 * 1024;

    //! 小さいほうのアライメントでの登録時に指定するサイズはこの定数の倍数である必要があります
    static const int64_t SizeBlockSmall   = AlignmentSmall;

    //! 大きいほうのアライメントでの登録時に指定するサイズはこの定数の倍数である必要があります
    static const int64_t SizeBlockLarge   = AlignmentLarge;

    //! 登録できるストレージの最大数です
    static const int     CountMaxStorages = 4;

    NN_STATIC_ASSERT(AlignmentSmall < AlignmentLarge);
    NN_STATIC_ASSERT((AlignmentSmall & (AlignmentSmall - 1)) == 0);
    NN_STATIC_ASSERT((AlignmentLarge & (AlignmentLarge - 1)) == 0);

    NN_STATIC_ASSERT(0 < CountMaxStorages);

public:
    /**
    * @brief        メタデータサイズを取得します。
    *
    * @return       メタデータサイズを返します。
    *
    * @details      メタデータサイズを取得します。
    */
    static int64_t QueryMetaSize() NN_NOEXCEPT;

    /**
    * @brief        マッピングを指定の回数更新できるエントリテーブルを格納するために必要なサイズを取得します。
    *
    * @param[in]    countMapUpdate  マッピングを更新できる回数の最大
    *
    * @return       エントリテーブルを格納するために必要なサイズを返します。
    *
    * @pre
    *               - 0 < countMapUpdate
    *
    * @details      マッピングを指定の回数更新できるエントリテーブルを格納するために必要なサイズを取得します。
    */
    static int64_t QueryEntryTableSize(int countMapUpdate) NN_NOEXCEPT;

    /**
    * @brief        指定のサイズのストレージにエントリを記録した場合に
    *               マッピングの登録および拡張を必ず行うことができる回数を取得します。
    *
    * @param[in]    sizeEntry   エントリを記録するストレージのサイズ
    *
    * @return       マッピングの登録および拡張を必ず行うことができる回数を返します。
    *
    * @pre
    *               - 0 < sizeEntry
    *
    * @details      本関数は登録と拡張の合計数を取得します。
    *               登録のみの最大回数はフォーマット時のパラメータによって決定され、
    *               本関数の返り値よりも少なくなる可能性があります。
    *
    *               マッピングの登録や拡張時に渡す引数により、
    *               本関数の返り値を超える回数の登録や拡張を行える可能性があります。
    */
    static int64_t GetMapUpdateCountLowerBound(int64_t sizeEntry) NN_NOEXCEPT;

    /**
    * @brief        指定のサイズのストレージにエントリを記録した場合に
    *               マッピングの登録および拡張を行うことができる回数の最大を取得します。
    *
    * @param[in]    sizeEntry   エントリを記録するストレージのサイズ
    *
    * @return       マッピングの登録および拡張を行うことができる回数の最大を返します。
    *
    * @pre
    *               - 0 < sizeEntry
    *
    * @details      本関数は登録と拡張の合計数を取得します。
    *               登録のみの最大回数はフォーマット時のパラメータによって決定され、
    *               本関数の返り値よりも少なくなる可能性があります。
    *
    *               マッピングの登録や拡張時に渡す引数により、
    *               本関数の返り値未満の回数しか登録や拡張を行えない可能性があります。
    */
    static int64_t GetMapUpdateCountUpperBound(int64_t sizeEntry) NN_NOEXCEPT;

    /**
    * @brief        メタデータ領域をフォーマットします。
    *
    * @param[in]    storageMeta     メタデータファイルを配置するレイヤー
    * @param[in]    countMaxMaps    マッピングを新規登録できる回数の最大
    *
    * @return       関数の処理結果を返します。
    *
    * @pre
    *               - QueryMetaSize() <= storageMeta.GetSize()
    *               - 0 < countMaxMaps
    *
    * @details      メタデータ領域をフォーマットします。
    */
    static Result Format(
                      fs::SubStorage storageMeta,
                      int countMaxMaps
                  ) NN_NOEXCEPT;

    /**
    * @brief        コンストラクタです。
    *
    * @details      コンストラクタです。
    */
    RemapStorage() NN_NOEXCEPT;

    /**
    * @brief        デストラクタです。
    *
    * @details      デストラクタです。
    */
    virtual ~RemapStorage() NN_NOEXCEPT NN_OVERRIDE;

    /**
    * @brief        再マップレイヤーとしてマウントします。
    *
    * @param[in]    storageMeta     メタデータファイルを配置するレイヤー
    * @param[in]    storageEntry    エントリデータを配置するレイヤー
    *
    * @return       関数の処理結果を返します。
    *
    * @pre
    *               - 未初期化である
    *               - QueryMetaSize() <= storageMeta.GetSize()
    *               - 0 < storageEntry.GetSize()
    *
    * @details
    */
    Result Initialize(
               fs::SubStorage storageMeta,
               fs::SubStorage storageEntry
           ) NN_NOEXCEPT;

    /**
    * @brief        アンマウントします。
    *
    * @details
    */
    void Finalize() NN_NOEXCEPT;

    /**
    * @brief        マッピングを登録します。
    *
    * @param[out]   outOffsetVirtual    確定した論理オフセット
    * @param[in]    size                登録するサイズ
    * @param[in]    alignment           登録するマッピングのアライメント
    * @param[in]    storageType         登録するストレージ種別
    *
    * @return       関数の処理結果を返します。
    *
    * @pre
    *               - outOffsetVirtual != nullptr
    *               - alignment == AlignmentSmall || alignment == AlignmentLarge
    *
    * @details
    */
    Result RegisterMap(
               int64_t* outOffsetVirtual,
               int64_t size,
               int64_t alignment,
               int storageType
           ) NN_NOEXCEPT;

    /**
    * @brief        マッピングを拡張します。
    *
    * @param[in]    virtualOffset   拡張するマッピングの論理オフセット
    * @param[in]    size            拡張後のサイズ
    *
    * @return       関数の処理結果を返します。
    *
    * @details      マッピングを拡張します。
    */
    Result ExpandMap(int64_t virtualOffset, int64_t size) NN_NOEXCEPT;

    /**
    * @brief        ストレージを登録します。
    *
    * @param[in]    storage     登録するストレージ
    * @param[in]    storageType 登録するストレージ種別
    *
    * @return       関数の処理結果を返します。
    *
    * @pre
    *               - 0 <= storageType < CountMaxStorages
    *               - 指定の storageType で RegisterStorage に成功していない
    *
    * @details
    */
    Result RegisterStorage(fs::SubStorage storage, int storageType) NN_NOEXCEPT;

    /**
    * @brief        登録できるマッピングサイズの最大を取得します。
    *
    * @return       登録できるマッピングサイズの最大を返します。
    *
    * @details      登録できるマッピングサイズの最大を取得します。
    */
    int64_t GetMapSizeMax() const NN_NOEXCEPT;

    /**
    * @brief        ストレージの内容をバッファに読み込みます。
    *
    * @param[in]    offset  読み込み開始位置
    * @param[out]   buffer  読み込んだ内容をコピーするバッファ
    * @param[in]    size    読み込むデータサイズ
    *
    * @return       関数の処理結果を返します。
    *
    * @details      ストレージの内容をバッファに読み込みます。
    */
    virtual Result Read(int64_t offset, void* buffer, size_t size) NN_NOEXCEPT NN_OVERRIDE;

    /**
    * @brief        バッファの内容をストレージに書き込みます。
    *
    * @param[in]    offset  書き込み開始位置
    * @param[in]    buffer  書き込むデータ
    * @param[in]    size    書き込むデータサイズ
    *
    * @return       関数の処理結果を返します。
    *
    * @details      バッファの内容をストレージに書き込みます。
    */
    virtual Result Write(int64_t offset, const void* buffer, size_t size) NN_NOEXCEPT NN_OVERRIDE;

    /**
    * @brief        フラッシュします。
    *
    * @return       関数の処理結果を返します。
    *
    * @details      フラッシュします。
    */
    virtual Result Flush() NN_NOEXCEPT NN_OVERRIDE;

    /**
    * @brief        リマップストレージのサイズは取得できません。
    *
    * @param[out]   outValue    取得したサイズ格納先
    *
    * @return       関数の処理結果を返します。
    *
    * @details      リマップストレージのサイズは取得できません。
    */
    virtual Result GetSize(int64_t* outValue) NN_NOEXCEPT NN_OVERRIDE;

    /**
    * @brief        範囲指定処理を行います。
    *
    * @param[out]   outBuffer       範囲指定処理の結果を格納するバッファ
    * @param[in]    outBufferSize   範囲指定処理の結果を格納するバッファのサイズ
    * @param[in]    operationId     範囲指定処理の種類
    * @param[in]    offset          範囲指定処理開始位置
    * @param[in]    size            範囲指定処理を行うデータサイズ
    * @param[in]    inBuffer        範囲指定処理に渡すバッファ
    * @param[in]    inBufferSize    範囲指定処理に渡すバッファのサイズ
    *
    * @return       関数の処理結果を返します。
    */
    virtual Result OperateRange(
                       void* outBuffer,
                       size_t outBufferSize,
                       fs::OperationId operationId,
                       int64_t offset,
                       int64_t size,
                       const void* inBuffer,
                       size_t inBufferSize
                   ) NN_NOEXCEPT NN_OVERRIDE;

    /**
    * @brief       エントリテーブルのサイズを取得します。
    *
    * @return      エントリテーブルのサイズを返します。
    */
    int64_t GetEntryTableSize() const NN_NOEXCEPT
    {
        int64_t sizeEntry = 0;
        const auto result = m_StorageEntry.GetSize(&sizeEntry);
        NN_SDK_ASSERT(result.IsSuccess()); // SubStorage::GetSize() は失敗しないはず
        NN_UNUSED(result);
        return sizeEntry;
    }

    /**
    * @brief       登録と拡張ができることが保証される回数を取得します。
    *
    * @return      登録と拡張ができることが保証される回数を返します。
    *
    * @details     既に登録または拡張が行われている場合、
    *              この関数の戻り値の回数だけ登録や拡張ができない可能性があります。
    */
    int64_t GetMapUpdateCount() const NN_NOEXCEPT
    {
        return GetMapUpdateCountLowerBound(GetEntryTableSize());
    }

private:
    //! ストレージを登録するエントリ
    class StorageEntry
    {
    public:
        /**
        * @brief        コンストラクタです。
        *
        * @details      コンストラクタです。
        */
        StorageEntry() NN_NOEXCEPT
            : m_IsRegistered(false)
        {
        }

        /**
        * @brief        登録されているストレージを取得します。
        *
        * @return       登録されているストレージ
        *
        * @details      登録されているストレージを取得します。
        */
        fs::SubStorage& GetStorage() NN_NOEXCEPT
        {
            return m_Storage;
        }

        /**
        * @brief        ストレージを登録します。
        *
        * @param[in]    storage     登録するストレージ
        *
        * @pre
        *               ストレージが登録されていない
        *
        * @details      ストレージを登録します。
        */
        void SetStorage(const fs::SubStorage& storage) NN_NOEXCEPT
        {
            NN_SDK_REQUIRES(!m_IsRegistered);
            m_Storage = storage;
            m_IsRegistered = true;
        }

        /**
        * @brief        初期化します。
        *
        * @param[in]    smallAlignedTail    小さいアライメントで配置されるアドレスの末尾
        * @param[in]    largeAlignedTail    大きいアライメントで配置されるアドレスの末尾
        *
        * @details      初期化します。
        */
        void Initialize(int64_t smallAlignedTail, int64_t largeAlignedTail) NN_NOEXCEPT;

        /**
        * @brief        ストレージが登録されているかどうかを返します。
        *
        * @return       処理結果を返します。
        * @retval       true            ストレージ登録済み
        * @retval       false           ストレージ未登録
        *
        * @details      ストレージが登録されているかどうかを返します。
        */
        bool IsRegistered() const NN_NOEXCEPT
        {
            return m_IsRegistered;
        }

        /**
        * @brief        次の小さいアライメントでの登録時に割り当てる物理オフセットを返します。
        *
        * @return       次の小さいアライメントでの登録時に割り当てる物理オフセット
        *
        * @details      次の小さいアライメントでの登録時に割り当てる物理オフセットを返します。
        */
        int64_t GetOffsetPhysicalNextSmall() NN_NOEXCEPT
        {
            return m_OffsetPhysicalNextSmall;
        }

        /**
        * @brief        次の小さいアライメントでの登録時に割り当てる物理オフセットを登録します。
        *
        * @param[in]    offsetPhysicalNextSmall     次の小さいアライメントでの登録時に割り当てる物理オフセット
        *
        * @details      次の小さいアライメントでの登録時に割り当てる物理オフセットを登録します。
        */
        void SetOffsetPhysicalNextSmall(int64_t offsetPhysicalNextSmall) NN_NOEXCEPT
        {
            m_OffsetPhysicalNextSmall = offsetPhysicalNextSmall;
        }

        /**
        * @brief        次の大きいアライメントでの登録時に割り当てる物理オフセットを返します。
        *
        * @return       次の大きいアライメントでの登録時に割り当てる物理オフセット
        *
        * @details      次の大きいアライメントでの登録時に割り当てる物理オフセットを返します。
        */
        int64_t GetOffsetPhysicalNextLarge() NN_NOEXCEPT
        {
            return m_OffsetPhysicalNextLarge;
        }

        /**
        * @brief        次の大きいアライメントでの登録時に割り当てる物理オフセットを登録します。
        *
        * @param[in]    offsetPhysicalNextSmall     次の大きいアライメントでの登録時に割り当てる物理オフセット
        *
        * @details      次の大きいアライメントでの登録時に割り当てる物理オフセットを登録します。
        */
        void SetOffsetPhysicalNextLarge(int64_t offsetPhysicalNextLarge) NN_NOEXCEPT
        {
            m_OffsetPhysicalNextLarge = offsetPhysicalNextLarge;
        }

    private:
        int64_t m_OffsetPhysicalNextSmall;  //!< 次の小さいアライメントでの登録時に割り当てる物理オフセット
        int64_t m_OffsetPhysicalNextLarge;  //!< 次の大きいアライメントでの登録時に割り当てる物理オフセット
        fs::SubStorage m_Storage;           //!< 登録されたストレージ
        bool m_IsRegistered;                //!< ストレージが登録されているかどうか
    };

    //! 論理オフセットから物理オフセットへのマッピングを登録するエントリ
    struct MapEntry
    {
        int64_t offsetVirtual;  //!< 論理オフセット
        int64_t offsetPhysical; //!< 物理オフセット
        int64_t size;           //!< サイズ
        int32_t alignment;      //!< アライメント
        int32_t storageType;    //!< ストレージ種別
    };
    NN_STATIC_ASSERT(std::is_pod<MapEntry>::value);

    //! 同じ論理オフセットに関する一連のマッピングエントリにアクセスするためのイテレータ
    class MapIterator
    {
    public:
        /**
        * @brief        指定の論理オフセットに関するマッピングエントリに
        *               アクセスするイテレータを作成します。
        *
        * @param[in]    pRemapStorage   ストレージ本体
        * @param[in]    offsetVirtual   アクセスする論理オフセット
        *
        * @pre
        *               - pRemapStorage != nullptr
        *
        * @details      指定の論理オフセットに関するマッピングエントリに
        *               アクセスするイテレータを作成します。
        */
        MapIterator(RemapStorage* pRemapStorage, int64_t offsetVirtual) NN_NOEXCEPT;

        /**
        * @brief        次のマッピングエントリを読み込みます。
        *
        * @param[in]    size    読み込むサイズの残り
        *
        * @return       処理結果を返します。
        * @retval       true            次のマッピングエントリの読み込みに成功しました。
        * @retval       false           次のマッピングエントリの読み込みに失敗したか、
        *                               またはイテレーションが終了しました。
        *
        * @details      GetMapEntry() を呼び出す前に少なくとも一度本関数を呼び出し、
        *               true が返っている必要があります。
        *               処理に失敗した場合 GetlastResult() でエラー内容を取得できます。
        */
        bool MoveNext(int64_t size) NN_NOEXCEPT;

        /**
        * @brief        現在のマッピングエントリを取得します。
        *
        * @return       現在のマッピングエントリを返します。
        *
        * @pre
        *               - MoveNext() が少なくとも一度 true を返している
        *
        * @details      現在のマッピングエントリを取得します。
        */
        const MapEntry& GetMapEntry() const NN_NOEXCEPT;

        /**
        * @brief        アクセスしているマッピングエントリに紐付けられたストレージを取得します。
        *
        * @return       アクセスしているマッピングエントリに紐付けられたストレージを返します。
        * @retval       nullptr ストレージ種別またはストレージが登録されていません。
        *
        * @pre
        *               - MoveNext() が少なくとも一度 true を返している
        *
        * @details      アクセスしているマッピングエントリに紐付けられたストレージを取得します。
        */
        IStorage* GetStorage() const NN_NOEXCEPT;

        /**
        * @brief        最後に実行された処理の結果を取得します。
        *
        * @return       最後に実行された処理の結果を返します。
        *
        * @details      最後に実行された処理の結果を取得します。
        */
        Result GetLastResult() const NN_NOEXCEPT;

    private:
        RemapStorage* m_pRemapStorage; //!< ストレージ
        MapEntry      m_MapEntry;      //!< 現在のマッピングエントリ
        int           m_IndexMapEntry; //!< 現在のマッピングエントリのインデックス
        int           m_StorageType;   //!< 関連付けられたストレージエントリのインデックス
        Result        m_Result;        //!< 処理結果

    private:
        friend class ::RemapStoragePrivateTest;
    };

    //! マップエントリのキャッシュ
    class MapEntryCache
    {
    public:
        static const int CountNodes = 20; //!< @brief ノードの数

    public:
        /**
        * @brief        空のマッピングエントリを作成します。
        */
        MapEntryCache() NN_NOEXCEPT;

        /**
        * @brief        マッピングエントリをキャッシュします。
        *
        * @param[in]    mapEntry        キャッシュするマッピングエントリ
        * @param[in]    indexMapEntry   キャッシュするマッピングエントリのインデックス
        *
        * @details      同じエントリはキャッシュされていないものとし、重複があるかどうかは検査しません。
        */
        void SetMapEntry(const MapEntry& mapEntry, int indexMapEntry) NN_NOEXCEPT;

        /**
        * @brief        指定の論理オフセットを含むキャッシュを取得します。
        *
        * @param[out]   outMapEntry         キャッシュされたエントリをコピーするバッファ
        * @param[out]   outIndexmapEntry    キャッシュされたインデックスをコピーするバッファ
        * @param[in]    offsetVirtual       取得するエントリの論理オフセット
        *
        * @return       見つかったかどうかを返します。
        * @retval       true                キャッシュされたエントリが見つかりました。
        * @retval       false               指定の論理オフセットを含むエントリはキャッシュされていませんでした。
        *
        * @pre
        *               - outMapEntry != nullptr
        *               - outIndexMapEntry != nullptr
        *
        * @details
        */
        bool GetMapEntry(
                 MapEntry* outMapEntry,
                 int* outIndexMapEntry,
                 int64_t offsetVirtual
             ) const NN_NOEXCEPT;

        /**
        * @brief        キャッシュを空にします。
        */
        void Clear() NN_NOEXCEPT;

    private:
        /**
        * @brief        マッピングエントリを保持するノード
        */
        struct Node
        {
            MapEntry mapEntry;      //!< マッピングエントリ
            int      indexMapEntry; //!< マッピングエントリのストレージにおけるインデックス
        };
        NN_STATIC_ASSERT(std::is_pod<Node>::value);

    private:
        static const int InvalidMapEntryIndex = -1; //!< 無効なノードを表すインデックス

    private:
        Node                  m_Nodes[CountNodes]; //!< ノード配列
        int                   m_NextNodeIndex;     //!< 次に使われるノード
        mutable nn::os::Mutex m_Mutex;             //!< 排他用ミューテックス

    private:
        friend class ::RemapStoragePrivateTest;
    };

private:
    /**
    * @brief        ストレージエントリを初期化します。
    *
    * @return       処理結果を返します。
    *
    * @details      ストレージエントリを初期化します。
    */
    Result InitializeStorageEntries() NN_NOEXCEPT;

    /**
    * @brief        指定のインデックスのマッピングエントリを読み込みます。
    *
    * @param[out]   outEntry    読み込んだエントリをコピーするバッファ
    * @param[in]    index       読み込むインデックス
    *
    * @return       処理結果を返します。
    *
    * @pre
    *               - outEntry != nullptr
    *               - 0 <= index < m_ControlArea.countMapEntries
    *
    * @details      指定のインデックスのマッピングエントリを読み込みます。
    */
    Result ReadMapEntry(MapEntry* outEntry, int index) const NN_NOEXCEPT;

    /**
    * @brief        指定のインデックスのマッピングエントリを書き込みます。
    *
    * @param[out]   pEntry  書き込むエントリ
    * @param[in]    index   書き込むインデックス
    *
    * @return       処理結果を返します。
    *
    * @pre
    *               - outEntry != nullptr
    *               - 0 <= index < GetMapEntryCountMax(m_ControlArea.countMaxMaps)
    *
    * @details      指定のインデックスのマッピングエントリを書き込みます。
    */
    Result WriteMapEntry(const MapEntry* pEntry, int index) NN_NOEXCEPT;

    /**
    * @brief        仮想オフセットを作成します。
    *
    * @param[in]    higherBits  上位ビット
    * @param[in]    lowerBits   下位ビット
    *
    * @return       仮想オフセットを返します。
    *
    * @pre
    *               - lowerBits の上位ビットが全てゼロ
    *
    * @details      仮想オフセットを作成します。
    */
    int64_t MakeVirtualOffset(int64_t higherBits, int64_t lowerBits) const NN_NOEXCEPT;

    /**
    * @brief        仮想オフセットの下位ビットのマスクを取得します。
    *
    * @return       マスクを返します。
    *
    * @details      仮想オフセットの下位ビットのマスクを取得します。
    */
    int64_t GetVirtualOffsetLowerBitsMask() const NN_NOEXCEPT;

    /**
    * @brief        仮想オフセットの上位ビットのマスクを取得します。
    *
    * @return       マスクを返します。
    *
    * @details      仮想オフセットの上位ビットのマスクを取得します。
    */
    int64_t GetVirtualOffsetHigherBitsMask() const NN_NOEXCEPT;

    /**
    * @brief        マッピングエントリの最大数を取得します。
    *
    * @return       マッピングエントリの最大数を返します。
    *
    * @details      マッピングエントリの最大数を取得します。
    */
    int64_t GetMapEntriesCountMax() const NN_NOEXCEPT;

    /**
    * @brief        指定の論理オフセットを含むエントリのキャッシュを管理するオブジェクトを取得します。
    *
    * @param[in]    offsetVirtual   論理オフセット
    *
    * @return       指定の論理オフセットを含むエントリのキャッシュを管理するオブジェクトを返します。
    *
    * @details      指定の論理オフセットを含むエントリのキャッシュを管理するオブジェクトを取得します。
    */
    MapEntryCache& GetCache(int64_t offsetVirtual) NN_NOEXCEPT;

    /**
    * @brief        指定の論理オフセットを含むエントリのキャッシュを管理するオブジェクトを取得します。
    *
    * @param[in]    offsetVirtual   論理オフセット
    *
    * @return       指定の論理オフセットを含むエントリのキャッシュを管理するオブジェクトを返します。
    *
    * @details      指定の論理オフセットを含むエントリのキャッシュを管理するオブジェクトを取得します。
    */
    const MapEntryCache& GetCache(int64_t offsetVirtual) const NN_NOEXCEPT;

    /**
    * @brief        全てのキャッシュを破棄します。
    *
    * @details      全てのキャッシュを破棄します。
    */
    void ClearAllCaches() NN_NOEXCEPT;

    /**
    * @brief        マッピングを新規登録します。
    *
    * @param[in]    offsetVirtual   登録する論理オフセット
    * @param[in]    sizeNew         登録後のサイズ
    * @param[in]    sizeOld         登録前のサイズ
    * @param[in]    alignment       登録するアライメント
    * @param[in]    storageType     登録するストレージ種別
    *
    * @return       処理結果を返します。
    *
    * @pre
    *               - sizeOld < sizeNew
    *               - alignment == AlignmentSmall || alignment == AlignmentLarge
    *               - 0 <= storageType < CountMaxStorages
    *
    * @details      マッピングを新規登録します。
    */
    Result RegisterMapCore(
               int64_t offsetVirtual,
               int64_t sizeNew,
               int64_t sizeOld,
               int64_t alignment,
               int storageType
           ) NN_NOEXCEPT;

    /**
    * @brief            マッピングエントリを新規作成します。
    *
    * @param[out]       outMapEntry     作成したエントリを格納する領域
    * @param[in,out]    pStorageEntry   ストレージエントリ
    * @param[in,out]    pControlArea    管理領域
    * @param[in]        offsetVirtual   作成するエントリの論理オフセット
    * @param[in]        size            作成するエントリのサイズ
    * @param[in]        alignment       作成するエントリのアライメント
    * @param[in]        storageType     作成するエントリのストレージ種別
    *
    * @return           処理結果を返します。
    *
    * @pre
    *                   - outMapEntry != nullptr
    *                   - pStorageEntry != nullptr
    *                   - pControlArea != nullptr
    *                   - alignment == AlignmentSmall || alignment == AlignmentLarge
    *                   - 0 <= storageType < CountMaxStorages
    *
    * @details          本関数はマッピングエントリの登録は行いませんが、
    *                   指定の管理領域とストレージエントリはマッピングが登録された後の状態に更新されます。
    */
    Result MakeMapEntry(
               MapEntry* outMapEntry,
               StorageEntry* pStorageEntry,
               ControlArea* pControlArea,
               int64_t offsetVirtual,
               int64_t size,
               int64_t alignment,
               int storageType
           ) const NN_NOEXCEPT;

    /**
    * @brief        マッピングエントリをテーブルに追加します。
    *
    * @param[in]    pMapEntries 追加するエントリ
    * @param[in]    count       追加するエントリの数
    *
    * @return       処理結果を返します。
    *
    * @pre
    *               - pMapEntries != nullptr
    *               - 1 <= count
    *               - pMapEntries[n - 1].offsetVirtual + pMapEntries[n - 1].size
    *                     == pMapEntries[n].offsetVirtual (n = 1, 2, ..., count - 1)
    *
    * @details      マッピングエントリをテーブルに追加します。
    */
    Result AddMapEntries(const MapEntry* pMapEntries, int count) NN_NOEXCEPT;

    /**
    * @brief        マッピングエントリのイテレーションを行います。
    *
    * @param[in]    offset      イテレーションを行う開始位置
    * @param[in]    size        データサイズ
    * @param[in]    func        エントリごとに行う処理
    *
    * @return       処理結果を返します。
    *
    * @details      マッピングエントリのイテレーションを行います。
    */
    template<typename TFunc>
    Result IterateMappingEntries(int64_t offset, int64_t size, TFunc func) NN_NOEXCEPT;

private:
    ControlArea m_ControlArea;                          //!< 管理領域

    fs::SubStorage m_StorageMeta;                       //!< メタデータ用ストレージ
    mutable fs::SubStorage m_StorageEntry;              //!< エントリデータ用ストレージ

    MapEntryCache m_MapEntryCache;                      //!< マッピングエントリキャッシュ

    StorageEntry m_StorageEntries[CountMaxStorages];    //!< ストレージエントリ

    bool m_IsInitialized;                               //!< 初期化済みかどうか
    bool m_IsControlAreaModified;                       //!< 管理領域が変更されたか
    char reserved[2];

private:
    friend class ::RemapStoragePrivateTest;
};

}}} // namespace nn::fssystem::save
