﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Common.h>

#include <nn/fssystem/save/fs_JournalStorage.h>
#include <nn/fssystem/save/fs_HierarchicalIntegrityVerificationStorage.h>

namespace nn { namespace fssystem { namespace save {

class IInternalStorageFileSystemVisitor;

/**
* @brief 完全性セーブデータストレージクラスです。
*/
class JournalIntegritySaveDataStorage : public nn::fs::IStorage
{
    NN_DISALLOW_COPY(JournalIntegritySaveDataStorage);

public:
    static const char InternalStorageFileNameJournal[];
    static const char InternalStorageFileNameIntegrity[];
    static const char InternalStorageFileNameIntegrityWithZeroFree[];

public:
    /**
    * @brief        デフォルトコンストラクタ。
    */
    JournalIntegritySaveDataStorage() NN_NOEXCEPT {}

    /**
    * @brief        デストラクタ。
    */
    virtual ~JournalIntegritySaveDataStorage() NN_NOEXCEPT NN_OVERRIDE {}

    /**
    * @brief        ストレージに必要なサイズを計算します。
    *
    * @param[out]   outSizeJournalTable                 ジャーナリング用テーブルに必要なサイズ
    * @param[out]   outSizeJournalBitmapUpdatedPhysical 更新済み物理ビットマップに必要なサイズ
    * @param[out]   outSizeJournalBitmapUpdatedVirtual  更新済み論理ビットマップに必要なサイズ
    * @param[out]   outSizeJournalBitmapUnassigned      未割当ビットマップに必要なサイズ
    * @param[out]   outIntegritySizeSet                 各階層ハッシュに必要なサイズ
    * @param[in]    integrityInputParam                 階層ハッシュのブロックサイズ
    * @param[in]    sizeJournalReservedArea             ジャーナリング用予約領域サイズ
    * @param[in]    sizeJournalBlock                    ジャーナリング領域のブロックサイズ
    * @param[in]    sizeData                            データ領域のサイズ
    *
    * @return       関数の処理結果を返します。
    */
    static Result QuerySize(
                      int64_t* outSizeJournalTable,
                      int64_t* outSizeJournalBitmapUpdatedPhysical,
                      int64_t* outSizeJournalBitmapUpdatedVirtual,
                      int64_t* outSizeJournalBitmapUnassigned,
                      HierarchicalIntegrityVerificationSizeSet* outIntegritySizeSet,
                      const HierarchicalIntegrityVerificationStorageControlArea::InputParam& integrityInputParam,
                      int64_t sizeJournalReservedArea,
                      int64_t sizeJournalBlock,
                      int64_t sizeData
                  ) NN_NOEXCEPT;

    /**
    * @brief        ストレージをフォーマットします。
    *
    * @param[in]    storageJournalControlArea           ジャーナリング管理領域を格納したストレージ
    * @param[in]    storageJournalTable                 ジャーナリング用テーブルを格納したストレージ
    * @param[in]    storageJournalBitmapUpdatedPhysical 更新済み物理ビットマップを格納したストレージ
    * @param[in]    storageJournalBitmapUpdatedVirtual  更新済み論理ビットマップを格納したストレージ
    * @param[in]    storageJournalBitmapUnassigned      未割当ビットマップを格納したストレージ
    * @param[in]    storageIntegrityMeta                階層ハッシュのメタデータを格納したストレージ
    * @param[in]    storageIntegrityMasterHash          マスターハッシュを格納したストレージ
    * @param[in]    sizeJournalReservedArea             ジャーナリング用予約領域サイズ
    * @param[in]    sizeJournalBlock                    ジャーナリング領域のブロックサイズ
    * @param[in]    integrityMeta                       階層ハッシュのブロックサイズ
    * @param[in]    sizeData                            データ領域のサイズ
    *
    * @return       関数の処理結果を返します。
    */
    static Result Format(
                      fs::SubStorage storageJournalControlArea,
                      fs::SubStorage storageJournalTable,
                      fs::SubStorage storageJournalBitmapUpdatedPhysical,
                      fs::SubStorage storageJournalBitmapUpdatedVirtual,
                      fs::SubStorage storageJournalBitmapUnassigned,
                      fs::SubStorage storageIntegrityMeta,
                      fs::SubStorage storageIntegrityMasterHash,
                      int64_t sizeJournalReservedArea,
                      int64_t sizeJournalBlock,
                      const HierarchicalIntegrityVerificationMetaInformation& integrityMeta,
                      int64_t sizeData
        ) NN_NOEXCEPT;

    /**
    * @brief        ストレージを拡張します。
    *
    * @param[in]    storageJournalControlArea           ジャーナリング管理領域を格納したストレージ
    * @param[in]    storageJournalTable                 ジャーナリング用テーブルを格納したストレージ
    * @param[in]    storageJournalBitmapUpdatedPhysical 更新済み物理ビットマップを格納したストレージ
    * @param[in]    storageJournalBitmapUpdatedVirtual  更新済み論理ビットマップを格納したストレージ
    * @param[in]    storageJournalBitmapUnassigned      未割当ビットマップを格納したストレージ
    * @param[in]    storageIntegrityMeta                階層ハッシュのメタデータを格納したストレージ
    * @param[in]    storageIntegrityMasterHash          マスターハッシュを格納したストレージ
    * @param[in]    sizeJournalReservedAreaNew          ジャーナリング用予約領域サイズ
    * @param[in]    integrityMeta                       階層ハッシュのパラメータ
    * @param[in]    sizeDataNew                         データ領域のサイズ
    *
    * @return       関数の処理結果を返します。
    */
    static Result Expand(
                      fs::SubStorage storageJournalControlArea,
                      fs::SubStorage storageJournalTable,
                      fs::SubStorage storageJournalBitmapUpdatedPhysical,
                      fs::SubStorage storageJournalBitmapUpdatedVirtual,
                      fs::SubStorage storageJournalBitmapUnassigned,
                      fs::SubStorage storageIntegrityMeta,
                      fs::SubStorage storageIntegrityMasterHash,
                      int64_t sizeJournalReservedAreaNew,
                      const HierarchicalIntegrityVerificationMetaInformation& integrityMeta,
                      int64_t sizeDataNew
                  ) NN_NOEXCEPT;

    /**
    * @brief        ストレージをマウントします。
    *
    * @param[in]    storageJournalControlArea           ジャーナリング管理領域を格納したストレージ
    * @param[in]    storageJournalTable                 ジャーナリング用テーブルを格納したストレージ
    * @param[in]    storageJournalBitmapUpdatedPhysical 更新済み物理ビットマップを格納したストレージ
    * @param[in]    storageJournalBitmapUpdatedVirtual  更新済み論理ビットマップを格納したストレージ
    * @param[in]    storageJournalBitmapUnassigned      未割当ビットマップを格納したストレージ
    * @param[in]    storageIntegrityMasterHash          マスターハッシュを格納したストレージ
    * @param[in]    storageIntegrityLayeredhashL1       L1 階層ハッシュを格納したストレージ
    * @param[in]    storageIntegrityLayeredhashL2       L2 階層ハッシュを格納したストレージ
    * @param[in]    storageIntegrityLayeredhashL3       L3 階層ハッシュを格納したストレージ
    * @param[in]    storageData                         実データを格納したストレージ
    * @param[in]    integrityInfo                       階層ハッシュのパラメータ
    * @param[in]    pBufferManagerSet                   ハッシュ検証用キャッシュマネージャ
    * @param[in]    pLocker                             キャッシュマネージャのスレッド間同期オブジェクト
    *
    * @return       関数の処理結果を返します。
    */
    Result Initialize(
               fs::SubStorage storageJournalControlArea,
               fs::SubStorage storageJournalTable,
               fs::SubStorage storageJournalBitmapUpdatedPhysical,
               fs::SubStorage storageJournalBitmapUpdatedVirtual,
               fs::SubStorage storageJournalBitmapUnassigned,
               fs::SubStorage storageIntegrityMasterHash,
               fs::SubStorage storageIntegrityLayeredHashL1,
               fs::SubStorage storageIntegrityLayeredHashL2,
               fs::SubStorage storageIntegrityLayeredHashL3,
               fs::SubStorage storageData,
               const HierarchicalIntegrityVerificationInformation& integrityInfo,
               FilesystemBufferManagerSet* pBufferManagerSet,
               nn::os::Mutex* pLocker
           ) NN_NOEXCEPT;

    /**
    * @brief        完全性検証ストレージレイヤーをアンマウントします。
    */
    void Finalize() NN_NOEXCEPT
    {
        m_IntegrityStorage.Finalize();
    }

    /**
    * @brief        データを書き込みます。
    *
    * @param[in]    offset  書き込み開始位置
    * @param[in]    buffer  書き込むデータ
    * @param[in]    size    書き込むデータサイズ
    *
    * @return       関数の処理結果を返します。
    */
    virtual Result Write(int64_t offset, const void* buffer, size_t size) NN_NOEXCEPT NN_OVERRIDE
    {
        return m_IntegrityStorage.Write(offset, buffer, size);
    }

    /**
    * @brief        データを読み込みます。
    *
    * @param[in]    offset  読み込み開始位置
    * @param[out]   buffer  読み込んだ内容をコピーするバッファ
    * @param[in]    size    読み込むデータサイズ
    *
    * @return       関数の処理結果を返します。
    */
    virtual Result Read(int64_t offset, void* buffer, size_t size) NN_NOEXCEPT NN_OVERRIDE
    {
        return m_IntegrityStorage.Read(offset, buffer, size);
    }

    /**
    * @brief        書き込みのフラッシュを行ないます。
    *
    * @return       関数の処理結果を返します。
    */
    virtual Result Flush() NN_NOEXCEPT NN_OVERRIDE
    {
        return m_IntegrityStorage.Flush();
    }

    /**
    * @brief        ファイルサイズを取得します。
    *
    * @param[out]   outValue    ファイルサイズ
    *
    * @return       関数の処理結果を返します。
    */
    virtual Result GetSize(int64_t* outValue) NN_NOEXCEPT NN_OVERRIDE
    {
        return m_IntegrityStorage.GetSize(outValue);
    }

    /**
    * @brief        範囲を指定して操作を実行します。
    *
    * @param[out]   outBuffer       操作の結果を格納するバッファ
    * @param[in]    outBufferSize   操作の結果を格納するバッファのサイズ
    * @param[in]    operationId     実行する操作
    * @param[in]    offset          操作を実行する範囲の先頭オフセット
    * @param[in]    size            操作を実行する範囲のサイズ
    * @param[in]    inBuffer        操作に渡すバッファ
    * @param[in]    inBufferSize    操作に渡すバッファのサイズ
    *
    * @return       関数の処理結果を返します。
    */
    virtual Result OperateRange(
                       void* outBuffer,
                       size_t outBufferSize,
                       fs::OperationId operationId,
                       int64_t offset,
                       int64_t size,
                       const void* inBuffer,
                       size_t inBufferSize
                   ) NN_NOEXCEPT NN_OVERRIDE;

    using IStorage::OperateRange;

    /**
    * @brief        コミット操作を実行します。
    *
    * @return       関数の処理結果を返します。
    */
    Result Commit() NN_NOEXCEPT;

    /**
    * @brief        ロールバック前処理を行います。
    *
    * @return       関数の処理結果を返します。
    */
    Result OnRollback() NN_NOEXCEPT;

    /**
    * @brief        バッファセットを取得します。
    *
    * @return       バッファセットを返します。
    */
    FilesystemBufferManagerSet* GetBufferManagerSet() NN_NOEXCEPT
    {
        return m_IntegrityStorage.GetBuffers();
    }

    /**
    * @brief        予約領域を除くデータ領域のサイズを取得します。
    *
    * @return       予約領域を除くデータ領域のサイズを返します。
    */
    int64_t GetDataAreaSize() const NN_NOEXCEPT
    {
        return m_JournalStorage.GetDataAreaSize();
    }

    /**
    * @brief        予約領域のサイズを取得します。
    *
    * @return       予約領域のサイズを返します。
    */
    int64_t GetReservedAreaSize() const NN_NOEXCEPT
    {
        return m_JournalStorage.GetReservedAreaSize();
    }

    /**
    * @brief        ブロックサイズを取得します。
    *
    * @return       ブロックサイズを返します。
    */
    int64_t GetBlockSize() const NN_NOEXCEPT
    {
        return m_JournalStorage.GetBlockSize();
    }

    /**
    * @brief        パラメータを取得します。
    *
    * @return       パラメータを返します。
    */
    void GetParameters(
        HierarchicalIntegrityVerificationStorageControlArea::InputParam* outParam) const NN_NOEXCEPT
    {
        m_IntegrityStorage.GetParameters(outParam);
    }

    //! セーブデータ内部ストレージアクセス用の Vistor を受け入れます。
    nn::Result AcceptVisitor(IInternalStorageFileSystemVisitor* pVisitor) NN_NOEXCEPT;

private:
    JournalStorage m_JournalStorage;
    HierarchicalIntegrityVerificationStorage m_IntegrityStorage;
};

}}}

