﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <cstring>
#include <nn/nn_Common.h>
#include <nn/fs/fs_SaveDataTypes.h>
#include <nn/fs/fs_Types.h>
#include <nn/crypto/crypto_Sha256Generator.h>
#include <nn/fssystem/save/fs_ISaveFile.h>
#include <nn/fssystem/save/fs_IntegrityVerificationStorage.h>
#include <nn/fssystem/save/fs_BlockCacheBufferedStorage.h>

class FsHierarchicalIntegrityVerificationStorageTest;

namespace nn { namespace fssystem { namespace save {

//!< 完全性検証階層の情報
struct HierarchicalIntegrityVerificationLevelInformation
{
    nn::fs::Int64 offset;          //!< その層のデータがおいてあるオフセット
    nn::fs::Int64 size;            //!< その層のデータサイズ
    int32_t orderBlock;            //!< その層のハッシュ計算単位
    char reserved[4];
};
NN_STATIC_ASSERT(std::is_pod<HierarchicalIntegrityVerificationLevelInformation>::value);

//!< 完全性検証ストレージに与える階層ハッシュの情報
struct HierarchicalIntegrityVerificationInformation
{
    //!< 最大回数層
    uint32_t maxLayers;

    //!< 層ごとの情報
    HierarchicalIntegrityVerificationLevelInformation info[IntegrityMaxLayerCount - 1];

    //! ソルトのシード
    nn::fs::HashSalt seed;

    //!< 階層ハッシュの総サイズを取得します。
    int64_t GetLayeredHashSize() const
    {
        return this->info[this->maxLayers - 2].offset;
    }

    //!< 実データ層のオフセットを取得します。
    int64_t GetDataOffset() const
    {
        return this->info[this->maxLayers - 2].offset;
    }

    //!< 実データ層のサイズを取得します。
    int64_t GetDataSize() const
    {
        return this->info[this->maxLayers - 2].size;
    }
};
NN_STATIC_ASSERT(std::is_pod<HierarchicalIntegrityVerificationInformation>::value);

//!< メタデータ領域の形式
struct HierarchicalIntegrityVerificationMetaInformation
{
    //!< マジックコード
    uint32_t magic;

    //!< バージョン
    uint32_t version;

    //!< マスターハッシュの情報
    uint32_t sizeMasterHash;

    //!< 階層ハッシュの情報
    HierarchicalIntegrityVerificationInformation infoLevelHash;

    //! メタデータ領域を初期化します。
    void Format() NN_NOEXCEPT;
    void Format(const nn::fs::SaveDataHashSalt& hashSalt) NN_NOEXCEPT;
};
NN_STATIC_ASSERT(std::is_pod<HierarchicalIntegrityVerificationMetaInformation>::value);

//! データサイズ
struct HierarchicalIntegrityVerificationSizeSet
{
    int64_t controlSize;                                   //!< 管理領域 + オプションデータ領域の合計サイズ
    int64_t masterHashSize;                                //!< マスターハッシュのサイズ
    int64_t layeredHashSizes[IntegrityMaxLayerCount - 2];  //!< 階層ハッシュのサイズ
};
NN_STATIC_ASSERT(std::is_pod<HierarchicalIntegrityVerificationSizeSet>::value);

//!< メタデータとオプションデータ領域を管理するクラスです。
class HierarchicalIntegrityVerificationStorageControlArea
{
    NN_DISALLOW_COPY(HierarchicalIntegrityVerificationStorageControlArea);

public:
    static const size_t HashSize = nn::crypto::Sha256Generator::HashSize;

    //!< 入力パラメータ
    struct InputParam
    {
        size_t sizeBlockLevel[IntegrityMaxLayerCount - 1];
    };
    NN_STATIC_ASSERT(std::is_pod<InputParam>::value);

public:
    /**
    * @brief        完全性検証ストレージの総データサイズを取得します。
    *
    * @param[out]   outSizeSet          サイズ
    * @param[in]    inputParam          入力パラメータ
    * @param[in]    countLayer          階層数
    * @param[in]    sizeData            実データサイズ
    *
    * @return       関数の処理結果を返します。
    *
    * @details      レイヤーハッシュと本体のサイズを別計算します
    */
    static Result QuerySize(
                      HierarchicalIntegrityVerificationSizeSet* outSizeSet,
                      const InputParam& inputParam,
                      int32_t countLayer,
                      int64_t sizeData
                  ) NN_NOEXCEPT;

    /**
    * @brief        完全性検証ストレージ用メタデータ領域をフォーマットします。
    *
    * @param[in]    storageMeta メタデータをフォーマットするストレージ
    * @param[in]    meta        メタデータ
    *
    * @return       関数の処理結果を返します。
    *
    * @details      この関数では以下の処理を行います。
    *
    *               ・管理領域の生成。
    *               ・あらかじめ計算されたハッシュ領域、実データ領域の、それぞれのオフセット、サイズを管理領域に覚えます。
    *                 オフセット情報はフォーマット後は基本的に読み取り専用のデータですが、拡張時に上書きされる可能性があります。
    */
    static Result Format(
                      fs::SubStorage storageMeta,
                      const HierarchicalIntegrityVerificationMetaInformation& meta
                  ) NN_NOEXCEPT;

    /**
    * @brief        完全性検証ストレージ用メタデータ領域を拡張します。
    *
    * @param[in]    storageMeta メタデータを再フォーマットするストレージ
    * @param[in]    meta        拡張後のメタデータ
    *
    * @return       関数の処理結果を返します。
    *
    * @details      この関数では以下の処理を行います。
    *
    *               ・あらかじめ計算されたハッシュ領域、実データ領域の、それぞれのオフセット、サイズを管理領域に覚え直します。
    */
    static Result Expand(
                      fs::SubStorage storageMeta,
                      const HierarchicalIntegrityVerificationMetaInformation& meta
                  ) NN_NOEXCEPT;

    /**
    * @brief        コンストラクタ
    */
    HierarchicalIntegrityVerificationStorageControlArea() NN_NOEXCEPT
    {
    }

    /**
    * @brief        完全性検証ストレージ用メタデータをマウントします。
    *
    * @param[in]    storageMeta メタデータストレージ
    *
    * @return       関数の処理結果を返します。
    */
    Result Initialize(fs::SubStorage storageMeta) NN_NOEXCEPT;

    /**
    * @brief        完全性検証ストレージ用メタデータをアンマウントします。
    */
    void Finalize() NN_NOEXCEPT;

    /**
    * @brief        マスターハッシュのサイズを取得します。
    *
    * @return       マスターハッシュのサイズ
    */
    uint32_t GetMasterHashSize() const NN_NOEXCEPT
    {
        return m_Meta.sizeMasterHash;
    }

    /**
    * @brief        階層ハッシュの情報を取得します。
    *
    * @param[out]   outInfo 階層ハッシュの情報
    */
    void GetLevelHashInfo(
             HierarchicalIntegrityVerificationInformation* outInfo
         ) const NN_NOEXCEPT
    {
        NN_SDK_ASSERT_NOT_NULL(outInfo);
        *outInfo = m_Meta.infoLevelHash;
    }

protected:
    fs::SubStorage m_Storage;
    HierarchicalIntegrityVerificationMetaInformation m_Meta; //!< メタデータ
};

/**
* @brief 階層的完全性検証ストレージです。
*
*        実データ領域への読み込み時には、ハッシュ検証を行い
*        データが不正に書き換えられていないことを保障します。
*        不正に書き換えられていることを検出した場合には、
*        ハッシュ検証失敗を示すエラーを返します。
*
*        実データ領域への書き込み時には、各階層のハッシュを更新します。
*
*        このクラスでは 3 階層のハッシュ検証レイヤーを使用します。
*        ストレージサイズ 4G バイト、ブロックサイズ 4K バイト、
*        ハッシュブロックサイズ 32 バイト(SHA256換算)と仮定した場合の
*        ハッシュデータサイズの概算は以下のようになります。
*
*                      ハッシュ検証対象データのサイズ  ハッシュデータのサイズ
*        マスター  <=   2KB
*        階層1     <=   256KB                          2KB   (256K / 128)
*        階層2     <=   32MB                           256KB (32M / 128)
*        階層3     <=   4GB                            32MB  (4G / 128)
*
*        ストレージサイズ 4G バイト、ブロックサイズ 512 バイト、
*        ハッシュブロックサイズ 32 バイト(SHA256換算)と仮定した場合の
*        ハッシュデータサイズの概算は以下のようになります。
*
*                      ハッシュ検証対象データのサイズ  ハッシュデータのサイズ
*        マスター  <=   32KB
*        階層1     <=   512KB                          32KB  (512M / 16)
*        階層2     <=   8MB                            512KB (8M / 16)
*        階層3     <=   128MB                          8MB   (128M / 16)
*        階層4     <=   4GB                            128MB (4G / 16)
*/
class HierarchicalIntegrityVerificationStorage : public fs::IStorage
{
    NN_DISALLOW_COPY(HierarchicalIntegrityVerificationStorage);

protected:
    static const int64_t HASH_SIZE = nn::crypto::Sha256Generator::HashSize;
    static const size_t MAX_LAYERS = IntegrityMaxLayerCount;

public:
    class HierarchicalStorageInformation
    {
    public:
        enum {
            MasterStorage = 0,
            Layer1Storage,
            Layer2Storage,
            Layer3Storage,
            Layer4Storage,
            Layer5Storage,
            DataStorage
        };
    public:
        void SetMasterHashStorage(fs::SubStorage storage)
        {
            m_Storage[MasterStorage] = storage;
        }
        void SetLayer1HashStorage(fs::SubStorage storage)
        {
            m_Storage[Layer1Storage] = storage;
        }
        void SetLayer2HashStorage(fs::SubStorage storage)
        {
            m_Storage[Layer2Storage] = storage;
        }
        void SetLayer3HashStorage(fs::SubStorage storage)
        {
            m_Storage[Layer3Storage] = storage;
        }
        void SetLayer4HashStorage(fs::SubStorage storage)
        {
            m_Storage[Layer4Storage] = storage;
        }
        void SetLayer5HashStorage(fs::SubStorage storage)
        {
            m_Storage[Layer5Storage] = storage;
        }
        void SetDataStorage(fs::SubStorage storage)
        {
            m_Storage[DataStorage] = storage;
        }

        fs::SubStorage& operator[](int32_t index)
        {
            NN_SDK_ASSERT( index >= MasterStorage && index <= DataStorage );
            return m_Storage[index];
        }

    private:
        fs::SubStorage m_Storage[DataStorage + 1];
    };

public:
    typedef void (*GenerateRandomFunction)(void* pData, size_t size);

public:
    /**
    * @brief        コンストラクタ
    */
    HierarchicalIntegrityVerificationStorage() NN_NOEXCEPT;

    /**
    * @brief        デストラクタ
    */
    virtual ~HierarchicalIntegrityVerificationStorage() NN_NOEXCEPT;

    /**
    * @brief        完全性検証ストレージをマウントします。
    *
    * @param[in]    info                    L1, L2, L3, 実データの情報
    * @param[in]    storage                 マスターハッシュ、L1 ～ Ln ハッシュ、実データストレージ
    * @param[in]    pBuffer                 キャッシュバッファマネージャ
    * @param[in]    pLocker                 排他制御用のロックオブジェクト
    * @param[in]    isRomFs                 ROMFSかどうか
    *
    * @return       関数の処理結果を返します。
    *
    * @details      完全性検証ストレージレイヤーをフォーマットする、という操作は不要です。
    *               使用しているうちに、自動的にフォーマット相当の処理が進行していきます。
    */
    Result Initialize(
               const HierarchicalIntegrityVerificationInformation& info,
               HierarchicalStorageInformation storage,
               FilesystemBufferManagerSet* pBuffers,
               os::Mutex* pLocker,
               fs::StorageType storageType
           ) NN_NOEXCEPT;

    /**
    * @brief        完全検証ストレージを閉じます。
    */
    void Finalize() NN_NOEXCEPT;

    /**
    * @brief        ハッシュ検証つきで実データ領域からデータを読み込みます。
    *
    * @param[in]    offset  読み込み開始位置
    * @param[out]   buffer  読み込んだ内容をコピーするバッファ
    * @param[in]    size    読み込むデータサイズ
    *
    * @return       関数の処理結果を返します。
    *
    * @details      読み込もうとしたデータに対するハッシュが不正だった場合には、0 フィルされたデータが
    *               読込先バッファに書き込まれ、nn::fs::ResultFailedVerifySign を返します。
    */
    virtual Result Read(
                       int64_t offset,
                       void* buffer,
                       size_t size
                   ) NN_NOEXCEPT NN_OVERRIDE;

    /**
    * @brief        実データ領域にデータを書き込み、ハッシュを更新します。
    *
    * @param[in]    offset  書き込み開始位置
    * @param[in]    buffer  書き込むデータ
    * @param[in]    size    書き込むデータサイズ
    *
    * @return       関数の処理結果を返します。
    */
    virtual Result Write(
                       int64_t offset,
                       const void* buffer,
                       size_t size
                   ) NN_NOEXCEPT NN_OVERRIDE;

    /**
    * @brief        実データの総バイトサイズを取得します。
    *
    * @param[out]   outValue    実データの総バイトサイズ
    *
    * @return       関数の処理結果を返します。
    */
    virtual Result GetSize(int64_t* outValue) NN_NOEXCEPT NN_OVERRIDE;

    /**
    * @brief        サイズは変更できません。初期化時のみ指定できます。
    *
    * @return       常に nn::fs::ResultUnsupportedOperation
    */
    virtual Result SetSize(int64_t) NN_NOEXCEPT NN_OVERRIDE
    {
        return nn::fs::ResultUnsupportedOperation();
    }

    /**
    * @brief        フラッシュします。
    *
    * @return       関数の処理結果を返します。
    */
    virtual Result Flush() NN_NOEXCEPT NN_OVERRIDE;

    /**
    * @brief       範囲指定処理を行います。
    *
    * @param[out]  outBuffer        範囲指定処理の結果を格納するバッファ
    * @param[in]   outBufferSize    範囲指定処理の結果を格納するバッファのサイズ
    * @param[in]   operationId      範囲指定処理の種類
    * @param[in]   offset           範囲指定処理開始位置
    * @param[in]   size             範囲指定処理を行うデータサイズ
    * @param[in]   inBuffer         範囲指定処理に渡すバッファ
    * @param[in]   inBufferSize     範囲指定処理に渡すバッファのサイズ
    *
    * @return      関数の処理結果を返します。
    */
    virtual Result OperateRange(
                       void* outBuffer,
                       size_t outBufferSize,
                       fs::OperationId operationId,
                       int64_t offset,
                       int64_t size,
                       const void* inBuffer,
                       size_t inBufferSize
                   ) NN_NOEXCEPT NN_OVERRIDE;

    using IStorage::OperateRange;

    /**
    * @brief        完全性検証ストレージへの変更をコミットします。
    *
    * @return       関数の処理結果を返します。
    */
    Result Commit() NN_NOEXCEPT;

    /**
    * @brief        完全性検証ストレージへの変更を破棄します。
    *
    * @return       関数の処理結果を返します。
    */
    Result OnRollback() NN_NOEXCEPT;

    /**
    * @brief        初期化済みかどうかを取得します。
    */
    bool IsInitialized() const NN_NOEXCEPT
    {
        return (m_SizeData >= 0);
    }

    /**
    * @brief        マウント、コミット後に一度でも書き込みが発生したかどうかを取得します。
    *
    * @return       書き込みが発生したかどうか
    */
    bool IsWrittenForRollback() const NN_NOEXCEPT
    {
        return m_IsWrittenForRollback;
    }

    /**
    * @brief        バッファセットを取得します。
    *
    * @return       バッファセットを返します。
    */
    FilesystemBufferManagerSet* GetBuffers() NN_NOEXCEPT
    {
        return m_pBuffers;
    }

    /**
    * @brief        パラメータを取得します。
    *
    * @param[out]   outParam    パラメータの格納先
    */
    void GetParameters(
        HierarchicalIntegrityVerificationStorageControlArea::InputParam* outParam) const NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_NOT_NULL(outParam);
        for( auto level = 0; level <= m_MaxLayers - 2; ++level )
        {
            outParam->sizeBlockLevel[level]
                = static_cast<size_t>(m_StorageVerify[level].GetBlockSize());
        }
    }

    /**
    * @brief        乱数列生成関数をセットします。
    *
    * @return       なし
    */
    static void SetGenerateRandomFunction(
        GenerateRandomFunction function
    ) NN_NOEXCEPT;

    int64_t GetL1HashVerificationBlockSize() const NN_NOEXCEPT
    {
        return m_StorageVerify[m_MaxLayers - 2].GetBlockSize();
    }

    fs::SubStorage GetL1HashStorage() NN_NOEXCEPT
    {
        return fs::SubStorage(
                    &m_StorageBuffer[m_MaxLayers - 3],
                    0,
                    nn::util::DivideUp(m_SizeData, GetL1HashVerificationBlockSize()) * HASH_SIZE);
    }

private:
    static GenerateRandomFunction m_pGenerateRandom;

private:
    FilesystemBufferManagerSet* m_pBuffers;
    os::Mutex* m_pLocker;
    IntegrityVerificationStorage m_StorageVerify[MAX_LAYERS - 1]; //! 完全性検証の階層
    BlockCacheBufferedStorage m_StorageBuffer[MAX_LAYERS - 1];    //! 完全性検証対象データ
    int64_t m_SizeData;                                           //! 実データサイズ
    int32_t m_MaxLayers;                                          //! 最大レイヤー
    bool m_IsWrittenForRollback;                                  //! 書き込みが発生したかどうか (ロールバック用)

private:
    // 乱数列生成関数へのアクセスのため HierarchicalIntegrityVerificationMetaInformation をフレンドにします。
    friend struct HierarchicalIntegrityVerificationMetaInformation;

    // テスト用クラスには内部データを公開します。
    friend class ::FsHierarchicalIntegrityVerificationStorageTest;
};

}}}

