﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/fssystem/fs_IndirectStorage.h>

namespace nn { namespace fssystem {

namespace utilTool {
    class SparseStorageBuilder;
    class SparseStorageChecker;
}

/**
 * @brief   使用しない領域を前詰めしたストレージからデータを読み込むクラスです。
 */
class SparseStorage : public IndirectStorage
{
    NN_DISALLOW_COPY(SparseStorage);

public:
    /**
     * @brief   コンストラクタです。
     */
    SparseStorage() NN_NOEXCEPT
        : IndirectStorage()
        , m_ZeroStorage()
    {
    }

    /**
     * @brief   デストラクタです。
     */
    virtual ~SparseStorage() NN_NOEXCEPT NN_OVERRIDE
    {
    }

    using IndirectStorage::Initialize;

    /**
     * @brief   データをすべて削除したものとして初期化します。
     *
     * @param[in]   endOffset   管理する領域の終了オフセット
     *
     * @pre
     *      - 0 <= endOffset
     *      - 未初期化
     */
    void Initialize(int64_t endOffset) NN_NOEXCEPT
    {
        // 引数の事前検証は Initialize() に任せる
        GetEntryTable().Initialize(NodeSize, endOffset);

        SetZeroStorage();
    }

    /**
     * @brief   データが存在するストレージを設定します。
     *
     * @param[in]   storage     データストレージ
     */
    void SetDataStorage(fs::SubStorage storage) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES(IsInitialized());

        SetStorage(0, storage);
        SetZeroStorage();
    }

    /**
     * @brief   データが存在するストレージを設定します。
     *
     * @param[in]   pStorage    ストレージのポインタ
     * @param[in]   offset      ストレージのオフセット
     * @param[in]   size        ストレージのサイズ
     */
    template< typename T >
    void SetDataStorage(T pStorage, int64_t offset, int64_t size) NN_NOEXCEPT
    {
        // 引数の事前検証は SubStorage に任せる
        NN_SDK_REQUIRES(IsInitialized());

        SetStorage(0, pStorage, offset, size);
        SetZeroStorage();
    }

    /**
     * @brief   ストレージを読み込みます。
     *
     * @param[in]   offset  オフセット
     * @param[in]   buffer  バッファの先頭アドレス
     * @param[in]   size    サイズ
     *
     * @return  関数の実行結果を返します。
     */
    virtual Result Read(int64_t offset, void* buffer, size_t size) NN_NOEXCEPT NN_OVERRIDE;

private:
    // 常に 0 を返すストレージ
    class ZeroStorage : public fs::IStorage, public fs::detail::Newable
    {
    public:
        ZeroStorage() NN_NOEXCEPT {}
        virtual ~ZeroStorage() NN_NOEXCEPT NN_OVERRIDE {}

        virtual Result Read(int64_t offset, void* buffer, size_t size) NN_NOEXCEPT NN_OVERRIDE
        {
            NN_SDK_REQUIRES_GREATER_EQUAL(offset, 0);
            NN_SDK_REQUIRES((buffer != nullptr) || (size == 0));
            NN_UNUSED(offset);
            if( 0 < size )
            {
                std::memset(buffer, 0, size);
            }
            NN_RESULT_SUCCESS;
        }

        virtual Result Flush() NN_NOEXCEPT NN_OVERRIDE
        {
            NN_RESULT_SUCCESS;
        }

        virtual Result GetSize(int64_t* outValue) NN_NOEXCEPT NN_OVERRIDE
        {
            NN_SDK_REQUIRES_NOT_NULL(outValue);
            // IndirectStorage 側で offset が調整されるので ZeroStorage のサイズは最大にしておく
            *outValue = std::numeric_limits<int64_t>::max();
            NN_RESULT_SUCCESS;
        }

        virtual Result OperateRange(
                           void* outBuffer,
                           size_t outBufferSize,
                           fs::OperationId operationId,
                           int64_t offset,
                           int64_t size,
                           const void* inBuffer,
                           size_t inBufferSize
                       ) NN_NOEXCEPT NN_OVERRIDE
        {
            NN_UNUSED(outBuffer);
            NN_UNUSED(outBufferSize);
            NN_UNUSED(operationId);
            NN_UNUSED(offset);
            NN_UNUSED(size);
            NN_UNUSED(inBuffer);
            NN_UNUSED(inBufferSize);
            NN_RESULT_SUCCESS;
        }
    };

    void SetZeroStorage() NN_NOEXCEPT
    {
        SetStorage(1, &m_ZeroStorage, 0, std::numeric_limits<int64_t>::max());
    }

private:
    ZeroStorage m_ZeroStorage;

    friend class SparseStorageTest;
    friend class utilTool::SparseStorageBuilder;
    friend class utilTool::SparseStorageChecker;
};

}}
