﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Common.h>
#include <nn/nn_Result.h>
#include <nn/fs/fs_IStorage.h>
#include <nn/fs/detail/fs_Newable.h>
#include <nn/crypto/crypto_Sha256Generator.h>
#include <nn/fs/fs_ResultPrivate.h>

#include <nn/nn_Allocator.h>

namespace nn { namespace fssystem {

namespace detail {
    class PartitionFileSystemFormat
    {
    public:
        using ResultPartitionSignatureVerificationFailed = fs::ResultPartitionSignatureVerificationFailed;

    public:
        static const int EntryNameLengthMax = 768;
        static const char VersionSignature[];
        static const int FileDataAlignmentSize = 32;

    public:
#pragma pack(push, 1)
        struct PartitionEntry
        {
            uint64_t offset;
            uint64_t size;
            uint32_t nameOffset;
            uint32_t reserved;
        };
        NN_STATIC_ASSERT(std::is_pod<PartitionEntry>::value);
#pragma pack(pop)

        struct FileEntryForConstruct
        {
            uint64_t offset;
            uint64_t size;
            char name[EntryNameLengthMax + 1];
        };
        NN_STATIC_ASSERT(std::is_pod<FileEntryForConstruct>::value);
    };

    class Sha256PartitionFileSystemFormat
    {
    public:
        using ResultPartitionSignatureVerificationFailed = fs::ResultSha256PartitionSignatureVerificationFailed;

    public:
        static const int EntryNameLengthMax = 768;
        static const char VersionSignature[];
        static const int FileDataAlignmentSize = 512;
        static const int HashSize = nn::crypto::Sha256Generator::HashSize;

#pragma pack(push, 1)
        struct PartitionEntry
        {
            uint64_t offset;
            uint64_t size;
            uint32_t nameOffset;
            uint32_t hashTargetSize;
            uint64_t hashTargetOffset;
            char     hash[HashSize];
        };
        NN_STATIC_ASSERT(std::is_pod<PartitionEntry>::value);
#pragma pack(pop)

        struct FileEntryForConstruct
        {
            uint64_t offset;
            uint64_t size;
            uint64_t hashTargetOffset;
            uint32_t hashTargetSize;
            char hash[HashSize];
            char name[EntryNameLengthMax + 1];
        };
        NN_STATIC_ASSERT(std::is_pod<FileEntryForConstruct>::value);

    };

} // detail

    template <typename Format>
    class PartitionFileSystemMetaCore : public nn::fs::detail::Newable
    {
    public:
        struct PartitionFileSystemHeader;

        typedef typename Format::PartitionEntry PartitionEntry;


    protected:
        bool m_IsInitialized;
        PartitionFileSystemHeader* m_pHeader;
        PartitionEntry* m_pEntries;
        char* m_pNameTable;

        size_t m_MetaDataSize;

        MemoryResource* m_pAllocator;
        char* m_pBufferToBeFreed;

    public:

        typedef typename Format::FileEntryForConstruct FileEntryForConstruct;
        static const int FileDataAlignmentSize = Format::FileDataAlignmentSize;
        static const int EntryNameLengthMax = 768;

        PartitionFileSystemMetaCore();
        ~PartitionFileSystemMetaCore();

        // 渡された PartitionFileSystem フォーマットの IStorage から初期化する。
        // あらかじめ QueryMetaDataBufferSize() で取得したサイズのバッファを渡す必要がある。
        // バッファはインスタンスが破棄されるまで利用される。
        Result Initialize(fs::IStorage* pBaseStorage, void* metaDataBuffer, size_t metaDataBufferSize) NN_NOEXCEPT;

        // 渡された PartitionFileSystem フォーマットの IStorage から初期化する。
        // 渡されたアロケータからバッファを確保する。
        Result Initialize(fs::IStorage* pBaseStorage, MemoryResource* pAllocator) NN_NOEXCEPT;

        int GetEntryIndex(const char* path) const NN_NOEXCEPT;
        const PartitionEntry* GetEntry(int index) const NN_NOEXCEPT;
        int GetEntryCount() const NN_NOEXCEPT;
        const char* GetEntryName(int index) const NN_NOEXCEPT;
        size_t GetMetaDataSize() const NN_NOEXCEPT;

        // 渡された PartitionFileSystem フォーマットの IStorage のメタデータサイズ = Initialize() に必要なバッファサイズを返す
        static Result QueryMetaDataSize(size_t* outValue, fs::IStorage* pBaseStorage) NN_NOEXCEPT;

        // 渡されたファイルリストで PartitionFileSystem フォーマットを構築する際のメタデータサイズ = ConstructMetaData() に必要なバッファサイズを返す
        static Result QueryMetaDataSize(size_t* outValue, const FileEntryForConstruct* pEntryArray, int entryCount) NN_NOEXCEPT;

        // 渡されたファイルリストで、渡されたバッファ上に PartitionFileSystem フォーマットを構築する
        static Result ConstructMetaData(void* buffer, size_t bufferSize, const FileEntryForConstruct* pEntryArray, int entryCount) NN_NOEXCEPT;

    protected:
        void DeallocateBuffer() NN_NOEXCEPT;
    };



    typedef PartitionFileSystemMetaCore<detail::PartitionFileSystemFormat> PartitionFileSystemMeta;

    class Sha256PartitionFileSystemMeta : public PartitionFileSystemMetaCore<detail::Sha256PartitionFileSystemFormat>
    {
    public:
        using PartitionFileSystemMetaCore<detail::Sha256PartitionFileSystemFormat>::Initialize;
        Result Initialize(fs::IStorage* pBaseStorage, MemoryResource* pAllocator, const void* pHash, size_t hashSize) NN_NOEXCEPT;
    };

}}
