﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/fs/fs_SubStorage.h>
#include <nn/fssystem/fs_NcaFileSystemDriver.h>
#if defined(NN_BUILD_CONFIG_OS_WIN)
#include <nn/fssystem/utilTool/fs_NcaSparseStorage.h>
#endif

namespace nn { namespace fssystem {

class NcaFileSystemDriver::StorageOption
{
public:
    explicit StorageOption(NcaFsHeaderReader* pHeaderReader) NN_NOEXCEPT;
    StorageOption(NcaFsHeaderReader* pHeaderReader, int index) NN_NOEXCEPT;

    int GetFsIndex() const NN_NOEXCEPT
    {
        return m_FsIndex;
    }

    NcaFsHeaderReader& GetHeaderReader() NN_NOEXCEPT
    {
        return *m_pHeaderReader;
    }

    const NcaFsHeaderReader& GetHeaderReader() const NN_NOEXCEPT
    {
        return *m_pHeaderReader;
    }

    fs::SubStorage GetDataStorage() const NN_NOEXCEPT
    {
        return fs::SubStorage(m_pDataStorage, 0, m_DataStorageSize);
    }

    fs::IStorage* GetAesCtrExTableStorage() const NN_NOEXCEPT
    {
        return m_pAesCtrExTableStorage;
    }

    fs::IStorage* GetAesCtrExStorage() const NN_NOEXCEPT
    {
        return m_pAesCtrExStorage;
    }

    AesCtrCounterExtendedStorage* GetAesCtrExStorageRaw() const NN_NOEXCEPT
    {
        return m_pAesCtrExStorageRaw;
    }

    IndirectStorage* GetIndirectStorage() const NN_NOEXCEPT
    {
        return m_pIndirectStorage;
    }

    SparseStorage* GetSparseStorage() const NN_NOEXCEPT
    {
        return m_pSparseStorage;
    }

#if defined(NN_BUILD_CONFIG_OS_WIN)
    void SetOriginalSparseStorage(std::shared_ptr<utilTool::NcaSparseStorage> pSparseStorage) NN_NOEXCEPT
    {
        m_pOriginalSparseStorage = pSparseStorage;
    }

    std::shared_ptr<utilTool::NcaSparseStorage> GetOriginalSparseStorage() const NN_NOEXCEPT
    {
        return m_pOriginalSparseStorage;
    }

    void SetExternalSparseStorage(std::shared_ptr<utilTool::NcaSparseStorage> pSparseStorage) NN_NOEXCEPT
    {
        m_pExternalSparseStorage = pSparseStorage;
    }

    std::shared_ptr<utilTool::NcaSparseStorage> GetExternalSparseStorage() const NN_NOEXCEPT
    {
        return m_pExternalSparseStorage;
    }

    bool HasExternalSparseStorage() const NN_NOEXCEPT
    {
        return m_pExternalSparseStorage != nullptr;
    }
#endif

private:
    void SetDataStorage(fs::IStorage* pStorage, int64_t size) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_NOT_NULL(pStorage);
        NN_SDK_REQUIRES_GREATER_EQUAL(size, 0);

        m_pDataStorage = pStorage;
        m_DataStorageSize = size;
    }

    void SetAesCtrExTableStorage(fs::IStorage* pStorage) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_NOT_NULL(pStorage);
        m_pAesCtrExTableStorage = pStorage;
    }

    void SetAesCtrExStorage(fs::IStorage* pStorage) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_NOT_NULL(pStorage);
        m_pAesCtrExStorage = pStorage;
    }

    void SetAesCtrExStorageRaw(AesCtrCounterExtendedStorage* pStorage) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_NOT_NULL(pStorage);
        m_pAesCtrExStorageRaw = pStorage;
    }

    void SetIndirectStorage(IndirectStorage* pStorage) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_NOT_NULL(pStorage);
        m_pIndirectStorage = pStorage;
    }

    void SetSparseStorage(SparseStorage* pStorage) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_NOT_NULL(pStorage);
        m_pSparseStorage = pStorage;
    }

private:
    const int m_FsIndex;
    NcaFsHeaderReader* const m_pHeaderReader;
    fs::IStorage* m_pDataStorage;
    int64_t m_DataStorageSize;
    fs::IStorage* m_pAesCtrExTableStorage;
    AesCtrCounterExtendedStorage* m_pAesCtrExStorageRaw;
    fs::IStorage* m_pAesCtrExStorage;
    IndirectStorage* m_pIndirectStorage;
    SparseStorage* m_pSparseStorage;
#if defined(NN_BUILD_CONFIG_OS_WIN)
    std::shared_ptr<utilTool::NcaSparseStorage> m_pOriginalSparseStorage;
    std::shared_ptr<utilTool::NcaSparseStorage> m_pExternalSparseStorage;
#endif

    friend class NcaFileSystemDriver;
};

class NcaFileSystemDriver::StorageOptionWithHeaderReader : public NcaFileSystemDriver::StorageOption
{
public:
    explicit StorageOptionWithHeaderReader(int index) NN_NOEXCEPT
        : StorageOption(&m_HeaderReader, index)
    {
    }

private:
    NcaFsHeaderReader m_HeaderReader;
};

#if defined(NN_BUILD_CONFIG_OS_WIN)
class NcaFileSystemDriver::SparseParam
{
public:
    SparseParam(utilTool::SparseStorageBuilder* pBuilder, NcaFsHeaderReader* pHeaderReader, bool isModeEmpty, int fsIndex, int64_t physicalOffset) NN_NOEXCEPT;
    ~SparseParam() NN_NOEXCEPT;

    utilTool::SparseStorageBuilder& GetBuilder() const NN_NOEXCEPT
    {
        return *m_pBuilder;
    }

    NcaFsHeaderReader& GetHeaderReader() const NN_NOEXCEPT
    {
        return *m_pHeaderReader;
    }

    bool IsModeEmpty() const NN_NOEXCEPT
    {
        return m_IsModeEmpty;
    }

    int GetFsIndex() const NN_NOEXCEPT
    {
        return m_FsIndex;
    }

    int64_t GetPhysicalOffset() const NN_NOEXCEPT
    {
        return m_PhysicalOffset;
    }

    void SetSparseStorage(std::shared_ptr<utilTool::NcaSparseStorage> pStorage) NN_NOEXCEPT
    {
        m_pSparseStorage = pStorage;
    }

    std::shared_ptr<utilTool::NcaSparseStorage> GetSparseStorage() const NN_NOEXCEPT
    {
        return m_pSparseStorage;
    }

    bool HasSparseStorage() const NN_NOEXCEPT
    {
        return m_pSparseStorage != nullptr;
    }

    void SetStorageSparsified(bool isSparsified) NN_NOEXCEPT
    {
        m_IsStorageSparsified = isSparsified;
    }

    bool IsStorageSparsified() const NN_NOEXCEPT
    {
        return m_IsStorageSparsified;
    }

private:
    utilTool::SparseStorageBuilder* const m_pBuilder;
    NcaFsHeaderReader* const m_pHeaderReader;
    const bool m_IsModeEmpty;
    const int m_FsIndex;
    const int64_t m_PhysicalOffset;
    std::shared_ptr<utilTool::NcaSparseStorage> m_pSparseStorage;
    bool m_IsStorageSparsified;
};
#endif

// NcaFileSystemDriver の最下層のストレージを扱うクラス
//
// 一部のストレージが SubStorage を受け取らないことと、SubStorage のコピー（コンストラクタ・代入演算子）が
// メンバの shared_ptr をコピーしないため、少し複雑な使用状況になっている。
class NcaFileSystemDriver::BaseStorage
{
public:
    BaseStorage() NN_NOEXCEPT
        : m_pStorage()
        , m_SubStorage()
#if !defined(NN_SDK_BUILD_RELEASE)
        , m_Status(Status::Invalid)
#endif
        , m_StorageOffset(0)
    {
        m_AesCtrUpperIv.value = 0;
    }

    explicit BaseStorage(const fs::SubStorage& storage) NN_NOEXCEPT
        : m_pStorage()
        , m_SubStorage(storage)
#if !defined(NN_SDK_BUILD_RELEASE)
        , m_Status(Status::SetStorage)
#endif
        , m_StorageOffset(0)
    {
        m_AesCtrUpperIv.value = 0;
    }

    template< typename T >
    BaseStorage(T pStorage, int64_t offset, int64_t size) NN_NOEXCEPT
        : m_pStorage()
        , m_SubStorage(pStorage, offset, size)
#if !defined(NN_SDK_BUILD_RELEASE)
        , m_Status(Status::SetStorage)
#endif
        , m_StorageOffset(0)
    {
        m_AesCtrUpperIv.value = 0;
    }

    // どちらか一方の SetStorage() のみ実行する
    void SetStorage(std::unique_ptr<fs::IStorage>&& storage) NN_NOEXCEPT
    {
#if !defined(NN_SDK_BUILD_RELEASE)
        NN_SDK_REQUIRES(m_Status == Status::Invalid);
        m_Status = Status::SetStorage;
#endif
        m_pStorage = std::move(storage);
    }
    template< typename T >
    void SetStorage(T pStorage, int64_t offset, int64_t size) NN_NOEXCEPT
    {
#if !defined(NN_SDK_BUILD_RELEASE)
        NN_SDK_REQUIRES(m_Status == Status::Invalid);
        m_Status = Status::SetStorage;
#endif
        m_SubStorage = fs::SubStorage(pStorage, offset, size);
    }

    // こちらを実行するときは Get 系を呼び出さない
    std::unique_ptr<fs::IStorage> MakeStorage() NN_NOEXCEPT
    {
#if !defined(NN_SDK_BUILD_RELEASE)
        NN_SDK_REQUIRES(m_Status == Status::SetStorage);
        m_Status = Status::Invalid;
#endif
        if( m_pStorage != nullptr )
        {
            return std::move(m_pStorage);
        }
        return std::unique_ptr<fs::IStorage>(new fs::SubStorage(m_SubStorage));
    }

    // こちらを実行するときは Make 系を呼び出さない
    std::unique_ptr<fs::IStorage> GetStorage() NN_NOEXCEPT
    {
#if !defined(NN_SDK_BUILD_RELEASE)
        NN_SDK_REQUIRES(m_Status != Status::Invalid);
        m_Status = Status::Invalid;
#endif
        return std::move(m_pStorage);
    }
    fs::SubStorage GetSubStorage(int64_t offset, int64_t size) NN_NOEXCEPT
    {
#if !defined(NN_SDK_BUILD_RELEASE)
        NN_SDK_REQUIRES(m_Status != Status::Invalid);
        m_Status = Status::GetStorage;
#endif
        if( m_pStorage != nullptr )
        {
            return fs::SubStorage(m_pStorage.get(), offset, size);
        }
        else
        {
            return fs::SubStorage(&m_SubStorage, offset, size);
        }
    }

    void SetStorageOffset(int64_t offset) NN_NOEXCEPT
    {
        m_StorageOffset = offset;
    }

    int64_t GetStorageOffset() const NN_NOEXCEPT
    {
        return m_StorageOffset;
    }

    void SetAesCtrUpperIv(NcaAesCtrUpperIv upperIv) NN_NOEXCEPT
    {
        m_AesCtrUpperIv = upperIv;
    }

    const NcaAesCtrUpperIv GetAesCtrUpperIv() const NN_NOEXCEPT
    {
        return m_AesCtrUpperIv;
    }

#if !defined(NN_SDK_BUILD_RELEASE)
private:
    // 各関数の呼び出しチェック用のステータス
    enum class Status
    {
        Invalid,
        SetStorage,
        GetStorage
    };
#endif

private:
    std::unique_ptr<fs::IStorage> m_pStorage;
    fs::SubStorage m_SubStorage;
#if !defined(NN_SDK_BUILD_RELEASE)
    Status m_Status;
#endif
    int64_t m_StorageOffset;
    NcaAesCtrUpperIv m_AesCtrUpperIv;
};

}}
