﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <memory>
#include <cstring>

#include <nn/nn_Common.h>
#include <nn/nn_Result.h>

#include <nn/nn_SdkLog.h>
#include <nn/fs/detail/fs_Newable.h>
#include <nn/fs/fs_IStorage.h>
#include <nn/crypto/crypto_AesEncryptor.h>
#include <nn/fssystem/fs_NcaHeader.h>
#include <nn/fssystem/buffers/fs_IBufferManager.h>

namespace nn {
    class MemoryResource;

namespace fs {
    class SubStorage;
}}

namespace nn { namespace fssystem {

    class IndirectStorage;
    class SparseStorage;
    class AesCtrCounterExtendedStorage;

    struct NcaCryptoConfiguration;

    typedef void(*KeyGenerationFunction)(void* pOutKey, size_t outKeySize, const void* pEncryptedKey, size_t encryptedKeySize, int keyTypeValue, const NcaCryptoConfiguration& cryptoConfiguration);
    typedef void(*DecryptAesCtrFunction)(void* pDst, size_t dstSize, int keyTypeValue, const void* pEncryptedKey, size_t encryptedKeySize, const void* pIv, size_t ivSize, const void* pSrc, size_t srcSize);

    // TODO: HW で行う場合に必要なコンテキストを追加
    struct NcaCryptoConfiguration
    {
        static const int Rsa2048KeyModulusSize = 256;
        static const int Rsa2048KeyPublicExponentSize = 3;
        static const int Rsa2048KeyPrivateExponentSize = 256;
        static const int Aes128KeySize = crypto::AesEncryptor128::BlockSize;

        static const int KeyAreaEncryptionKeyIndexCount = 3;
        static const int EncryptionKeyCount = 2;

        static const int KeyGenerationMax = 32;

    public:
        uint8_t headerSign1KeyModulus[Rsa2048KeyModulusSize];
        uint8_t headerSign1KeyPublicExponent[Rsa2048KeyPublicExponentSize];

        uint8_t keyAreaEncryptionKeySource[KeyAreaEncryptionKeyIndexCount][Aes128KeySize];

        uint8_t headerEncryptionKeySource[Aes128KeySize];
        uint8_t headerEncryptedEncryptionKey[EncryptionKeyCount][Aes128KeySize];

        KeyGenerationFunction pGenerateKey;
        DecryptAesCtrFunction pDecryptAesCtr;
        DecryptAesCtrFunction pDecryptAesCtrForExternalKey;

        bool isAvailableRawHeader;
    };
    NN_STATIC_ASSERT(std::is_pod<NcaCryptoConfiguration>::value);

    inline bool IsInvalidKeyTypeValue(int keyTypeValue)
    {
        return keyTypeValue < 0;
    }

    inline int GetKeyTypeValue(uint8_t keyIndex, uint8_t keyGeneration) NN_NOEXCEPT
    {
        const int InvalidKeyTypeValue = -1;
        if (keyIndex >= NcaCryptoConfiguration::KeyAreaEncryptionKeyIndexCount)
        {
            return InvalidKeyTypeValue;
        }
        return NcaCryptoConfiguration::KeyAreaEncryptionKeyIndexCount * keyGeneration + keyIndex;
    };

    const int KeyAreaEncryptionKeyCount = NcaCryptoConfiguration::KeyAreaEncryptionKeyIndexCount * (NcaCryptoConfiguration::KeyGenerationMax - 1) + NcaCryptoConfiguration::KeyAreaEncryptionKeyIndexCount;
    // KeyAreaEncryptionKeyCount = GetKeyTypeValue(NcaCryptoConfiguration::KeyAreaEncryptionKeyIndexCount - 1, NcaCryptoConfiguration::KeyGenerationMax - 1) + 1;

    // 鍵の識別子（永続化されない）
    enum class KeyType : int
    {
        // NcaKey は GetKeyTypeValue() の値
        NcaHeaderKey = KeyAreaEncryptionKeyCount,
        NcaExternalKey,
        SaveDataDeviceUniqueMac,
        SaveDataSeedUniqueMac,
    };

    class NcaReader : public nn::fs::detail::Newable
    {
        NN_DISALLOW_COPY(NcaReader);

    public:
        NcaReader() NN_NOEXCEPT;
        ~NcaReader() NN_NOEXCEPT;

        /*
            @brief NCA フォーマットを構成している IStorage を引数として初期化します。

            @pre pBaseStorage は有効な NCA フォーマットの IStorage を指している。

            @details pBaseStorage はインスタンスが破棄されるまで参照されます。
        */
        Result Initialize(fs::IStorage* pBaseStorage, const NcaCryptoConfiguration& cryptoConfiguration) NN_NOEXCEPT;
        Result Initialize(std::shared_ptr<fs::IStorage> pBaseStorage, const NcaCryptoConfiguration& cryptoConfiguration) NN_NOEXCEPT;

        fs::IStorage* GetBodyStorage() NN_NOEXCEPT;
        uint32_t GetSignature() const NN_NOEXCEPT;
        NcaHeader::DistributionType GetDistributionType() const NN_NOEXCEPT;
        NcaHeader::ContentType GetContentType() const NN_NOEXCEPT;
        uint8_t GetKeyGeneration() const NN_NOEXCEPT;
        uint8_t GetKeyIndex() const NN_NOEXCEPT;
        uint64_t GetContentSize() const NN_NOEXCEPT;
        uint64_t GetProgramId() const NN_NOEXCEPT;
        uint32_t GetContentIndex() const NN_NOEXCEPT;
        uint32_t GetSdkAddonVersion() const NN_NOEXCEPT;
        void GetRightsId(uint8_t* pOutBuffer, size_t outBufferSize) const NN_NOEXCEPT;
        bool HasFsInfo(int index) const NN_NOEXCEPT;
        int GetFsCount() const NN_NOEXCEPT;
        const Hash& GetFsHeaderHash(int index) const NN_NOEXCEPT;
        void GetFsHeaderHash(Hash* outValue, int index) const NN_NOEXCEPT;
        void GetFsInfo(NcaHeader::FsInfo* outValue, int index) const NN_NOEXCEPT;
        uint64_t GetFsOffset(int index) const NN_NOEXCEPT;
        uint64_t GetFsEndOffset(int index) const NN_NOEXCEPT;
        uint64_t GetFsSize(int index) const NN_NOEXCEPT;
        void GetEncryptedKey(void* outValue, size_t size) const NN_NOEXCEPT;
        const void* GetDecryptionKey(int index) const NN_NOEXCEPT;
        bool HasValidInternalKey() const NN_NOEXCEPT; // For Test
        bool HasInternalDecryptionKeyForAesHw() const NN_NOEXCEPT;
        bool IsSwAesPrioritized() const NN_NOEXCEPT;
        void PrioritizeSwAes() NN_NOEXCEPT;
        void SetExternalDecryptionKey(const void* pKey, size_t keySize) NN_NOEXCEPT;
        const void* GetExternalDecryptionKey() const NN_NOEXCEPT;
        bool HasExternalDecryptionKey() const NN_NOEXCEPT;
        void GetRawData(void* outValue, size_t outSize) const NN_NOEXCEPT;
        DecryptAesCtrFunction GetExternalDecryptAesCtrFunction() const NN_NOEXCEPT;
        DecryptAesCtrFunction GetExternalDecryptAesCtrFunctionForExternalKey() const NN_NOEXCEPT;
        NcaHeader::EncryptionType GetEncryptionType() const NN_NOEXCEPT;
        Result ReadHeader(NcaFsHeader* pOutHeader, int index) const NN_NOEXCEPT;

        Result VerifyHeaderSign2(const void* pHeaderSign2Key, size_t headerSign2KeySize) NN_NOEXCEPT;
        Result Verify() NN_NOEXCEPT; // オーサリングツール向け

    private:
        NcaHeader m_Data;
        char m_DataDecryptionKeys[NcaHeader::DecryptionKey_Count][NcaCryptoConfiguration::Aes128KeySize];
        std::shared_ptr<fs::IStorage> m_pSharedBaseStorage;
        std::unique_ptr<fs::IStorage> m_pHeaderStorage;
        fs::IStorage* m_pBodyStorage;
        char m_ExternalDataDecryptionKey[NcaCryptoConfiguration::Aes128KeySize];
        DecryptAesCtrFunction m_pDecryptAesCtr;
        DecryptAesCtrFunction m_pDecryptAesCtrForExternalKey;
        bool m_IsSwAesPrioritized; // AES 鍵が SW/HW 両方利用可能な場合に SW を優先するフラグ
        NcaHeader::EncryptionType m_HeaderEncryptionType;

        friend class NcaFileSystemDriverTest;
    };

    // FS n ヘッダの Reader
    class NcaFsHeaderReader : public fs::detail::Newable
    {
        NN_DISALLOW_COPY(NcaFsHeaderReader);

    public:
        NcaFsHeaderReader() NN_NOEXCEPT;

        Result Initialize(const NcaReader& reader, int index) NN_NOEXCEPT;
        bool IsInitialized() const NN_NOEXCEPT
        {
            return 0 <= m_FsIndex;
        }

        NcaFsHeader& GetData() NN_NOEXCEPT
        {
            return m_Data;
        }
        const NcaFsHeader& GetData() const NN_NOEXCEPT
        {
            return m_Data;
        }
        void GetRawData(void* pOutValue, size_t outSize) const NN_NOEXCEPT;

        NcaFsHeader::HashData& GetHashData() NN_NOEXCEPT;
        const NcaFsHeader::HashData& GetHashData() const NN_NOEXCEPT;
        uint16_t GetVersion() const NN_NOEXCEPT;
        int GetFsIndex() const NN_NOEXCEPT;
        NcaFsHeader::FsType GetFsType() const NN_NOEXCEPT;
        NcaFsHeader::HashType GetHashType() const NN_NOEXCEPT;
        NcaFsHeader::EncryptionType GetEncryptionType() const NN_NOEXCEPT;

        NcaPatchInfo& GetPatchInfo() NN_NOEXCEPT;
        const NcaPatchInfo& GetPatchInfo() const NN_NOEXCEPT;
        const NcaAesCtrUpperIv GetAesCtrUpperIv() const NN_NOEXCEPT;

        bool ExistsSparseLayer() const NN_NOEXCEPT;
        NcaSparseInfo& GetSparseInfo() NN_NOEXCEPT;
        const NcaSparseInfo& GetSparseInfo() const NN_NOEXCEPT;

        // オーサリングツール向け
        Result Verify(NcaHeader::ContentType contentType) const NN_NOEXCEPT;

    private:
        NcaFsHeader m_Data;
        int m_FsIndex;

        friend class NcaFsHeaderReaderTest;
    };

    class NcaFileSystemDriver : public nn::fs::detail::Newable
    {
        NN_DISALLOW_COPY(NcaFileSystemDriver);

    public:
        class StorageOption;
        class StorageOptionWithHeaderReader;
#if defined(NN_BUILD_CONFIG_OS_WIN)
        class SparseParam;
#endif

    public:
        static Result SetupFsHeaderReader(NcaFsHeaderReader* pOutHeaderReader, const NcaReader& ncaReader, int fsIndex) NN_NOEXCEPT;

    public:
        NcaFileSystemDriver(
            std::shared_ptr<NcaReader> pNcaReader,
            MemoryResource* pAllocator,
            IBufferManager* pBufferManager
        ) NN_NOEXCEPT;

        NcaFileSystemDriver(
            std::shared_ptr<NcaReader> pOriginalNcaReader,
            std::shared_ptr<NcaReader> pCurrentNcaReader,
            MemoryResource* pAllocator,
            IBufferManager* pBufferManager
        ) NN_NOEXCEPT;

        // FS n のデータ領域へ直接アクセスする IStorage を取得します。
        Result OpenRawStorage(std::shared_ptr<fs::IStorage>* pOutValue, int fsIndex) NN_NOEXCEPT;

        // FS n のデータ領域への IStorage を取得します。
        Result OpenStorage(std::shared_ptr<fs::IStorage>* pOutValue, int fsIndex) NN_NOEXCEPT
        {
            NcaFsHeaderReader fsHeaderReader;
            return OpenStorage(pOutValue, &fsHeaderReader, fsIndex);
        }
        Result OpenStorage(std::shared_ptr<fs::IStorage>* pOutValue, NcaFsHeaderReader* pOutHeaderReader, int fsIndex) NN_NOEXCEPT;
        Result OpenStorage(std::shared_ptr<fs::IStorage>* pOutValue, StorageOption* pOption) NN_NOEXCEPT;

        // 検証レイヤを除いた FS n のデータ領域への IStorage を取得します。
        Result OpenDecryptableStorage(std::shared_ptr<fs::IStorage>* pOutValue, StorageOption* pOption, bool isIndirectStorageNeeded) NN_NOEXCEPT;

#if defined(NN_BUILD_CONFIG_OS_WIN)
        // nca のスパース化を行います。
        Result SparsifyStorage(SparseParam* pParam) NN_NOEXCEPT;
#endif

    private:
        class BaseStorage;

        Result CreateBaseStorage(BaseStorage* pOutValue, StorageOption* pOption) NN_NOEXCEPT;

        Result CreateDecryptableStorage(std::unique_ptr<fs::IStorage>* pOutStorage, StorageOption* pOption, BaseStorage* pBaseStorage) NN_NOEXCEPT;
        Result CreateAesXtsStorage(std::unique_ptr<fs::IStorage>* pOutStorage, BaseStorage* pBaseStorage) NN_NOEXCEPT;
        Result CreateAesCtrStorage(std::unique_ptr<fs::IStorage>* pOutStorage, BaseStorage* pBaseStorage) NN_NOEXCEPT;
        Result CreateAesCtrExStorage(std::unique_ptr<fs::IStorage>* pOutStorage, StorageOption* pOption, BaseStorage* pBaseStorage) NN_NOEXCEPT;

        Result CreateIndirectStorage(std::unique_ptr<fs::IStorage>* pOutStorage, StorageOption* pOption, std::unique_ptr<fs::IStorage> pBaseStorage) NN_NOEXCEPT;

        Result CreateVerificationStorage(std::unique_ptr<fs::IStorage>* pOutStorage, std::unique_ptr<fs::IStorage> pBaseStorage, NcaFsHeaderReader* pHeaderReader) NN_NOEXCEPT;
        Result CreateSha256Storage(std::unique_ptr<fs::IStorage>* pOutStorage, std::unique_ptr<fs::IStorage> pBaseStorage, NcaFsHeaderReader* pHeaderReader) NN_NOEXCEPT;
        Result CreateIntegrityVerificationStorage(std::unique_ptr<fs::IStorage>* pOutStorage, std::unique_ptr<fs::IStorage> pBaseStorage, NcaFsHeaderReader* pHeaderReader) NN_NOEXCEPT;

#if defined(NN_BUILD_CONFIG_OS_WIN)
        Result SparsifyBasicStorage(SparseParam* pParam) NN_NOEXCEPT;
        Result SparsifyEmptyStorage(SparseParam* pParam) NN_NOEXCEPT;
#endif

    private:
        std::shared_ptr<NcaReader> m_pReader;
        std::shared_ptr<NcaReader> m_pOriginalReader;
        nn::MemoryResource* const m_pAllocator;
        nn::fssystem::IBufferManager* const m_pBufferManager;

        friend class NcaFileSystemDriverTest;
    };


}}

