﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <type_traits>

#include <nn/nn_Assert.h>
#include <nn/nn_Common.h>
#include <nn/util/util_BinTypes.h>

namespace nn { namespace fssystem {

    struct DeltaHeader
    {
        static const int Size = 64;

        enum class Signature : uint32_t
        {
            V0 = NN_UTIL_CREATE_SIGNATURE_4('N', 'D', 'V', '0'),
        };

        uint32_t signature;
        int32_t _reserved;
        int64_t sizeSource;
        int64_t sizeDestination;
        int64_t offsetBody;
        int64_t sizeBody;
    };

    NN_STATIC_ASSERT(std::is_pod<DeltaHeader>::value);
    NN_STATIC_ASSERT(sizeof(DeltaHeader) <= DeltaHeader::Size);

    enum class DeltaCommandType
    {
        Write,
    };

    namespace DeltaCommand
    {
        const int CommandTypeShift = 6;
        const int CommandTypeMask = 0x03;

        inline DeltaCommandType GetCommandType(unsigned char data) NN_NOEXCEPT
        {
            return static_cast<DeltaCommandType>((data >> CommandTypeShift) & CommandTypeMask);
        }
    }

    namespace DeltaCommandWrite
    {
        const int WriteSizeShift = 3;
        const int WriteSizeMask = 0x07;
        const int WriteOffsetShift = 0;
        const int WriteOffsetMask = 0x07;

        inline int GetValue(int sizeOfDataSize, int sizeOfDataOffset) NN_NOEXCEPT
        {
            NN_SDK_ASSERT_MINMAX(sizeOfDataSize, 1, 8);
            NN_SDK_ASSERT_MINMAX(sizeOfDataOffset, 1, 8);
            return (static_cast<int>(DeltaCommandType::Write) << DeltaCommand::CommandTypeShift)
                | ((sizeOfDataSize - 1) << WriteSizeShift)
                | ((sizeOfDataOffset - 1) << WriteOffsetShift)
                ;
        }

        inline int GetCommandParameterSize(int64_t data) NN_NOEXCEPT
        {
            if (data == 0)
            {
                return 1;
            }
            else
            {
                int sizeParameter = 0;
                while (data > 0)
                {
                    data /= 256;
                    ++sizeParameter;
                }
                return sizeParameter;
            }
        }

        inline void SetData(char* pOutBuffer, int64_t data, int size) NN_NOEXCEPT
        {
            NN_SDK_ASSERT_NOT_NULL(pOutBuffer);
            NN_SDK_ASSERT_MINMAX(size, 1, 8);
            for (int i = 0; i < size; i++)
            {
                pOutBuffer[i] = (data & 0xFF);
                data = (data >> 8);
            }
        }

        inline void Create(char* pBuffer, size_t sizeBuffer, int64_t offsetValue, int64_t sizeValue) NN_NOEXCEPT
        {
            NN_SDK_ASSERT_NOT_NULL(pBuffer);

            auto sizeCommandOffset = GetCommandParameterSize(offsetValue);
            auto sizeCommandSize = GetCommandParameterSize(sizeValue);
            NN_SDK_ASSERT_GREATER_EQUAL(sizeBuffer, static_cast<size_t>(1 + sizeCommandOffset + sizeCommandSize));
            NN_UNUSED(sizeBuffer);

            pBuffer[0] = static_cast<char>(GetValue(sizeCommandSize, sizeCommandOffset));
            SetData(pBuffer + 1, sizeValue, sizeCommandSize);
            SetData(pBuffer + 1 + sizeCommandSize, offsetValue, sizeCommandOffset);
        }

        inline int GetCommandSize(int64_t offsetValue, int64_t sizeValue) NN_NOEXCEPT
        {
            return 1 + DeltaCommandWrite::GetCommandParameterSize(offsetValue) + DeltaCommandWrite::GetCommandParameterSize(sizeValue);
        }

        inline int GetSizeOfDataSize(unsigned char data) NN_NOEXCEPT
        {
            return ((data >> WriteSizeShift) & WriteSizeMask) + 1;
        }

        inline int GetSizeOfDataOffset(unsigned char data) NN_NOEXCEPT
        {
            return ((data >> WriteOffsetShift) & WriteOffsetMask) + 1;
        }
    }

}}
