﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Common.h>
#include <nn/nn_Result.h>
#include <nn/nn_SdkAssert.h>

#include <nn/fssystem/fs_DbmRomTypes.h>


namespace nn { namespace fssystem {

// エントリーを格納するアドレスをアライメントします。
inline uint32_t AlignROMAddress(uint32_t x)
{
    // ARM は 4 バイト単位でのアクセスでないと読み込めないのでアライメントします。
    return ((x + 3) & ~0x3);
}

/*!
    :private

    @brief 可変長キーバリューストレージテンプレートクラスです。

           2種類のキー(固定サイズのキーと可変サイズのキー)とバリューの組合わせを記憶します。

           キーバリューエレメントの直後に可変サイズのキ－が配置されます。
*/
template <
    typename BucketStorage_,
    typename EntryStorage_,
    typename Key_,
    typename Value_,
    int MAX_EXTRA_SIZE_
>
class KeyValueRomStorageTemplate
{
    //!< テストのためTestSuiteに対して実装を公開します
    friend class KeyValueRomStorageTemplateTestSuite;

public:
    typedef BucketStorage_ BucketStorage;
    typedef EntryStorage_ EntryStorage;
    typedef Key_ Key;
    typedef Value_ Value;
    typedef uint32_t Position;                  //!< キーバリューストレージのオフセット
    typedef uint32_t IndexBucket;               //!< バケットストレージのインデックス

    /*!
        @brief キーと値の列挙に使用する探索位置指示構造体です。
    */
    struct FindIndex
    {
        IndexBucket bucket;
        Position    pos;
    };
    NN_STATIC_ASSERT(std::is_pod<FindIndex>::value);

private:
    /*!
        :private

        @brief キーバリューストレージのエントリーです。
    */
    struct StorageElement
    {
        Key       key;                     //!< キー
        Value     value;                   //!< 実データへのインデックス
        Position  next;                    //!< 同じハッシュキーを共有する次のデータへのオフセット
        uint32_t       size;                    //!< 拡張キーのサイズ
    };
    NN_STATIC_ASSERT(std::is_pod<StorageElement>::value);

    //!< アドレスとして無効な値をフリーエントリーとして扱います。
    static const Position KVSTORAGE_FREEENTRY = 0xFFFFFFFF;

private:
//    util::Int64<int64_t> m_OffsetBucket;                    //!< バケットに割り当てる領域のオフセット
    int64_t m_OffsetBucket;                    //!< バケットに割り当てる領域のオフセット
    uint32_t m_CountBucket;                     //!< バケットに割り当てる領域のサイズ(要素数)
    BucketStorage* m_pBufBucket;           //!< バケットストレージ

//    util::Int64<int64_t> m_OffsetKeyValue;                  //!< キーバリューに割り当てる領域のオフセット
    int64_t m_OffsetKeyValue;                  //!< キーバリューに割り当てる領域のオフセット
    uint32_t m_SizeKeyValue;                    //!< キーバリューに割り当てる領域のサイズ(要素数)
    EntryStorage* m_pBufKeyValue;          //!< キーバリューストレージ

    uint32_t m_TotalEntrySize;                  //!< 使用領域サイズ
    uint32_t m_EntryCount;                      //!< エントリー数

public:

    /*!
        @brief エントリーに必要なサイズを求めます。

        @param[in] extraSize 可変長キーの長さ

        @return エントリーに必要なサイズ
    */
    static uint32_t QueryEntrySize(uint32_t extraSize)
    {
        return AlignROMAddress(sizeof(StorageElement) + extraSize);
    }

    /*!
        @brief バケット数からバケットストレージのサイズを求めます。

        @param[in] countBucket バケット数

        @return 必要なストレージのバイトサイズ
    */
    static uint32_t QueryBucketStorageSize(uint32_t countBucket)
    {
        return countBucket * sizeof(Position);
    }

    /*!
        @brief バケットストレージのサイズからからバケット数を求めます。

        @param[in] countBucket バケット数

        @return バケット数
    */
    static uint32_t QueryBucketCount(uint32_t size)
    {
        return size / sizeof(Position);
    }

    /*!
        @brief 格納可能なキー数からキーバリューストレージのサイズを求めます。

        @param[in] countKeyValue キーバリュー数

        @return 必要なストレージのバイトサイズ
    */
    static uint32_t QueryKeyValueStorageSize(uint32_t countKeyValue)
    {
        return countKeyValue * sizeof(StorageElement);
    }

    /*!
        @brief バケットとキーバリューストレージを初期化(フォーマット)します。

               IStorage を初期化します。使用する前に一度呼び出します。

        @param[in] pBucket バケットストレージ
        @param[in] offsetBucket バケットストレージのオフセット(バイト)
        @param[in] countBucket バケット数(要素数)
        @param[in] pKeyValue キーバリューストレージ
        @param[in] offsetKeyValue キーバリューストレージのオフセット(バイト)
        @param[in] sizeStorage ストレージのサイズ(バイト)

        @return 関数の処理結果を返します。
    */
    static Result Format(BucketStorage* pBucket, int64_t offsetBucket, uint32_t countBucket, EntryStorage* pKeyValue, int64_t offsetKeyValue, uint32_t sizeKeyValue)
    {
        NN_SDK_ASSERT_NOT_NULL(pBucket);
        NN_SDK_ASSERT_NOT_NULL(pKeyValue);
        NN_SDK_ASSERT(sizeKeyValue >= 0);

        NN_UNUSED(pKeyValue);
        NN_UNUSED(offsetKeyValue);
        NN_UNUSED(sizeKeyValue);

        // 割り当て領域を初期化します。
        const int64_t pos = KVSTORAGE_FREEENTRY;
        for (uint32_t i = 0; i < countBucket; i++)
        {
            NN_RESULT_DO(pBucket->Write(offsetBucket + i * sizeof(Position), &pos, sizeof(Position)));
        }

        return ResultSuccess();
    }

    /*!
        @brief コンストラクタです。
    */
    KeyValueRomStorageTemplate()
        : m_OffsetBucket(0),
          m_CountBucket(0),
          m_pBufBucket(NULL),
          m_OffsetKeyValue(0),
          m_SizeKeyValue(0),
          m_pBufKeyValue(NULL),
          m_TotalEntrySize(0),
          m_EntryCount(0)
    {
    }

    /*!
        @brief ストレージをマウントします。

        @param[in] pBucket バケットストレージ
        @param[in] offsetBucket バケットストレージのオフセット(バイト)
        @param[in] countBucket バケット数(要素数)
        @param[in] pKeyValue キーバリューストレージ
        @param[in] offsetKeyValue ストレージのオフセット(バイト)
        @param[in] sizeStorage ストレージのサイズ(バイト)

        @return 関数の処理結果を返します。
    */
    Result Initialize(BucketStorage* pBucket, int64_t offsetBucket, uint32_t countBucket, EntryStorage* pKeyValue, int64_t offsetKeyValue, uint32_t sizeKeyValue)
    {
        NN_SDK_ASSERT_NOT_NULL(pBucket);
        NN_SDK_ASSERT_NOT_NULL(pKeyValue);
        NN_SDK_ASSERT(countBucket > 0);

        m_pBufBucket = pBucket;
        m_OffsetBucket = offsetBucket;
        m_CountBucket = countBucket;

        m_pBufKeyValue = pKeyValue;
        m_OffsetKeyValue = offsetKeyValue;
        m_SizeKeyValue = sizeKeyValue;

        return ResultSuccess();
    }

    /*!
        @brief ストレージをアンマウントします。
    */
    void Finalize()
    {
        m_pBufBucket = NULL;
        m_OffsetBucket = 0;
        m_CountBucket = 0;

        m_pBufKeyValue = NULL;
        m_OffsetKeyValue = 0;
        m_SizeKeyValue = 0;
    }

    /*!
        @brief キーと値を追加します。

               同名のキーが既に存在する場合、ResultDbmAlreadyExists を返します。

        @param[in] key キー
        @param[in] hashKey キーのハッシュ値
        @param[in] pExtraKey 可変長キー
        @param[in] extraSize 可変長キーの長さ
        @param[in] value 値

        @return 関数の処理結果を返します。
    */
    Result Add(const Key& key, uint32_t hashKey, const void* pExtraKey, size_t extraSize, const Value& value)
    {
        NN_SDK_ASSERT_NOT_NULL(pExtraKey);
        NN_SDK_ASSERT(extraSize <= MAX_EXTRA_SIZE_);
        Position posAdded;
        return AddInternal(&posAdded, key, hashKey, pExtraKey, extraSize, value);
    }

    /*!
        @brief キーに対する値を取得します。

        @param[out] pValue 値
        @param[in] key キー
        @param[in] hashKey キーのハッシュ値
        @param[in] pExtraKey 可変長キー
        @param[in] extraSize 可変長キーの長さ

        @return 関数の処理結果を返します。
    */
    Result Get(Value* pValue, const Key& key, uint32_t hashKey, const void* pExtraKey, size_t extraSize) const
    {
        NN_SDK_ASSERT_NOT_NULL(pExtraKey);
        NN_SDK_ASSERT(extraSize <= MAX_EXTRA_SIZE_);
        Position pos;
        return GetInternal(&pos, pValue, key, hashKey, pExtraKey, extraSize);
    }

    /*!
        @brief キーと値の列挙を開始します。

        @param[in,out] pfi イテレータ
    */
    void FindOpen(FindIndex* pfi) const
    {
        NN_SDK_ASSERT_NOT_NULL(pfi);

        // イテレータを初期化します。
        pfi->bucket = static_cast<IndexBucket>(-1);
        pfi->pos = KVSTORAGE_FREEENTRY;
    }

    /*!
        @brief キーと値を列挙します。

        @param[out] pKey キー
        @param[out] pValue 値
        @param[in,out] pfi イテレータ

        @return 関数の処理結果を返します。

                列挙が終わったら、ResultDbmFindKeyFinished() を返します。
    */
    Result FindNext(Key* pKey, Value* pValue, FindIndex* pfi) const
    {
        StorageElement storeElement;

        NN_SDK_ASSERT_NOT_NULL(pKey);
        NN_SDK_ASSERT_NOT_NULL(pValue);
        NN_SDK_ASSERT_NOT_NULL(pfi);

        IndexBucket indexBucket = pfi->bucket;
        if ((indexBucket >= m_CountBucket) && (indexBucket != static_cast<IndexBucket>(-1)))
        {
            // キーの探索は終了しています。
            NN_SDK_ASSERT(indexBucket == m_CountBucket);
            return ResultDbmFindKeyFinished();
        }

        do
        {
            // KeyValueリストを探索します。
            if (pfi->pos != KVSTORAGE_FREEENTRY)
            {
                // このエントリーを返します。
                NN_RESULT_DO(ReadKeyValue(&storeElement, NULL, NULL, pfi->pos));

                NN_SDK_ASSERT((storeElement.next == KVSTORAGE_FREEENTRY) || (storeElement.next < m_SizeKeyValue));

                // 次に進めておきます。
                // リスト終端時は KVSTORAGE_FREEENTRY が入ります。
                pfi->pos = storeElement.next;

                // キーと値を返します。
                *pKey = storeElement.key;
                *pValue = storeElement.value;

                return ResultSuccess();
            }

            // バケットリストを探索します。
            while (true)
            {
                // 使用しているバケットリストを読み込みます。
                indexBucket++;
                if (indexBucket == m_CountBucket)
                {
                    // リスト探索終了を示します。
                    pfi->bucket = indexBucket;
                    pfi->pos = KVSTORAGE_FREEENTRY;

                    return ResultDbmFindKeyFinished();
                }

                Position pos;
                NN_RESULT_DO(ReadBucket(&pos, indexBucket));
                NN_SDK_ASSERT((pos == KVSTORAGE_FREEENTRY) || (pos < m_SizeKeyValue));

                if (pos != KVSTORAGE_FREEENTRY)
                {
                    // バケットの先頭を捕まえました。
                    pfi->bucket = indexBucket;
                    pfi->pos = pos;
                    break;
                }
            }

        } while (true);
    }

    //!< 使用エントリー領域のサイズを取得します。
    inline uint32_t GetTotalEntrySize() const
    {
        return m_TotalEntrySize;
    }

    //!< フリー領域のサイズを取得します。
    inline uint32_t GetFreeSize() const
    {
        return (m_SizeKeyValue - m_TotalEntrySize);
    }

    //!< エントリー数を取得します。
    inline uint32_t GetEntryCount() const
    {
        return m_EntryCount;
    }

protected:

    /*!
        @brief キーと値を追加します。

               同名のキーが既に存在する場合、値を更新します。

        @param[out] 追加されたストレージ位置
        @param[in] key キー
        @param[in] hashKey キーのハッシュ値
        @param[in] pExtraKey 可変長キー
        @param[in] extraSize 可変長キーの長さ
        @param[in] value 値

        @return 関数の処理結果を返します。
    */
    Result AddInternal(
               Position* pPosition,
               const Key& key,
               uint32_t hashKey,
               const void* pExtraKey,
               size_t extraSize,
               const Value& value
           )
    {
        Position pos;
        Position posPrevious;
        StorageElement storeElement;

        NN_SDK_ASSERT_NOT_NULL(pPosition);
        NN_SDK_ASSERT(m_CountBucket > 0);
        NN_SDK_ASSERT(m_SizeKeyValue >= 0);

        // キーが登録済みかどうかチェックします。
        Result result;
        result = FindInternal(
                     &pos,
                     &posPrevious,
                     &storeElement,
                     key,
                     hashKey,
                     pExtraKey,
                     extraSize
                 );
        if (! result.IsFailure())
        {
            // すでにキーが登録されています。
            return ResultDbmAlreadyExists();
        }
        else if (! ResultDbmKeyNotFound::Includes(result))
        {
            // キーが見つからない、以外のコードは
            // ファイルシステムで発生しています。
            return result;
        }

        // キーが見つからない場合、新規にキーを追加します。

        // フリーリストから要素を一つ得てリンクリストに繋ぎます
        NN_RESULT_DO(AllocateEntry(&pos, extraSize));

        Position posNext;
        NN_RESULT_DO(LinkEntry(&posNext, pos, hashKey));

        storeElement.key = key;
        storeElement.value = value;
        storeElement.next = posNext;
        storeElement.size = static_cast<uint32_t>(extraSize);
        *pPosition = pos;

        // エントリーに書き込みます。
        NN_RESULT_DO(WriteKeyValue(&storeElement, pos, pExtraKey, extraSize, nn::fs::WriteOption()));

        // エントリー数を更新します。
        m_EntryCount++;

        return ResultSuccess();
    }

    /*!
        @brief キーに対する値を取得します。

        @param[out] pPosition 格納されているストレージ位置
        @param[out] pValue 値
        @param[in] key キー
        @param[in] hashKey キーのハッシュ値
        @param[in] pExtraKey 可変長キー
        @param[in] extraSize 可変長キーの長さ

        @return 関数の処理結果を返します。
    */
    Result GetInternal(
               Position* pPosition,
               Value* pValue,
               const Key& key,
               uint32_t hashKey,
               const void* pExtraKey,
               size_t extraSize
           ) const
    {
        Position pos;
        Position posPrevious;
        StorageElement storeElement;

        NN_SDK_ASSERT_NOT_NULL(pPosition);
        NN_SDK_ASSERT_NOT_NULL(pValue);

        NN_RESULT_DO(FindInternal(
                     &pos,
                     &posPrevious,
                     &storeElement,
                     key,
                     hashKey,
                     pExtraKey,
                     extraSize
                 ));

        *pPosition = pos;
        *pValue = storeElement.value;

        return ResultSuccess();
    }

    /*!
        @brief ストレージ位置を用いてキーバリューを取得します。

        @param[out] pKey キー
        @param[out] pValue 値
        @param[out] pExtraKey 可変長キー
        @param[out] pExtraSize 可変長キーの長さ
        @param[in] pos ストレージ位置

        @return 関数の処理結果を返します。
    */
    Result GetByPosition(
               Key* pKey,
               Value* pValue,
               void* pExtraKey,
               size_t* pExtraSize,
               Position pos
           ) const
    {
        NN_SDK_ASSERT_NOT_NULL(pKey);
        NN_SDK_ASSERT_NOT_NULL(pValue);

        StorageElement storeElement;
        NN_RESULT_DO(ReadKeyValue(&storeElement, pExtraKey, pExtraSize, pos));

        *pKey = storeElement.key;
        *pValue = storeElement.value;

        return ResultSuccess();
    }

    /*!
        @brief ストレージ位置を用いてバリューを更新します。

        @param[in] pos ストレージ位置
        @param[in] value 値
        @param[in] option 書き込みオプション

        @return 関数の処理結果を返します。
    */
    Result SetByPosition(Position pos, const Value &value, nn::fs::WriteOption option) const
    {
        StorageElement storeElement;
        NN_RESULT_DO(ReadKeyValue(&storeElement, NULL, NULL, pos));
        storeElement.value = value;
        return WriteKeyValue(&storeElement, pos, NULL, 0, option);
    }

private:

    //!< ハッシュ値からバケット位置を取得します。
    IndexBucket HashToBucket(uint32_t hashKey) const
    {
        return hashKey % m_CountBucket;
    }

    /*!
        :private

        @brief キーを指定し、対応する値セットとストレージ位置を取得します。(内部用)

        @param[out] pPosition 得られたストレージ位置
        @param[out] pPreviousPosition 得られたストレージ位置を指し示すストレージ位置
        @param[out] pElement 得られた値
        @param[in] key キー
        @param[in] hashKey キーのハッシュ値
        @param[in] pExtraKey 可変長キー
        @param[in] extraSize 可変長キーの長さ

        @return 関数の処理結果を返します。
    */
    Result FindInternal(
               Position* pPosition,
               Position* pPreviousPosition,
               StorageElement* pStoreElement,
               const Key& key,
               uint32_t hashKey,
               const void* pExtraKey,
               size_t extraSize
           ) const
    {
        NN_SDK_ASSERT_NOT_NULL(pPosition);
        NN_SDK_ASSERT_NOT_NULL(pPreviousPosition);
        NN_SDK_ASSERT_NOT_NULL(pStoreElement);
        NN_SDK_ASSERT(m_CountBucket > 0);
        NN_SDK_ASSERT(m_SizeKeyValue >= 0);

        *pPosition = 0;
        *pPreviousPosition = 0;

        // バケット位置を取得します。
        IndexBucket indexBucket = HashToBucket(hashKey);

        // バケットから先頭位置を取得します。
        Position posTop;
        NN_RESULT_DO(ReadBucket(&posTop, indexBucket));
        NN_SDK_ASSERT((posTop == KVSTORAGE_FREEENTRY) || (posTop < m_SizeKeyValue));

        // バケットが空だった場合
        if (posTop == KVSTORAGE_FREEENTRY)
        {
            // バケットには何も入っていません。
            return ResultDbmKeyNotFound();
        }

        uint8_t buf[MAX_EXTRA_SIZE_];

        // アローケーションリンクを辿り、エントリーを探します。
        Position pos = posTop;
        while (NN_STATIC_CONDITION(true))
        {
            // 一つエントリーを得て比較します。
            size_t currExtraSize = 0;
            NN_RESULT_DO(ReadKeyValue(pStoreElement, buf, &currExtraSize, pos));

            // キー同士が一致すれば検索終了です。
            if (key.IsEqual(pStoreElement->key, pExtraKey, extraSize, buf, currExtraSize))
            {
                *pPosition = pos;
                return ResultSuccess();
            }

            // 次の位置を探します。
            *pPreviousPosition = pos;
            pos = pStoreElement->next;
            if (KVSTORAGE_FREEENTRY == pos)
            {
                // 一致したキーが見つかりません。
                return ResultDbmKeyNotFound();
            }
        }

        // ここには到達しないはず。
    }

    /*!
        :private

        @brief フリーエントリーからエントリーを確保します。

        @param[out] pEntry 得られたエントリー位置
        @param[in] 可変長キーのサイズ

        @return 関数の処理結果を返します。
    */
    Result AllocateEntry(Position* pEntry, size_t extraSize)
    {
        NN_SDK_ASSERT_NOT_NULL(pEntry);

        // フリー領域に十分な領域が存在しているかチェックします。
        if (m_TotalEntrySize + sizeof(StorageElement) + extraSize > m_SizeKeyValue)
        {
            // 割り当て可能領域はありません。
            return ResultDbmKeyFull();
        }

         // 得られたエントリーを設定します。
        *pEntry = m_TotalEntrySize;

        // フリー領域からエントリー + サイズ分確保します。
        m_TotalEntrySize = AlignROMAddress(m_TotalEntrySize + sizeof(StorageElement) + static_cast<uint32_t>(extraSize));

        return ResultSuccess();
    }

    /*!
        :private

        @brief エントリーに繋ぎます。

        @param[out] pPreviousPosition 以前の先頭エントリー
        @param[in] pos 先頭にするエントリー
        @param[in] hashKey キーのハッシュ値

        @return 関数の処理結果を返します。
    */
    Result LinkEntry(Position* pPrevPosition, Position pos, uint32_t hashKey)
    {
        NN_SDK_ASSERT_NOT_NULL(pPrevPosition);

        // バケット位置を求めます。
        IndexBucket indexBucket = HashToBucket(hashKey);

        // バケットの内容を取得します。
        Position posNext;
        NN_RESULT_DO(ReadBucket(&posNext, indexBucket));
        NN_SDK_ASSERT((posNext == KVSTORAGE_FREEENTRY) || (posNext < m_SizeKeyValue));

        // バケットを更新します。
        NN_RESULT_DO(WriteBucket(pos, indexBucket, nn::fs::WriteOption()));
        *pPrevPosition = posNext;

        return ResultSuccess();
    }

    /*!
        :private

        @brief バケットストレージから読み込みます。

        @param[out] pKvStorePosition バケットに書かれていた値
        @param[in] index バケットインデックス

        @return 関数の処理結果を返します。
    */
    inline Result ReadBucket(Position* pKvStorePosition, IndexBucket index) const
    {
        NN_SDK_ASSERT_NOT_NULL(pKvStorePosition);
        NN_SDK_ASSERT_NOT_NULL(m_pBufBucket);
        NN_SDK_ASSERT(index < m_CountBucket);

        int64_t offset = m_OffsetBucket + index * sizeof(Position);
        return m_pBufBucket->Read(offset, pKvStorePosition, sizeof(Position));
    }

    /*!
        :private

        @brief バケットストレージに書き込みます。

        @param[in] KvStorePosition バケットに書き込む値
        @param[in] index バケットインデックス
        @param[in] option 書き込みオプション

        @return 関数の処理結果を返します。
    */
    inline Result WriteBucket(Position KvStorePosition, IndexBucket index, nn::fs::WriteOption option) const
    {
        NN_UNUSED(option);
        NN_SDK_ASSERT_NOT_NULL(m_pBufBucket);
        NN_SDK_ASSERT(index < m_CountBucket);

        int64_t offset = m_OffsetBucket + index * sizeof(Position);
        return m_pBufBucket->Write(offset, &KvStorePosition, sizeof(Position));
    }

    /*!
        :private

        @brief キーストレージから読み込みます。

        @param[out] pElement キーバリューストレージに書かれていた値
        @param[out] pExtraKey 可変長キーを読み込むバッファ(MAX_EXTRA_SIZE_バイト)
        @param[out] pExtraSize 可変長キーのサイズ
        @param[in] pos ストレージ位置

        @return 関数の処理結果を返します。
    */
    inline Result ReadKeyValue(
                      StorageElement* pElement,
                      void* pExtraKey,
                      size_t* pExtraSize,
                      Position pos
                  ) const
    {
        NN_SDK_ASSERT_NOT_NULL(pElement);
        NN_SDK_ASSERT_NOT_NULL(m_pBufKeyValue);
        NN_SDK_ASSERT(pos < m_SizeKeyValue);

        int64_t offset = m_OffsetKeyValue + pos;
        NN_RESULT_DO(m_pBufKeyValue->Read(offset, pElement, sizeof(StorageElement)));

        if ((pExtraKey != NULL) && (pExtraSize != NULL))
        {
            *pExtraSize = pElement->size;
            if (pElement->size > 0)
            {
                NN_RESULT_DO(m_pBufKeyValue->Read(offset + sizeof(StorageElement), pExtraKey, pElement->size));
            }
        }

        return ResultSuccess();
    }

    /*!
        :private

        @brief キーストレージに書き込みます。

        @param[out] pElement キーバリューストレージに書き込む値
        @param[in] pos ストレージ位置
        @param[in] pExtraKey 可変長キー
        @param[in] extraSize 可変長キーのサイズ
        @param[in] option 書き込みオプション

        @return 関数の処理結果を返します。
    */
    inline Result WriteKeyValue(
                      const StorageElement* pElement,
                      Position pos,
                      const void* pExtraKey,
                      size_t extraSize,
                      nn::fs::WriteOption option
                  ) const
    {
        NN_UNUSED(option);
        NN_SDK_ASSERT_NOT_NULL(pElement);
        NN_SDK_ASSERT_NOT_NULL(m_pBufKeyValue);
        NN_SDK_ASSERT(pos < m_SizeKeyValue);

        int64_t offset = m_OffsetKeyValue + pos;
        NN_RESULT_DO(m_pBufKeyValue->Write(offset, pElement, sizeof(StorageElement)));

        if ((pExtraKey != NULL) && (extraSize > 0))
        {
            NN_RESULT_DO(m_pBufKeyValue->Write(offset + sizeof(StorageElement), pExtraKey, extraSize));
        }

        return ResultSuccess();
    }

};

}}
