﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/fssystem/fs_BucketTree.h>

namespace nn { namespace fssystem { namespace detail {

/**
 * @brief   安全に値を扱うクラスです。
 */
class SafeValue
{
public:
    static int64_t GetInt64(const void* ptr) NN_NOEXCEPT
    {
        // NOTE: ８バイトアライメント対策。最適化を期待
        int64_t value;
        std::memcpy(&value, ptr, sizeof(int64_t));
        return value;
    }

    static int64_t GetInt64(const int64_t* ptr) NN_NOEXCEPT
    {
        return GetInt64(static_cast<const void*>(ptr));
    }

    static int64_t GetInt64(const int64_t& value) NN_NOEXCEPT
    {
        return GetInt64(&value);
    }

    static void SetInt64(void* pOut, const void* pSrc) NN_NOEXCEPT
    {
        std::memcpy(pOut, pSrc, sizeof(int64_t));
    }

    static void SetInt64(void* pOut, const int64_t* pSrc) NN_NOEXCEPT
    {
        return SetInt64(pOut, static_cast<const void*>(pSrc));
    }

    static void SetInt64(void* pOut, const int64_t& value) NN_NOEXCEPT
    {
        return SetInt64(pOut, &value);
    }
};

/**
 * @brief   BucketTree のノードを表します。
 */
template< typename TIterator >
struct BucketTreeNode
{
    typedef BucketTree::NodeHeader Header;

    Header header;
    // NOTE: この構造体はヘッダに続いてオフセットまたはエントリの配列が並んでいる
    //int64_t offset[];
    //    or
    //Entry entry[];

public:
    //! 要素数を取得します。
    int GetCount() const NN_NOEXCEPT
    {
        return this->header.count;
    }

    //! 配列の先頭アドレスを取得します。
    void* GetArray() NN_NOEXCEPT
    {
        return &this->header + 1;
    }

    template< typename T >
    T* GetArray() NN_NOEXCEPT
    {
        return reinterpret_cast<T*>(GetArray());
    }

    //! 配列の先頭アドレスを取得します。
    const void* GetArray() const NN_NOEXCEPT
    {
        return &this->header + 1;
    }

    template< typename T >
    const T* GetArray() const NN_NOEXCEPT
    {
        return reinterpret_cast<const T*>(GetArray());
    }

    //! ノードが管理するオフセットの先頭を取得します。
    int64_t GetBeginOffset() const NN_NOEXCEPT
    {
        return *GetArray<int64_t>();
    }

    //! ノードが管理するオフセットの末尾を取得します。
    int64_t GetEndOffset() const NN_NOEXCEPT
    {
        return this->header.offset;
    }

    //! 先頭のイテレータを取得します。
    TIterator GetBegin() NN_NOEXCEPT
    {
        return TIterator(GetArray<int64_t>());
    }

    //! 末尾のイテレータを取得します。
    TIterator GetEnd() NN_NOEXCEPT
    {
        return TIterator(GetArray<int64_t>()) + this->header.count;
    }

    //! 先頭のイテレータを取得します。
    TIterator GetBegin() const NN_NOEXCEPT
    {
        return TIterator(GetArray<int64_t>());
    }

    //! 末尾のイテレータを取得します。
    TIterator GetEnd() const NN_NOEXCEPT
    {
        return TIterator(GetArray<int64_t>()) + this->header.count;
    }

    //! 先頭のイテレータを取得します。
    TIterator GetBegin(size_t entrySize) NN_NOEXCEPT
    {
        return TIterator(GetArray(), entrySize);
    }

    //! 末尾のイテレータを取得します。
    TIterator GetEnd(size_t entrySize) NN_NOEXCEPT
    {
        return TIterator(GetArray(), entrySize) + this->header.count;
    }

    //! 先頭のイテレータを取得します。
    TIterator GetBegin(size_t entrySize) const NN_NOEXCEPT
    {
        return TIterator(GetArray(), entrySize);
    }

    //! 末尾のイテレータを取得します。
    TIterator GetEnd(size_t entrySize) const NN_NOEXCEPT
    {
        return TIterator(GetArray(), entrySize) + this->header.count;
    }
};

/**
 * @brief   エントリのオフセットを取得します。
 */
inline int64_t GetBucketTreeEntryOffset(
                   int64_t entrySetOffset,
                   size_t entrySize,
                   int entryIndex
               ) NN_NOEXCEPT
{
    return entrySetOffset +
           sizeof(BucketTree::NodeHeader) +
           entryIndex * static_cast<int64_t>(entrySize);
}

/**
 * @brief   エントリのオフセットを取得します。
 */
inline int64_t GetBucketTreeEntryOffset(
                   int entrySetIndex,
                   size_t nodeSize,
                   size_t entrySize,
                   int entryIndex
               ) NN_NOEXCEPT
{
    return GetBucketTreeEntryOffset(
        entrySetIndex * static_cast<int64_t>(nodeSize), entrySize, entryIndex);
}

}}}
