﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/fssystem/fs_BucketTree.h>

namespace nn { namespace fssystem {

/**
 * @brief   BucketTree で扱うデータを構築するクラスです。
 */
class BucketTreeBuilder
{
    NN_DISALLOW_COPY(BucketTreeBuilder);

public:
    //! アロケータです。
    //! @todo 適切な型に変更する
    typedef MemoryResource IAllocator;

public:
    /**
     * @brief   ヘッダ部分のストレージのサイズを取得します。
     *
     * @return  ヘッダ部分のストレージサイズを返します。
     */
    static int64_t QueryHeaderStorageSize() NN_NOEXCEPT
    {
        return BucketTree::QueryHeaderStorageSize();
    }

    /**
     * @brief   ノード部分のストレージのサイズを取得します。
     *
     * @param[in]   nodeSize    １ノードのサイズ
     * @param[in]   entrySize   １エントリのサイズ
     * @param[in]   entryCount  エントリの総数
     *
     * @return  ノード部分のストレージサイズを返します。
     *
     * @pre
     *      - sizeof(int64_t) <= entrySize
     *      - entrySize + sizeof(BucketTree::NodeHeader) <= nodeSize
     *      - BucketTree::NodeSizeMin <= nodeSize && nodeSize <= BucketTree::NodeSizeMax
     *      - nodeSize が２の累乗
     *      - 0 <= entryCount
     */
    static int64_t QueryNodeStorageSize(
                       size_t nodeSize,
                       size_t entrySize,
                       int entryCount
                   ) NN_NOEXCEPT
    {
        return BucketTree::QueryNodeStorageSize(nodeSize, entrySize, entryCount);
    }

    /**
     * @brief   エントリ部分のストレージのサイズを取得します。
     *
     * @param[in]   nodeSize    １ノードのサイズ
     * @param[in]   entrySize   １エントリのサイズ
     * @param[in]   entryCount  エントリの総数
     *
     * @return  エントリ部分のストレージサイズを返します。
     *
     * @pre
     *      - sizeof(int64_t) <= entrySize
     *      - entrySize + sizeof(BucketTree::NodeHeader) <= nodeSize
     *      - BucketTree::NodeSizeMin <= nodeSize && nodeSize <= BucketTree::NodeSizeMax
     *      - nodeSize が２の累乗
     *      - 0 <= entryCount
     */
    static int64_t QueryEntryStorageSize(
                       size_t nodeSize,
                       size_t entrySize,
                       int entryCount
                   ) NN_NOEXCEPT
    {
        return BucketTree::QueryEntryStorageSize(nodeSize, entrySize, entryCount);
    }

public:
    /**
     * @brief   コンストラクタです。
     */
    BucketTreeBuilder() NN_NOEXCEPT;

    /**
     * @brief   デストラクタです。
     */
    ~BucketTreeBuilder() NN_NOEXCEPT
    {
        Finalize();
    }

    /**
     * @brief   初期化します。
     *
     * @param[in]   pAllocator      アロケータのポインタ
     * @param[in]   headerStorage   ヘッダストレージ
     * @param[in]   nodeStorage     ノードストレージ
     * @param[in]   entryStorage    エントリストレージ
     * @param[in]   nodeSize        １ノードのサイズ
     * @param[in]   entrySize       １エントリのサイズ
     * @param[in]   entryCount      エントリの総数
     *
     * @retval  ResultSuccess                   正常に処理が終了しました。
     * @retval  ResultBufferAllocationFailed    メモリの確保に失敗しました。
     * @retval  上記以外                        ストレージの書き込みに失敗しました。
     *
     * @pre
     *      - pAllocator != nullptr
     *      - sizeof(int64_t) <= entrySize
     *      - entrySize + sizeof(BucketTree::NodeHeader) <= nodeSize
     *      - BucketTree::NodeSizeMin <= nodeSize && nodeSize <= BucketTree::NodeSizeMax
     *      - nodeSize が２の累乗
     *      - 0 <= entryCount
     *      - 未初期化
     *
     * この関数では、ストレージの書き込み処理を行います。
     */
    Result Initialize(
               IAllocator* pAllocator,
               fs::SubStorage headerStorage,
               fs::SubStorage nodeStorage,
               fs::SubStorage entryStorage,
               size_t nodeSize,
               size_t entrySize,
               int entryCount
           ) NN_NOEXCEPT;

    /**
     * @brief   終了処理をします。
     */
    void Finalize() NN_NOEXCEPT;

    /**
     * @brief   初期化済みかどうかを取得します。
     *
     * @retval true     初期化済みです。
     * @retval false    初期化していません。
     */
    bool IsInitialized() const NN_NOEXCEPT
    {
        return 0 < m_NodeSize;
    }

    /**
     * @brief   エントリを書き出します。
     *
     * @param[in]   entry   書き出すエントリ
     *
     * @return  関数の処理結果を返します。
     *
     * @pre
     *      - 初期化済み
     *
     * TEntry の先頭の要素は、int64_t 型の変数である必要があります。
     *
     * この関数では、ストレージの書き込み処理を行います。
     */
    template< typename TEntry >
    Result Write(const TEntry& entry) NN_NOEXCEPT
    {
        NN_STATIC_ASSERT(std::is_pod<TEntry>::value);
        NN_SDK_REQUIRES_EQUAL(sizeof(TEntry), m_EntrySize);

        return WriteImpl(&entry);
    }

    /**
     * @brief       エントリを書き出します。
     *
     * @param[in]   pEntry      書き出すエントリ配列の先頭アドレス
     * @param[in]   entryCount  書き出すエントリ配列の要素数
     *
     * @return  関数の処理結果を返します。
     *
     * @pre
     *      - pEntry != nullptr || entryCount == 0
     *      - 初期化済み
     *
     * TEntry の先頭の要素は、int64_t 型の変数である必要があります。
     *
     * この関数では、ストレージの書き込み処理を行います。
     */
    template< typename TEntry >
    Result Write(const TEntry* pEntry, int entryCount) NN_NOEXCEPT
    {
        NN_STATIC_ASSERT(std::is_pod<TEntry>::value);
        NN_SDK_REQUIRES_EQUAL(sizeof(TEntry), m_EntrySize);
        NN_SDK_REQUIRES(pEntry != nullptr || entryCount == 0);
        NN_SDK_REQUIRES_LESS_EQUAL(entryCount, m_TotalEntryCount);

        for( int i = 0; i < entryCount; ++i )
        {
            Result result = WriteImpl(&pEntry[i]);
            if(result.IsFailure())
            {
                return result;
            }
        }
        NN_RESULT_SUCCESS;
    }

    /**
     * @brief   エントリの書き出しを確定します。
     *
     * @param[in]   endOffset   エントリの最大オフセット
     *
     * @return  関数の処理結果を返します。
     *
     * @pre
     *      - 0 <= endOffset
     *      - 初期化済み
     *
     * この関数では、ストレージの書き込み処理を行います。
     */
    Result Commit(int64_t endOffset) NN_NOEXCEPT;

private:
    typedef BucketTree::Header Header;
    typedef BucketTree::NodeHeader NodeHeader;
    typedef BucketTree::NodeBuffer NodeBuffer;

private:
    /**
     * @brief   エントリを書き出します。
     *
     * @param[in]   pEntry  書き出すエントリの先頭アドレス
     *
     * @return  関数の処理結果を返します。
     *
     * @pre
     *      - pEntry != nullptr
     *      - 初期化済み
     *
     * この関数では、ストレージの書き込み処理を行います。
     */
    Result WriteImpl(const void* pEntry) NN_NOEXCEPT;

    /**
     * @brief   オフセットをキャッシュに設定します。
     */
    void UpdateOffset(int64_t entryOffset) NN_NOEXCEPT;

    /**
     * @brief   いっぱいになったキャッシュを書き出します。
     *
     * @return  関数の処理結果を返します。
     *
     * @pre
     *      - 0 <= entryOffset
     *      - 初期化済み
     *
     * この関数では、ストレージの書き込み処理を行います。
     */
    Result FlushCache(int64_t entryOffset) NN_NOEXCEPT;

private:
    NodeBuffer m_NodeL1;            //!< L1 ノードのキャッシュ
    NodeBuffer m_NodeL2;            //!< L2 ノードのキャッシュ
    NodeBuffer m_EntrySet;          //!< エントリセットのキャッシュ
    fs::SubStorage m_NodeStorage;   //!< ノードストレージ
    fs::SubStorage m_EntryStorage;  //!< エントリストレージ
    size_t m_NodeSize;              //!< １ノードのサイズ
    size_t m_EntrySize;             //!< １エントリのサイズ
    int m_TotalEntryCount;          //!< ツリーが管理するエントリの総数
    int m_EntryCount;               //!< １ノード当たりのエントリ数
    int m_OffsetCount;              //!< １ノード当たりのオフセット数
    int m_NodeL2Index;              //!< 出力中の L2 ノードのインデックス
    int m_EntryIndex;               //!< 出力中のエントリのインデックス
    int64_t m_EntryOffset;          //!< 直前に書き込んだエントリのオフセット
};

}} // fssystem
