﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <memory>
#include <nn/nn_Common.h>
#include <nn/util/util_MacroForVariadic.h>

namespace nn { namespace fssystem {

    typedef void* (*AllocateFunction)(size_t);
    typedef void(*DeallocateFunction)(void*, size_t);

    void InitializeAllocator(AllocateFunction alloc, DeallocateFunction dealloc) NN_NOEXCEPT;

    void* Allocate(size_t size) NN_NOEXCEPT;
    void Deallocate(void* p, size_t size) NN_NOEXCEPT;

    template <typename T>
    class StdAllocator : public std::allocator<T>
    {
    public:
        StdAllocator(){}
        StdAllocator(const StdAllocator&){}
        template<class U>
        StdAllocator(const StdAllocator<U>&){}

        template<typename U>
        struct rebind
        {
            typedef StdAllocator<U> other;
        };

        T* allocate(size_t size, const T* hint = 0)
        {
            NN_UNUSED(hint);
            return static_cast<T*>(nn::fssystem::Allocate(sizeof(T) * size));
        }
        void deallocate(T* p, size_t size)
        {
            return nn::fssystem::Deallocate(p, sizeof(T) * size);
        }
    };

#if defined(NN_BUILD_FOR_DOCUMENT_GENERATION)
    template <typename T, class... Args>
    std::shared_ptr<T> AllocateShared(Args&&... args) NN_NOEXCEPT;
#else
    template <typename T>
    std::shared_ptr<T> AllocateShared()
    {
        StdAllocator<T> allocator;
        return std::allocate_shared<T>(allocator);
    }

    #define NN_FSSYSTEM_TEMPLATE_OBJECT_DEFINE_VARIADIC_ALLOCATE_SHARED(n)  \
        template <typename T, NN_UTIL_VARIADIC_TEMPLATE_TEMPLATE_ARGUMENTS_##n (U)> \
        std::shared_ptr<T> AllocateShared(NN_UTIL_VARIADIC_TEMPLATE_ARGUMENT_LIST_##n (U, x)) \
        { \
            StdAllocator<T> allocator; \
            return std::allocate_shared<T>(allocator, NN_UTIL_VARIADIC_TEMPLATE_FORWARD_LIST_##n (U, x)); \
        }

        NN_UTIL_VARIADIC_DEFINE_MACROS(NN_FSSYSTEM_TEMPLATE_OBJECT_DEFINE_VARIADIC_ALLOCATE_SHARED)

    #undef NN_FSSYSTEM_TEMPLATE_OBJECT_DEFINE_VARIADIC_ALLOCATE_SHARED
#endif

}}
