﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/fssystem/dbm/fs_DbmParameters.h>
#include <nn/fssystem/dbm/fs_DbmPathTool.h>
#include <nn/fssystem/dbm/fs_KeyValueListTemplate.h>

class HierarchicalFileTableTest;

namespace nn { namespace fssystem { namespace dbm {

/**
* @brief        ファイル、ディレクトリエントリー管理テーブルクラスです。
*
* @tparam       TDirectoryName  ディレクトリ名
* @tparam       TFileName       ファイル名
* @tparam       TDirectoryInfo  ディレクトリ情報
* @tparam       TFileInfo       ファイル情報
*
* @details
*   ファイル、ディレクトリエントリー管理テーブルクラスです。
*
*   エントリー情報のみを管理し、以下の操作はサポートしていません。
*   - ファイルアクセスの排他
*   - アロケーションテーブルからのメモリ取得
*
*   以下のような操作を行った場合はエラーを返します。
*
*   - パスに存在しないディレクトリが含まれている場合は ResultDirectoryNotFound を返します。
*   - パスに含まれているディレクトリ名が長すぎる場合は ResultDirectoryNameTooLong を返します。
*   - パスに含まれているファイル名が長すぎる場合は ResultFileNameTooLong を返します。
*   - パスに含まれている区切り文字が複数回連続している場合には ResultInvalidPathFormat を返します。
*   - パスの 1 文字目が "/" でない場合は ResultInvalidPathFormat を返します。
*   - ファイルに対してディレクトリ操作を行おうとした場合は ResultInvalidOperation を返します。
*   - ディレクトリに対してファイル操作を行おうとした場合は ResultInvalidOperation を返します。
*
*   作成処理用エラー
*   - すでに存在しているディレクトリ、ファイルを作成しようとした場合は
*     ResultAlreadyExists を返します。
*
*   削除処理用エラー
*   - 存在しないディレクトリを削除しようとした場合は ResultDirectoryNotFound を返します。
*   - 存在しないファイルを削除しようとした場合は ResultFileNotFound を返します。
*   - 削除できないディレクトリを削除しようとした場合は ResultOperationDenied を返します。
*
*   リネーム処理用エラー
*   - リネームできないディレクトリをリネームしようとした場合は ResultOperationDenied を返します。
*   - リネーム先に同名のファイルが存在した場合は ResultAlreadyExists を返します。
*   - リネーム元とリネーム先が同名の場合は ResultAlreadyExists を返します。
*   - リネーム先のディレクトリがリネーム元のディレクトリの子であった場合は
*     ResultDirectoryNotFound を返します。
*     (例: リネーム元:"/parent/from/" -> リネーム先:"/parent/from/to/")
*/
template <
    typename TDirectoryName,
    typename TFileName,
    typename TDirectoryInfo,
    typename TFileInfo
>
class HierarchicalFileTableTemplate
{
    NN_DISALLOW_COPY(HierarchicalFileTableTemplate);

public:
    //! インデックス
    typedef uint32_t StorageIndex;

    /**
    * @brief        エントリー探索のためのイテレータです。
    *
    * @details      FindOpen を呼び出すことで初期化します。
    */
    struct FindIndex
    {
        StorageIndex nextIndexDirectory;    //! 探索中のディレクトリエントリーのインデックス
        StorageIndex nextIndexFile;         //! 探索中のファイルエントリーのインデックス
    };
    NN_STATIC_ASSERT(std::is_pod<FindIndex>::value);

public:
    /**
    * @brief        エントリー数から必要なストレージのサイズを求めます。
    *
    * @param[in]    countDirectoryEntry     ディレクトリエントリー数
    * @param[in]    countFileEntry          ファイルエントリー数
    *
    * @return       必要なストレージのサイズ。
    *
    * @details      エントリー数から必要なストレージのサイズを求めます。
    *               - "エントリー数"は、作成可能なディレクトリ数、ファイル数です。
    *               - エントリー数以上にディレクトリやファイルを作成しようとすると、
    *                 ストレージの割り当て領域が増やされ、最大エントリー数が増えます。
    */
    static uint32_t QueryDirectoryEntryStorageSize(uint32_t countDirectoryEntry) NN_NOEXCEPT;
    static uint32_t QueryFileEntryStorageSize(uint32_t countFileEntry) NN_NOEXCEPT;

    /**
    * @brief        ファイル管理テーブルをフォーマットします。
    *
    * @param[in]    pDirectoryEntryStorage      ディレクトリエントリー用ストレージ
    * @param[in]    pFileEntryStorage           ファイルエントリー用ストレージ
    *
    * @return       関数の処理結果を返します。
    * @retval       ResultSuccess                   正常に初期化できました。
    * @retval       ResultNotInitialized            pDirectoryEntryStorage が初期化されていません。
    * @retval       ResultNotInitialized            pFileEntryStorage が初期化されていません。
    * @retval       ResultInvalidOffset             pDirectoryEntryStorage が小さすぎます。
    * @retval       ResultInvalidOffset             pFileEntryStorage が小さすぎます。
    * @retval       ResultAllocationTableFull       アロケーションテーブルに空きがありません。
    * @retval       ResultInvalidOffset             内部で不明な問題が発生しました。
    * @retval       ResultInvalidOffset             内部データに問題があります。
    * @retval       ResultDatabaseCorrupted         内部データに問題があります。
    *
    * @pre          pDirectoryEntryStorage != nullptr
    * @pre          pFileEntryStorage != nullptr
    *
    * @details      ファイル管理テーブルをフォーマットします。
    *               IStorage を初期化します。使用する前に一度呼び出します。
    */
    static Result Format(
                      BufferedAllocationTableStorage* pDirectoryEntryStorage,
                      BufferedAllocationTableStorage* pFileEntryStorage
                 ) NN_NOEXCEPT;

    /**
    * @brief        コンストラクタです。
    *
    * @details      コンストラクタです。
    */
    HierarchicalFileTableTemplate() NN_NOEXCEPT;

    /**
    * @brief        ストレージをマウントします。
    *
    * @param[in]    pDirectoryEntryStorage     ディレクトリエントリー用ストレージ
    * @param[in]    pFileEntryStorage          ファイルエントリー用ストレージ
    *
    * @pre          pDirectoryEntryStorage != nullptr
    * @pre          pFileEntryStorage != nullptr
    *
    * @details      ストレージをマウントします。
    */
    void Initialize(
             BufferedAllocationTableStorage* pDirectoryEntryStorage,
             BufferedAllocationTableStorage* pFileEntryStorage
         ) NN_NOEXCEPT;

    /**
    * @brief        ストレージをアンマウントします。
    *
    * @details      ストレージをアンマウントします。
    */
    void Finalize() NN_NOEXCEPT;

    /**
    * @brief        ディレクトリを作成します。
    *
    * @param[in]    pFullPath                   ディレクトリパス
    * @param[in]    dirInfo                     ディレクトリ情報
    *
    * @return       関数の処理結果を返します。
    * @retval       ResultSuccess                   正常に作成できました。
    * @retval       ResultNotInitialized            初期化されていません。
    * @retval       ResultInvalidPathFormat         pFullPath がルートディレクトリから始まっていません。
    * @retval       ResultTooLongPath               ディレクトリ名が長すぎます。
    * @retval       ResultDirectoryUnobtainable     ルートディレクトリの親ディレクトリは取得できません。
    * @retval       ResultIncompatiblePath          pFullPath がファイルでした。
    * @retval       ResultAlreadyExists             同名のファイルまたはディレクトリが存在します。
    * @retval       ResultAllocationTableFull       アロケーションテーブルに空きがありません。
    * @retval       ResultInvalidOffset             内部データに問題があります。
    * @retval       ResultDatabaseCorrupted         内部データに問題があります。
    * @retval       上記以外                        ストレージでの読み書きに失敗しました。
    *
    * @pre          pFullPath != nullptr
    *
    * @details      ディレクトリを作成します。
    */
    Result CreateDirectory(
               const PathChar* pFullPath,
               const TDirectoryInfo& dirInfo
           ) NN_NOEXCEPT;

    /**
    * @brief        ファイルを作成します。
    *
    * @param[out]   outIndex                        ファイルインデックス
    * @param[in]    pFullPath                       ファイルパス
    * @param[in]    fileInfo                        ファイル情報
    *
    * @retval       ResultSuccess                   正常に作成できました。
    * @retval       ResultNotInitialized            初期化されていません。
    * @retval       ResultAlreadyExists             同名のファイルまたはディレクトリが存在します。
    * @retval       ResultInvalidPathFormat         pFullPath がルートディレクトリから始まっていません。
    * @retval       ResultTooLongPath               ディレクトリ名が長すぎます。
    * @retval       ResultDirectoryUnobtainable     ルートディレクトリの親ディレクトリは取得できません。
    * @retval       ResultIncompatiblePath          pFullPath がディレクトリでした。
    * @retval       ResultAllocationTableFull       アロケーションテーブルに空きがありません。
    * @retval       ResultInvalidOffset             内部データに問題があります。
    * @retval       ResultDatabaseCorrupted         内部データに問題があります。
    * @retval       上記以外                        ストレージでの読み書きに失敗しました。
    *
    * @pre          マウントしている。
    * @pre          outIndex != nullptr
    * @pre          pFullPath != nullptr
    *
    * @details      ファイルを作成します。
    */
    Result CreateFile(
               StorageIndex* outIndex,
               const PathChar* pFullPath,
               const TFileInfo& fileInfo
           ) NN_NOEXCEPT;
    Result CreateFile(
               const PathChar* pFullPath,
               const TFileInfo& fileInfo
           ) NN_NOEXCEPT
    {
        StorageIndex tmp;
        return CreateFile(&tmp, pFullPath, fileInfo);
    }

    /**
    * @brief        指定したディレクトリを削除します。
    *
    * @param[in]    pFullpath                   ディレクトリパス
    *
    * @return       関数の処理結果を返します。
    * @retval       ResultSuccess                   正常に取得できました。
    * @retval       ResultNotInitialized            初期化されていません。
    * @retval       ResultInvalidPathFormat         pFullPath がルートディレクトリから始まっていません。
    * @retval       ResultTooLongPath               ディレクトリ名が長すぎます。
    * @retval       ResultDirectoryUnobtainable     ルートディレクトリの親ディレクトリは取得できません。
    * @retval       ResultDirectoryUndeletable      ルートディレクトリを削除することはできません。
    * @retval       ResultIncompatiblePath          pFullPath がファイルでした。
    * @retval       ResultIncompatiblePath          内部データに問題があります。
    * @retval       ResultDirectoryNotFound         内部データに問題があります。
    * @retval       ResultDatabaseKeyNotFound       内部データに問題があります。
    * @retval       ResultInvalidOffset             内部データに問題があります。
    * @retval       ResultDatabaseCorrupted         内部データに問題があります。
    * @retval       上記以外                        ストレージでのデータ読み書きに失敗しました。
    *
    * @pre          pFullPath != nullptr
    *
    * @details      指定したディレクトリを削除します。
    */
    Result DeleteDirectory(const PathChar* pFullPath) NN_NOEXCEPT;

    /**
    * @brief        指定したファイルを削除します。
    *
    * @param[in]    pFullpath                       ファイルパス
    *
    * @return       関数の処理結果を返します。
    * @retval       ResultSuccess                   正常に取得できました。
    * @retval       ResultNotInitialized            初期化されていません。
    * @retval       ResultTooLongPath               ディレクトリ名が長すぎます。
    * @retval       ResultDirectoryUnobtainable     ルートディレクトリの親ディレクトリは取得できません。
    * @retval       ResultInvalidPathFormat         pFullPath がルートディレクトリから始まっていません。
    * @retval       ResultIncompatiblePath          pFullPath がディレクトリでした。
    * @retval       ResultIncompatiblePath          内部データに問題があります。
    * @retval       ResultFileNotFound              内部データに問題があります。
    * @retval       ResultDatabaseKeyNotFound       内部データに問題があります。
    * @retval       ResultInvalidOffset             内部データに問題があります。
    * @retval       ResultDatabaseCorrupted         内部データに問題があります。
    * @retval       上記以外                        ストレージでのデータ読み書きに失敗しました。
    *
    * @pre          pFullPath != nullptr
    *
    * @details      指定したファイルを削除します。
    */
    Result DeleteFile(const PathChar* pFullPath) NN_NOEXCEPT;

    /**
    * @brief        指定したディレクトリの名前を変更します。
    *
    * @param[out]   outIsFile                   リネーム先に同名のエントリーが存在したときにファイルであれば true、ディレクトリであれば false
    * @param[out]   outIsSameEntry              変更前と変更後のパスが同じエントリーを指しているかどうか
    * @param[out]   outIsParentEntry            変更後のパスが変更前のパスの親ディレクトリかどうか
    * @param[in]    pNewFullPath                変更後のファイルパス
    * @param[in]    pOldFullPath                変更前のファイルパス
    *
    * @return       関数の処理結果を返します。
    * @retval       ResultSuccess                   正常に変更できました。
    * @retval       ResultSuccess                   変更前後で名前が同じです。
    * @retval       ResultNotInitialized            初期化されていません。
    * @retval       ResultInvalidPathFormat         パスがルートディレクトリから始まっていません。
    * @retval       ResultTooLongPath               ディレクトリ名が長すぎます。
    * @retval       ResultDirectoryUnobtainable     ルートディレクトリの親ディレクトリは取得できません。
    * @retval       ResultIncompatiblePath          パスがファイルへのパスでした。
    * @retval       ResultDirectoryUnrenamable      ルートディレクトリはリネームはできません。
    * @retval       ResultNotInitialized            初期化されていません。
    * @retval       ResultDirectoryUnrenamable      pNewFullPath と pOldFullPath は親子関係にあります。
    * @retval       ResultAlreadyExists             同名のファイルまたはディレクトリが存在します。
    * @retval       ResultIncompatiblePath          内部データに問題があります。
    * @retval       ResultDirectoryNotFound         内部データに問題があります。
    * @retval       ResultDatabaseKeyNotFound       内部データに問題があります。
    * @retval       ResultInvalidOffset             内部データに問題があります。
    * @retval       ResultDatabaseCorrupted         内部データに問題があります。
    * @retval       上記以外                        ストレージでの読み書きに失敗しました。
    *
    * @pre          outIsFile != nullptr
    * @pre          outIsSameEntry != nullptr
    * @pre          outIsParentEntry != nullptr
    * @pre          pNewFullPath != nullptr
    * @pre          pOldFullPath != nullptr
    *
    * @details      指定したディレクトリの名前を変更します。
    */
    Result RenameDirectory(
               bool* outIsFile,
               bool* outIsSameEntry,
               bool* outIsParentEntry,
               const PathChar* pNewFullPath,
               const PathChar* pOldFullPath
           ) NN_NOEXCEPT;

    Result RenameDirectory(
               const PathChar* pNewFullPath,
               const PathChar* pOldFullPath
           ) NN_NOEXCEPT
    {
        bool isFile;
        bool isSameEntry;
        bool isParentEntry;
        return RenameDirectory(
                   &isFile,
                   &isSameEntry,
                   &isParentEntry,
                   pNewFullPath,
                   pOldFullPath
               );
    }

    /**
    * @brief        指定したファイル名を変更します。
    *
    * @param[out]   outIsFile                   リネーム先に同名のエントリーが存在したときにファイルであれば true、ディレクトリであれば false
    * @param[out]   outIsSameEntry              変更前と変更後のパスが同じエントリーを挿しているかどうか
    * @param[in]    pNewFullPath                変更後のファイルパス
    * @param[in]    pOldFullPath                変更前のファイルパス
    *
    * @return       関数の処理結果を返します。
    * @retval       ResultSuccess                   正常に取得できました。
    * @retval       ResultSuccess                   変更前後で同じファイル名です。
    * @retval       ResultNotInitialized            初期化されていません。
    * @retval       ResultInvalidPathFormat         パスがルートディレクトリから始まっていません。
    * @retval       ResultTooLongPath               ディレクトリ名が長すぎます。
    * @retval       ResultDirectoryUnobtainable     ルートディレクトリの親ディレクトリは取得できません。
    * @retval       ResultIncompatiblePath          パスがディレクトリへのパスでした。
    * @retval       ResultAlreadyExists             同名のファイルまたはディレクトリが存在します。
    * @retval       ResultIncompatiblePath          内部データに問題があります。
    * @retval       ResultFileNotFound              内部データに問題があります。
    * @retval       ResultDatabaseKeyNotFound       内部データに問題があります。
    * @retval       ResultInvalidOffset             内部データに問題があります。
    * @retval       ResultDatabaseCorrupted         内部データに問題があります。
    * @retval       上記以外                        ストレージでの読み書きに失敗しました。
    *
    * @pre          outIsFile != nullptr
    * @pre          outIsSameEntry != nullptr
    * @pre          pNewFullPath != nullptr
    * @pre          pOldFullPath != nullptr
    *
    * @details      指定したファイル名を変更します。
    */
    Result RenameFile(
               bool* outIsFile,
               bool* outIsSameEntry,
               const PathChar* pNewFullPath,
               const PathChar* pOldFullPath
           ) NN_NOEXCEPT;

    Result RenameFile(
               const PathChar* pNewFullPath,
               const PathChar* pOldFullPath
           ) NN_NOEXCEPT
    {
        bool isFile;
        bool isSameEntry;
        return RenameFile(&isFile, &isSameEntry, pNewFullPath, pOldFullPath);
    }

    /**
    * @brief        指定したディレクトリ情報を取得します。
    *
    * @param[out]   outIndex                    インデックス
    * @param[out]   outDirectoryInfo            ディレクトリ情報
    * @param[in]    pFullPath                   ディレクトリのフルパス
    *
    * @return       関数の処理結果を返します。
    * @retval       ResultSuccess                   正常に取得できました。
    * @retval       ResultNotInitialized            初期化されていません。
    * @retval       ResultInvalidPathFormat         pFullPath がルートディレクトリから始まっていません。
    * @retval       ResultTooLongPath               ディレクトリ名が長すぎます。
    * @retval       ResultDirectoryUnobtainable     ルートディレクトリの親ディレクトリは取得できません。
    * @retval       ResultIncompatiblePath          pFullPath がファイルでした。
    * @retval       ResultNotInitialized            初期化されていません。
    * @retval       ResultIncompatiblePath          内部データに問題があります。
    * @retval       ResultDirectoryNotFound         内部データに問題があります。
    * @retval       ResultInvalidOffset             内部データに問題があります。
    * @retval       ResultDatabaseCorrupted         内部データに問題があります。
    * @retval       上記以外                        ストレージからのデータ読み込みに失敗しました。
    *
    * @pre          outIndex != nullptr
    * @pre          outDirectoryInfo != nullptr
    * @pre          pFullPath != nullptr
    *
    * @details      指定したディレクトリ情報を取得します。
    */
    Result GetDirectoryInformation(
               StorageIndex* outIndex,
               TDirectoryInfo* pDirectoryInfo,
               const PathChar* pFullPath
           ) const NN_NOEXCEPT;

    /**
    * @brief        ファイルを開きます。
    *
    * @param[out]   outIndex                    インデックス
    * @param[out]   outFileInfo                 ファイル情報
    * @param[in]    pFullPath                   ファイルのフルパス
    *
    * @return       関数の処理結果を返します。
    * @retval       ResultSuccess                   正常に取得できました。
    * @retval       ResultNotInitialized            初期化されていません。
    * @retval       ResultInvalidPathFormat         pFullPath がルートディレクトリから始まっていません。
    * @retval       ResultTooLongPath               ディレクトリ名が長すぎます。
    * @retval       ResultDirectoryUnobtainable     ルートディレクトリの親ディレクトリは取得できません。
    * @retval       ResultIncompatiblePath          pFullPath がディレクトリでした。
    * @retval       ResultIncompatiblePath          内部データに問題があります。
    * @retval       ResultFileNotFound              内部データに問題があります。
    * @retval       ResultInvalidOffset             内部データに問題があります。
    * @retval       ResultDatabaseCorrupted         内部データに問題があります。
    * @retval       上記以外                        ストレージからのデータ読み込みに失敗しました。
    *
    * @pre          outIndex != nullptr
    * @pre          outFileInfo != nullptr
    * @pre          pFullPath != nullptr
    *
    * @details      ファイルを開きます。
    */
    Result OpenFile(
               StorageIndex* outIndex,
               TFileInfo* outFileInfo,
               const PathChar* pFullPath
           ) const NN_NOEXCEPT;

    Result OpenFile(
               TFileInfo* outFileInfo,
               const PathChar* pFullPath
           ) const NN_NOEXCEPT
    {
        StorageIndex tmp;
        return OpenFile(&tmp, outFileInfo, pFullPath);
    }

    /**
    * @brief        指定したディレクトリ情報を更新します。
    *
    * @param[in]    pFullPath                   ディレクトリパス
    * @param[in]    dirInfo                     ディレクトリ情報
    *
    * @return       関数の処理結果を返します。
    * @retval       ResultSuccess                   正常に更新できました。
    * @retval       ResultNotInitialized            初期化されていません。
    * @retval       ResultInvalidPathFormat         pFullPath がルートディレクトリから始まっていません。
    * @retval       ResultTooLongPath               ディレクトリ名が長すぎます。
    * @retval       ResultDirectoryUnobtainable     ルートディレクトリの親ディレクトリは取得できません。
    * @retval       ResultIncompatiblePath          pFullPath がファイルでした。
    * @retval       ResultIncompatiblePath          内部データに問題があります。
    * @retval       ResultDirectoryNotFound         内部データに問題があります。
    * @retval       ResultInvalidOffset             内部データに問題があります。
    * @retval       ResultDatabaseCorrupted         内部データに問題があります。
    * @retval       上記以外                        ストレージでの読み書きに失敗しました。
    *
    * @pre          pFullPath != nullptr
    *
    * @details      指定したディレクトリ情報を更新します。
    */
    Result UpdateDirectoryInformation(
               const PathChar* pFullPath,
               const TDirectoryInfo& dirInfo
           ) NN_NOEXCEPT;

    /**
    * @brief        指定したファイル情報を更新します。
    *
    * @param[in]    pFullPath                   ファイルパス
    * @param[in]    fileInfo                    ファイル情報
    *
    * @return       関数の処理結果を返します。
    * @retval       ResultSuccess                   正常に更新できました。
    * @retval       ResultNotInitialized            初期化されていません。
    * @retval       ResultInvalidPathFormat         pFullPath がルートディレクトリから始まっていません。
    * @retval       ResultTooLongPath               ディレクトリ名が長すぎます。
    * @retval       ResultDirectoryUnobtainable     ルートディレクトリの親ディレクトリは取得できません。
    * @retval       ResultIncompatiblePath          pFullPath がディレクトリでした。
    * @retval       ResultIncompatiblePath          内部データに問題があります。
    * @retval       ResultFileNotFound              内部データに問題があります。
    * @retval       ResultInvalidOffset             内部データに問題があります。
    * @retval       ResultDatabaseCorrupted         内部データに問題があります。
    * @retval       上記以外                        ストレージでのデータ読み書きに失敗しました。
    *
    * @pre          pFullPath != nullptr
    *
    * @details      指定したファイル情報を更新します。
    */
    Result UpdateFileInformation(
               const PathChar* pFullPath,
               const TFileInfo& fileInfo
           ) NN_NOEXCEPT;

    /**
    * @brief        指定したディレクトリ以下のディレクトリとファイル列挙を開始します。
    *
    * @param[out]   iter                        イテレータ
    * @param[in]    pFullPath                   ディレクトリパス
    *
    * @return       関数の処理結果を返します。
    * @retval       ResultSuccess                   正常に取得できました。
    * @retval       ResultNotInitialized            初期化されていません。
    * @retval       ResultInvalidPathFormat         pFullPath がルートディレクトリから始まっていません。
    * @retval       ResultTooLongPath               ディレクトリ名が長すぎます。
    * @retval       ResultDirectoryUnobtainable     ルートディレクトリの親ディレクトリは取得できません。
    * @retval       ResultIncompatiblePath          pFullPath がファイルでした。
    * @retval       ResultIncompatiblePath          内部データに問題があります。
    * @retval       ResultDirectoryNotFound         内部データに問題があります。
    * @retval       ResultInvalidOffset             内部データに問題があります。
    * @retval       ResultDatabaseCorrupted         内部データに問題があります。
    * @retval       上記以外                        ストレージからのデータ読み込みに失敗しました。
    *
    * @pre          iter != nullptr
    * @pre          pFullPath != nullptr
    *
    * @details      指定したディレクトリ以下のディレクトリとファイル列挙を開始します。
    *               イテレータを @see FindNextDirectory、@see FindNextFile に
    *               渡すことでディレクトリ名、ファイル名をイテレーションできます。
    *               また、イテレーションはファイル、ディレクトリ各々に対して行うことができます。
    */
    Result FindOpen(FindIndex* pIter, const PathChar* pFullPath) NN_NOEXCEPT;

    /**
    * @brief        ディレクトリ以下の次のディレクトリを取得します。
    *
    * @param[out]   outDirectoryName        ディレクトリ名
    * @param[out]   outFinished             ディレクトリ名
    * @param[in]    iter                    イテレータ
    *
    * @return       関数の処理結果を返します。
    * @retval       ResultSuccess                   正常にディレクトリが取得できました。
    * @retval       ResultNotInitialized            初期化されていません。
    * @retval       ResultInvalidOffset             iter が範囲外です。
    * @retval       ResultInvalidOffset             内部で不明な問題が発生しました。
    * @retval       ResultDatabaseCorrupted         内部データに問題があります。
    * @retval       上記以外                        ストレージからのデータ読み込みに失敗しました。
    *
    * @pre          outDirectoryName != nullptr
    * @pre          outFinished != nullptr
    * @pre          iter != nullptr
    * @pre          @see FindOpen でイテレータを初期化している。
    *
    * @details      ディレクトリ以下の次のディレクトリを取得します。
    */
    Result FindNextDirectory(
               TDirectoryName* outDirectoryName,
               bool* outFinished,
               FindIndex* iter
           ) NN_NOEXCEPT;

    /**
    * @brief        ディレクトリ以下の次のファイルを取得します。
    *
    * @param[out]   outFileSize             ファイルのサイズ
    * @param[out]   outFileName             ファイル名
    * @param[out]   outFinished             イテレーションが終了したか(次ファイルが有るか)
    * @param[in]    iter                    イテレータ
    *
    * @return       関数の処理結果を返します。
    * @retval       ResultSuccess                   正常にファイルが取得できました。
    * @retval       ResultNotInitialized            初期化されていません。
    * @retval       ResultInvalidOffset             iter が範囲外です。
    * @retval       ResultInvalidOffset             内部で不明な問題が発生しました。
    * @retval       ResultDatabaseCorrupted         内部データに問題があります。
    * @retval       上記以外                        ストレージからのデータ読み込みに失敗しました。
    *
    * @pre          outFileSize != nullptr
    * @pre          outFileName != nullptr
    * @pre          outFinished != nullptr
    * @pre          iter != nullptr
    * @pre          @see FindOpen でイテレータを初期化している。
    *
    * @details      ディレクトリ以下の次のファイルを取得します。
    */
    Result FindNextFile(
               int64_t* outFileSize,
               TFileName* outFileName,
               bool* outFinished,
               FindIndex* iter
           ) NN_NOEXCEPT;

    /**
    * @brief        ID からエントリとディレクトリが親子関係にあるか調べる
    *
    * @param[out]   outIsSubEntry       checkEntryId が baseDirectoryId の子か否か
    * @param[in]    baseDirectoryId     ディレクトリの ID
    * @param[in]    checkEntryId        エントリの ID
    * @param[in]    isFile              ファイルであれば true、ディレクトリであれば false
    *
    * @return       関数の処理結果を返します。
    *
    * @details      ID からエントリとディレクトリが親子関係にあるか調べる
    */
    Result CheckSubEntry(
        bool* outIsSubEntry,
        StorageIndex baseDirectoryId,
        StorageIndex checkEntryId,
        bool isFile
    ) const NN_NOEXCEPT;

protected:
    //! ディレクトリエントリーです。
    struct DirectoryEntry
    {
        StorageIndex   indexNext;           //! 兄弟へのインデックス
        StorageIndex   indexDirectory;      //! 子ディレクトリー内にあるディレクトリのインデックス
        StorageIndex   indexFile;           //! 子ディレクトリー内にあるファイルのインデックス
        TDirectoryInfo info;                //! ディレクトリ付加情報
    };
    NN_STATIC_ASSERT(std::is_pod<DirectoryEntry>::value);

    //! ファイルエントリーです。
    struct FileEntry
    {
        StorageIndex   indexNext;           //! 兄弟へのインデックス
        TFileInfo      info;                //! ファイル付加情報
    };
    NN_STATIC_ASSERT(std::is_pod<FileEntry>::value);

    //! ディレクトリエントリーを管理するハッシュテーブル用キー。
    struct DirectoryKey
    {
        StorageIndex  parentDir;            //! 親ディレクトリ
        DirectoryName name;                 //! ディレクトリ名

        //! キーの比較を行います。
        inline bool operator==(const DirectoryKey& rKey) const
        {
            return (std::memcmp(this, &rKey, sizeof(rKey)) == 0);
        }
    };
    NN_STATIC_ASSERT(std::is_pod<DirectoryKey>::value);

    //! ファイルエントリーを管理するハッシュテーブル用キー。
    struct FileKey
    {
        StorageIndex parentDir;             //! 親ディレクトリ
        FileName     name;                  //! ファイル名

        //! キーの比較を行います。
        inline bool operator==(const FileKey& rKey) const
        {
            return (std::memcmp(this, &rKey, sizeof(rKey)) == 0);
        }
    };
    NN_STATIC_ASSERT(std::is_pod<FileKey>::value);

protected:
    //! 無効なインデックス。
    static const StorageIndex IndexNone = 0;

    //! システムが暗黙的に使用するディレクトリエントリー
    static const uint32_t DirectoryCountSystemReserved = 1;

    //! システムが暗黙的に使用するファイルエントリー
    static const uint32_t FileCountSystemReserved = 0;

protected:
    /**
    * @brief        指定したパスから親ディレクトリとディレクトリキーを取得します。
    *
    * @param[out]   outParentDirectoryKey       親ディレクトリキー
    * @param[out]   outParentDirectoryEntry     親ディレクトリエントリー
    * @param[out]   outDirectoryKey             ディレクトリキー
    * @param[in]    pFullPath                   フルパス
    *
    * @return       関数の処理結果を返します。
    * @retval       ResultSuccess                   正常に取得できました。
    * @retval       ResultNotInitialized            初期化されていません。
    * @retval       ResultInvalidPathFormat         pFullPath がルートディレクトリから始まっていません。
    * @retval       ResultTooLongPath               ディレクトリ名が長すぎます。
    * @retval       ResultDirectoryUnobtainable     ルートディレクトリの親ディレクトリは取得できません。
    * @retval       ResultIncompatiblePath          pFullPath がファイルでした。
    * @retval       ResultInvalidOffset             内部データに問題があります。
    * @retval       ResultDatabaseCorrupted         内部データに問題があります。
    * @retval       上記以外                        ストレージからのデータ読み込みに失敗しました。
    *
    * @pre          outParentDirectoryKey != nullptr
    * @pre          outParentDirectoryEntry != nullptr
    * @pre          outDirectoryKey != nullptr
    * @pre          pFullPath != nullptr
    *
    * @details      指定したパスから親ディレクトリとディレクトリキーを取得します。
    */
    Result FindDirectoryRecursive(
               DirectoryKey* outParentDirectoryKey,
               DirectoryEntry* outParentDirectoryEntry,
               DirectoryKey* outDirectoryKey,
               const PathChar* pFullPath
           ) const NN_NOEXCEPT;


    /**
    * @brief        指定したパスから親ディレクトリとファイルキーを取得します。
    *
    * @param[out]   outParentDirectoryKey       親ディレクトリキー
    * @param[out]   outParentDirectoryEntry     親ディレクトリエントリー
    * @param[out]   outFileKey                  ファイルキー
    * @param[in]    pFullPath                   フルパス
    *
    * @return       関数の処理結果を返します。
    * @retval       ResultSuccess                   正常に取得できました。
    * @retval       ResultNotInitialized            初期化されていません。
    * @retval       ResultInvalidPathFormat         pFullPath がルートディレクトリから始まっていません。
    * @retval       ResultTooLongPath               ディレクトリ名が長すぎます。
    * @retval       ResultDirectoryUnobtainable     ルートディレクトリの親ディレクトリは取得できません。
    * @retval       ResultIncompatiblePath          pFullPath がディレクトリでした。
    * @retval       ResultInvalidOffset             内部データに問題があります。
    * @retval       ResultDatabaseCorrupted         内部データに問題があります。
    * @retval       上記以外                        ストレージからのデータ読み込みに失敗しました。
    *
    * @pre          outParentDirectoryKey != nullptr
    * @pre          outParentDirectoryEntry != nullptr
    * @pre          outFileKey != nullptr
    * @pre          pFullPath != nullptr
    * @pre          マウントしている。
    *
    * @details      指定したパスから親ディレクトリとファイルキーを取得します。
    */
    Result FindFileRecursive(
               DirectoryKey* outParentDirectoryKey,
               DirectoryEntry* outParentDirectoryEntry,
               FileKey* outFileKey,
               const PathChar* pFullPath
           ) const NN_NOEXCEPT;

    /**
    * @brief        指定したディレクトリ情報を取得します。
    *
    * @param[out]   outIndex                        ディレクトリエントリーのインデックス
    * @param[out]   outDirectoryInfo                ディレクトリ情報
    * @param[in]    dirKey                          ディレクトリキー
    *
    * @return       関数の処理結果を返します。
    * @retval       ResultSuccess                   ディレクトリ情報が取得できました。
    * @retval       ResultNotInitialized            初期化されていません。
    * @retval       ResultIncompatiblePath          dirKey はディレクトリではなくファイルです。
    * @retval       ResultDirectoryNotFound         dirKey に紐づくディレクトリエントリーがありません。
    * @retval       ResultInvalidOffset             内部データに問題があります。
    * @retval       ResultDatabaseCorrupted         内部データに問題があります。
    * @retval       上記以外                        ストレージからのデータ読み込みに失敗しました。
    *
    * @pre          outIndex != nullptr
    * @pre          outDirectoryInfo != nullptr
    *
    * @details      指定したディレクトリ情報を取得します。
    */
    Result GetDirectoryInformationFromKey(
               StorageIndex* outIndex,
               TDirectoryInfo* outDirectoryInfo,
               const DirectoryKey& dirKey
           ) const NN_NOEXCEPT;

    /**
    * @brief        ファイルを開きます。
    *
    * @param[out]   outIndex                        ファイルエントリーのインデックス
    * @param[out]   outFileInfo                     ファイル情報
    * @param[in]    fileKey                         ファイルキー
    *
    * @return       関数の処理結果を返します。
    * @retval       ResultSuccess                   正常にファイル情報を取得しました。
    * @retval       ResultNotInitialized            初期化されていません。
    * @retval       ResultIncompatiblePath          fileKey はファイルではなくディレクトリです。
    * @retval       ResultFileNotFound              fileKey に紐づくファイルエントリーがありません。
    * @retval       ResultInvalidOffset             内部データに問題があります。
    * @retval       ResultDatabaseCorrupted         内部データに問題があります。
    * @retval       上記以外                        ストレージからのデータ読み込みに失敗しました。
    *
    * @pre          outIndex != nullptr
    * @pre          outFileInfo != nullptr
    *
    * @details      ファイルを開きます。
    */
    Result OpenFileFromKey(
               StorageIndex* outIndex,
               TFileInfo* outFileInfo,
               const FileKey& fileKey
           ) const NN_NOEXCEPT;

    Result OpenFileFromKey(
               TFileInfo* outFileInfo,
               const FileKey& key
           ) const NN_NOEXCEPT
    {
        StorageIndex tmp;
        return OpenFileFromKey(&tmp, outFileInfo, key);
    }

    /**
    * @brief        指定したディレクトリ情報を更新します。
    *
    * @param[in]    dirKey                      ディレクトリキー
    * @param[in]    dirInfo                     ディレクトリ情報
    *
    * @return       関数の処理結果を返します。
    * @retval       ResultSuccess                   ディレクトリエントリーが更新できました。
    * @retval       ResultNotInitialized            初期化されていません。
    * @retval       ResultIncompatiblePath          dirKey はディレクトリではなくファイルです。
    * @retval       ResultDirectoryNotFound         dirKey に紐づくディレクトリエントリーがありません。
    * @retval       ResultInvalidOffset             内部データに問題があります。
    * @retval       ResultDatabaseCorrupted         内部データに問題があります。
    * @retval       上記以外                        ストレージでの読み書きに失敗しました。
    *
    * @details      指定したディレクトリ情報を更新します。
    */
    Result UpdateDirectoryInformationFromKey(
            const DirectoryKey& dirKey,
            const TDirectoryInfo& dirInfo
        ) NN_NOEXCEPT;

    /**
    * @brief        指定したファイル情報を更新します。
    *
    * @param[in]    fileKey                     ファイルキー
    * @param[in]    fullInfo                    ファイル情報
    *
    * @return       関数の処理結果を返します。
    * @retval       ResultSuccess                   正常に更新できました。
    * @retval       ResultNotInitialized            初期化されていません。
    * @retval       ResultIncompatiblePath          fileKey はファイルではなくディレクトリです。
    * @retval       ResultFileNotFound              fileKey に紐づくファイルエントリーがありません。
    * @retval       ResultInvalidOffset             内部データに問題があります。
    * @retval       ResultDatabaseCorrupted         内部データに問題があります。
    * @retval       上記以外                        ストレージでのデータ読み書きに失敗しました。
    *
    * @details      指定したファイル情報を更新します。
    */
    Result UpdateFileInformationFromKey(
               const FileKey& fileKey,
               const TFileInfo& fileInfo
           ) NN_NOEXCEPT;

    /**
    * @brief        指定したディレクトリ以下のディレクトリとファイル列挙を開始します。
    *
    * @param[out]   iter                        イテレータ
    * @param[in]    dirKey                      ディレクトリキー
    *
    * @return       関数の処理結果を返します。
    * @retval       ResultSuccess                   正常にイテレータを取得できました。
    * @retval       ResultNotInitialized            初期化されていません。
    * @retval       ResultIncompatiblePath          dirKey はディレクトリではなくファイルです。
    * @retval       ResultDirectoryNotFound         dirKey に紐づくディレクトリエントリーがありません。
    * @retval       ResultInvalidOffset             内部データに問題があります。
    * @retval       ResultDatabaseCorrupted         内部データに問題があります。
    * @retval       上記以外                        ストレージからのデータ読み込みに失敗しました。
    *
    * @pre          iter != nullptr
    *
    * @details      指定したディレクトリ以下のディレクトリとファイル列挙を開始します。
    *               イテレータを @see FindNextDirectory、@see FindNextFile に
    *               渡すことでディレクトリ名、ファイル名をイテレーションできます。
    *               また、イテレーションはファイル、ディレクトリ各々に対して行うことができます。
    */
    Result FindOpenWithKey(
               FindIndex* iter,
               const DirectoryKey& dirKey
           ) NN_NOEXCEPT;

    /**
    * @brief        メタデータの更新を通知します。
    *
    * @param[out]   outFinished             イテレーションが終了したか
    * @param[in]    iter                    イテレータ
    * @param[in]    id                      更新したエントリー ID
    * @param[in]    isFile                  ファイルなら true, ディレクトリなら false
    *
    * @return       関数の処理結果を返します。
    * @retval       ResultSuccess                   正常にエントリーが取得できました。
    * @retval       ResultNotInitialized            初期化されていません。
    * @retval       ResultInvalidOffset             iter が範囲外です。
    * @retval       ResultInvalidOffset             内部で不明な問題が発生しました。
    * @retval       ResultDatabaseCorrupted         内部データに問題があります。
    * @retval       上記以外                        ストレージからのデータ読み込みに失敗しました。
    *
    * @pre          iter != nullptr
    *
    * @details      メタデータの更新を通知します。
    */
    Result Notify(
               bool* outFinished,
               FindIndex* iter,
               int64_t id,
               bool isFile
           ) NN_NOEXCEPT;

private:
    /**
    * @brief        エントリーリストの共通実装。
    *
    * @tparam       Key         キーとなるクラス
    * @tparam       Value       値となるクラス
    *
    * @details      エントリーリストの共通実装。
    */
    template <typename Key, typename Value>
    class EntryList
            : public KeyValueListTemplate<Key, Value>
    {
        NN_DISALLOW_COPY(EntryList);

    public:
        typedef Value TypeValue;        //!< テスト用
        typedef Key TypeKey;            //!< テスト用
        typedef uint32_t Index;
        typedef KeyValueListTemplate<Key, Value> BaseClass;

    public:
        EntryList() NN_NOEXCEPT
        {
        }

        //! KeyValueKustStorageTemplate::AddInternal を公開します。
        inline Result Add(Index* outIndex, const Key& key, const Value& value) NN_NOEXCEPT
        {
            return BaseClass::AddInternal(outIndex, key, value);
        }

        //! KeyValueListStorageTemplate::GetInternal を公開します。
        inline Result Get(Index* outIndex, Value* pValue, const Key& key) const NN_NOEXCEPT
        {
            return BaseClass::GetInternal(outIndex, pValue, key);
        }

        //! KeyValueListStorageTemplate::GetByIndex を公開します。
        inline Result GetByIndex(Key* pKey, Value* pValue, Index index) const NN_NOEXCEPT
        {
            return BaseClass::GetByIndex(pKey, pValue, index);
        }

        //! KeyValueListStorageTemplate::SetByIndex を公開します。
        inline Result SetByIndex(Index index, const Value& value) const NN_NOEXCEPT
        {
            return BaseClass::SetByIndex(index, value);
        }

        //! KeyValueListStorageTemplate::Remove を公開します。
        inline Result Remove(const Key& key) NN_NOEXCEPT
        {
            return BaseClass::Remove(key);
        }

        //! KeyValueListStorageTemplate::ModifyKey を公開します。
        inline Result Rename(const Key& to, const Key& from) NN_NOEXCEPT
        {
            return BaseClass::ModifyKey(to, from);
        }
    };

    class CacheBufferEnabler
    {
    public:
        explicit CacheBufferEnabler(const HierarchicalFileTableTemplate* pFileTable) NN_NOEXCEPT
            : m_pFileTable(pFileTable)
        {
            NN_SDK_REQUIRES_NOT_NULL(pFileTable);
            // この処理はメモリ確保失敗以外のエラーは返さないはず
            // キャッシュ無しでも処理は続行できるので失敗は無視する
            (void)pFileTable->m_TableFile.EnableCacheBuffer();
            (void)pFileTable->m_TableDirectory.EnableCacheBuffer();
        }

        ~CacheBufferEnabler() NN_NOEXCEPT
        {
            m_pFileTable->m_TableFile.DisableCacheBuffer();
            m_pFileTable->m_TableDirectory.DisableCacheBuffer();
        }

    private:
        const HierarchicalFileTableTemplate* m_pFileTable;
    };

private:
    //! ディレクトリエントリーを管理するリスト
    typedef EntryList<DirectoryKey, DirectoryEntry> DirectoryEntryList;

    //! ファイルエントリーを管理するリスト
    typedef EntryList<FileKey, FileEntry> FileEntryList;

    typedef TDirectoryInfo DirectoryInfo;
    typedef TFileInfo FileInfo;

protected:
    typedef typename DirectoryEntryList::StorageElement DirectoryElement;
    NN_STATIC_ASSERT(sizeof(DirectoryElement) == 96);
    typedef typename FileEntryList::StorageElement FileElement;
    NN_STATIC_ASSERT(sizeof(FileElement) == 96);

private:
    /**
    * @brief        指定した親ディレクトを取得します。
    *
    * @param[out]   outParentIndex              親ディレクトリインデックス
    * @param[out]   outParentDirectoryKey       親ディレクトリキー
    * @param[out]   outParentDirectoryEntry     親ディレクトリエントリー
    * @param[in]    parser                      パスパーサー
    *
    * @return       関数の処理結果を返します。
    * @retval       ResultSuccess                   正常に親ディレクトリが取得できました。
    * @retval       ResultNotInitialized            初期化されていません。
    * @retval       ResultTooLongPath               parser のディレクトリ名が長すぎます。
    * @retval       ResultIncompatiblePath          parser はディレクトリではなくファイルです。
    * @retval       ResultDirectoryNotFound         parser に紐づくディレクトリエントリーがありません。
    * @retval       ResultDirectoryUnobtainable     ルートディレクトリからは親ディレクトリを取得できません。
    * @retval       ResultInvalidOffset             内部データに問題があります。
    * @retval       ResultDatabaseCorrupted         内部データに問題があります。
    * @retval       上記以外                        ストレージからのデータ読み込みに失敗しました。
    *
    * @pre          マウントしている。
    * @pre          outParentIndex != nullptr
    * @pre          outParentDirectoryKey != nullptr
    * @pre          outParentDirectoryEntry != nullptr
    *
    * @details      指定した親ディレクトを取得します。
    */
    Result FindParentDirectoryRecursive(
               StorageIndex* outParentIndex,
               DirectoryKey* outParentDirectoryKey,
               DirectoryEntry* outParentDirectoryEntry,
               PathTool::PathParser& parser
           ) const NN_NOEXCEPT;

    /**
    * @brief        指定したエントリーと同名のファイル、ディレクトリが存在するかどうかを取得します。
    *
    * @param[out]   outIsFile                       同名のエントリーが存在したときに
    *                                               ファイルであれば true、ディレクトリであれば false
    * @param[in]    parentDir                       親ディレクトリインデックス
    * @param[in]    entryName                       エントリー名
    * @param[in]    nameChars                       エントリー名の長さ
    * @param[in]    resultIfExist                   同名のエントリーが存在したときに返すリザルト
    *
    * @return       関数の処理結果を返します。
    * @retval       ResultSuccess                   同名のエントリーは検出されませんでした。
    * @retval       ResultNotInitialized            初期化されていません。
    * @retval       ResultInvalidOffset             内部データに問題があります。
    * @retval       ResultDatabaseCorrupted         内部データに問題があります。
    * @retval       resultIfExist に渡した値         同名のファイルまたはディレクトリが存在します。
    * @retval       上記以外                        ストレージからのデータ読み込みに失敗しました。
    *
    * @pre          outIsFile != nullptr
    *
    * @details      指定したエントリーと同名のファイル、ディレクトリが存在するかどうかを取得します。
    */
    Result CheckSameEntryExists(
               bool* outIsFile,
               StorageIndex parentDir,
               const PathChar* entryName,
               size_t nameChars,
               Result resultIfExist
           ) const NN_NOEXCEPT;

    /**
    * @brief        ディレクトリエントリーを取得します。
    *
    * @param[out]   outIndex                    ディレクトリインデックス
    * @param[out]   outEntry                    ディレクトリエントリー
    * @param[in]    dirKey                      ディレクトリキー
    *
    * @return       関数の処理結果を返します。
    * @retval       ResultSuccess                   ディレクトリエントリーが取得できました。
    * @retval       ResultNotInitialized            初期化されていません。
    * @retval       ResultIncompatiblePath          key はディレクトリではなくファイルです。
    * @retval       ResultDirectoryNotFound         key に紐づくディレクトリエントリーがありません。
    * @retval       ResultInvalidOffset             内部データに問題があります。
    * @retval       ResultDatabaseCorrupted         内部データに問題があります。
    * @retval       上記以外                        ストレージからのデータ読み込みに失敗しました。
    *
    * @pre          outIndex != nullptr
    * @pre          outEntry != nullptr
    *
    * @details      ディレクトリエントリーを取得します。
    */
    Result GetDirectoryEntry(
               StorageIndex* outIndex,
               DirectoryEntry* outEntry,
               const DirectoryKey& dirKey
           ) const NN_NOEXCEPT;

    /**
    * @brief        ファイルエントリーを取得します。
    *
    * @param[out]   outIndex                    ファイルインデックス
    * @param[out]   outEntry                    ファイルエントリー
    * @param[in]    fileKey                     ファイルキー
    *
    * @return       関数の処理結果を返します。
    * @retval       ResultSuccess                   ファイルエントリーが取得できました。
    * @retval       ResultNotInitialized            初期化されていません。
    * @retval       ResultIncompatiblePath          key はファイルではなくディレクトリです。
    * @retval       ResultFileNotFound              key に紐づくファイルエントリーがありません。
    * @retval       ResultInvalidOffset             内部データに問題があります。
    * @retval       ResultDatabaseCorrupted         内部データに問題があります。
    * @retval       上記以外                        ストレージからのデータ読み込みに失敗しました。
    *
    * @pre          outIndex != nullptr
    * @pre          outEntry != nullptr
    *
    * @details      ファイルエントリーを取得します。
    */
    Result GetFileEntry(
               StorageIndex* outIndex,
               FileEntry* outEntry,
               const FileKey& fileKey
           ) const NN_NOEXCEPT;

    /**
    * @brief        指定したディレクトリエントリーへのリンクを削除します。
    *
    * @param[in]    pParentEntry        削除するエントリーの親ディレクトリエントリー
    * @param[in]    parentIndex         削除するエントリーの親ディレクトリインデックス
    * @param[in]    pDeleteEntry        削除するディレクトリエントリー
    * @param[in]    deleteIndex         削除するディレクトリインデックス
    *
    * @return       関数の処理結果を返します。
    * @retval       ResultSuccess                   正常に削除しました
    * @retval       ResultNotInitialized            初期化されていません。
    * @retval       ResultInvalidOffset             parentIndex が範囲外です。
    * @retval       ResultInvalidOffset             pParentEntry が範囲外です。
    * @retval       ResultInvalidOffset             内部データに問題があります。
    * @retval       ResultDatabaseCorrupted         内部データに問題があります。
    * @retval       上記以外                        ストレージでの読み書きに失敗しました。
    *
    * @pre          pParentEntry != nullptr
    * @pre          pDeleteEntry != nullptr
    * @pre          pParentEntry->indexDirectory != IndexNone
    *
    * @details      指定したディレクトリエントリーへのリンクを削除します。
    */
    Result RemoveDirectoryLink(
               DirectoryEntry* pParentEntry,
               StorageIndex parentIndex,
               DirectoryEntry* pDeleteEntry,
               StorageIndex deleteIndex
           ) NN_NOEXCEPT;

    /**
    * @brief        指定したファイルエントリーへのリンクを削除します。
    *
    * @param[in]    pParentEntry        削除するエントリーの親ディレクトリエントリー
    * @param[in]    parentIndex         削除するエントリーの親ディレクトリインデックス
    * @param[in]    pDeleteEntry        削除するファイルエントリー
    * @param[in]    deleteIndex         削除するファイルインデックス
    *
    * @return       関数の処理結果を返します。
    * @retval       ResultSuccess                   正常に削除しました。
    * @retval       ResultNotInitialized            初期化されていません。
    * @retval       ResultInvalidOffset             parentIndex が範囲外です。
    * @retval       ResultInvalidOffset             pParentEntry が範囲外です。
    * @retval       ResultInvalidOffset             内部データに問題があります。
    * @retval       ResultDatabaseCorrupted         内部データに問題があります。
    * @retval       上記以外                        ストレージでの読み書きに失敗しました。
    *
    * @pre          pParentEntry != nullptr
    * @pre          pDeleteEntry != nullptr
    * @pre          pParentEntry->indexFile != IndexNone
    *
    * @details      指定したファイルエントリーへのリンクを削除します。
    */
    Result RemoveFileLink(
               DirectoryEntry* pParentEntry,
               StorageIndex parentIndex,
               FileEntry* pDeleteEntry,
               StorageIndex deleteIndex
           ) NN_NOEXCEPT;

    /**
    * @brief        子ディレクトリかどうかを判定します。
    *
    * @param[in]    baseDirectoryKey        基準となるディレクトリ
    * @param[in]    dirKey                  探索するディレクトリ
    * @param[in]    resultIfSubDirectory    各ディレクトリが親子関係だった時に返すリザルト。
    *
    * @return       関数の処理結果を返します。
    * @retval       ResultSuccess                   baseDirectoryKey と dirKey の間に親子関係はありませんでした。
    * @retval       ResultNotInitialized            初期化されていません。
    * @retval       ResultInvalidOffset             dirKey が範囲外です。
    * @retval       ResultInvalidOffset             内部で不明な問題が発生しました。
    * @retval       ResultDatabaseCorrupted         内部データに問題があります。
    * @retval       resultIfSubDirectory            baseDirectoryKey と dirKey は親子関係にあります。
    * @retval       上記以外                        ストレージからのデータ読み込みに失敗しました。
    *
    * @details      子ディレクトリかどうかを判定します。
    *               直接の親子だけではなく、子孫と先祖の関係にあるかを調べます。
    *               また、baseDirectoryKey と dirKey が同一であるときも、resultIfSubDirectoryを返します。
    */
    Result CheckSubDirectory(
               const DirectoryKey& baseDirectoryKey,
               DirectoryKey dirKey,
               Result resultIfSubDirectory
           ) NN_NOEXCEPT;

private:
    DirectoryEntryList m_TableDirectory;                        //!< ディレクトリ管理テーブル
    FileEntryList m_TableFile;                                  //!< ファイル管理テーブル

private:
    // テスト用に private を公開します。
    friend class ::HierarchicalFileTableTest;
};

}}}

