﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <cstring>
#include <nn/fssystem/dbm/fs_DbmParameters.h>
#include <nn/fssystem/dbm/fs_HierarchicalFileTableTemplate.h>
#include <nn/fssystem/dbm/fs_FileSystemTemplate.h>
#include <nn/fssystem/dbm/fs_AllocationTable.h>

class FileObjectTestSuite;

namespace nn { namespace fssystem { namespace dbm {

template <typename TDirectoryName, typename TFileName>
class FileSystemObjectTemplate;

/**
* @brief ファイルオブジェクトです。
*
* @tparam   DirectoryName
* @tparam   FileName
*
* @details
*   @see nn::fssystem::dbm::FileSystemObject::OpenFile で初期化します。Finalize は不要です。
*
*   ファイルデータへのブロック単位イテレーションには、IterateBegin で初期ブロックインデックスを
*   取得し、以降は IterateNext、IteratePrevious で前後のブロックを取得することが出来ます。
*
*   制限事項
*
*   - OpenFileしているときに、削除されたファイルに対する操作は不定です。
*   - 複数個所からOpenFileしているときに、Resizeを呼んだときの動作は不定です。
*   - 複数個所からOpenFileしているときに、Set/GetOptionalInfoを呼んだときの動作は不定です。
*/
template <typename TDirectoryName, typename TFileName>
class FileObjectTemplate
{
    NN_DISALLOW_COPY(FileObjectTemplate);

public:
    /**
    * @brief        コンストラクタです。
    *
    * @details      コンストラクタです。
    */
    FileObjectTemplate() NN_NOEXCEPT;

    /**
    * @brief        ファイルサイズを取得します。
    *
    * @details      ファイルサイズを取得します。
    */
    inline int64_t GetSize() const NN_NOEXCEPT
    {
        return m_FileInfo.fs.size.Get();
    }

    /**
    * @brief        アロケーションテーブルでの開始位置を取得します。
    *
    * @details      アロケーションテーブルでの開始位置を取得します。
    */
    inline int64_t GetAllocationTableIndex() const NN_NOEXCEPT
    {
        return m_FileInfo.fs.index;
    }

    /**
    * @brief        ファイルの付加情報を取得します。
    *
    * @param[out]   outInfo            ファイルの付加情報
    *
    * @pre          outInfo != nullptr
    *
    * @details      ファイルの付加情報を取得します。
    */
    void GetOptionalInfo(FileOptionalInfo* outInfo) const NN_NOEXCEPT;

    /**
    * @brief        ファイルサイズを変更します。
    *
    * @param[in]    size                            変更後のファイルサイズ
    *
    * @return       関数の処理結果を返します。
    * @retval       ResultSuccess                   ファイルサイズが変更できました。
    * @retval       ResultNotInitialized            初期化されていません。
    * @retval       ResultIncompatiblePath          fileKey はファイルではなくディレクトリです。
    * @retval       ResultFileNotFound              fileKey に紐づくファイルエントリーがありません。
    * @retval       ResultOutOfRange                newSize が範囲外です。
    * @retval       ResultInvalidOffset             内部データに問題があります。
    * @retval       ResultDatabaseCorrupted         内部データに問題があります。
    * @retval       ResultAllocationTableFull       アロケーションテーブルに空きが無い。
    * @retval       上記以外                        ストレージでの読み書きに失敗しました。
    *
    * @pre          マウントしている。
    *
    * @details      ファイルサイズを変更します。
    *               あわせて領域の確保や開放処理をおこないます。
    */
    Result Resize(int64_t size) NN_NOEXCEPT;

    /**
    * @brief        ファイルデータ本体へのイテレーションを行います。
    *
    * @param[out]   outIndex            実データ領域でのインデックス
    * @param[out]   outFinished         イテレーションが完了したか
    * @param[in]    blockOffset         イテレーションを開始する先頭ブロックからのブロック数
    *
    * @return       関数の処理結果を返します。
    * @retval       ResultSuccess       正常に取得できました。
    * @retval       ResultOutOfRange    イテレータの値が不正です。
    * @retval       上記以外            ストレージからのデータ読み込みに失敗しました。
    *
    * @pre          outIndex != nullptr
    *
    * @details      ファイルの先頭から読む場合には blockOffset に 0 を指定します。
    */
    Result IterateBegin(
               uint32_t* outIndex,
               bool* outFinished,
               uint32_t blockOffset
           ) NN_NOEXCEPT;

    /**
    * @brief        ファイルデータの次のブロックを取得します。
    *
    * @param[out]   outIndex            実データ領域での次のブロックのインデックス
    * @param[out]   outFinished         イテレーションが完了したか
    * @param[in]    index               実データ領域でのブロックのインデックス
    *
    * @return       関数の処理結果を返します。
    * @retval       ResultSuccess       正常に取得できました。
    * @retval       ResultOutOfRange    イテレータの値が不正です。
    * @retval       上記以外            ストレージからのデータ読み込みに失敗しました。
    *
    * @pre          マウントしている。
    *
    * @details      ファイルデータの次のブロックを取得します。
    */
    Result IterateNext(
               uint32_t* outIndex,
               bool* outFinished,
               uint32_t index
           ) NN_NOEXCEPT;

    /**
    * @brief        ファイルデータの前のブロックを取得します。
    *
    * @param[out]   outIndex            実データ領域での前のブロックのインデックス
    * @param[out]   outFinished         イテレーションが完了したか
    * @param[in]    index               実データ領域でのブロックのインデックス
    *
    * @return       関数の処理結果を返します。
    * @retval       ResultSuccess       正常に取得できました。
    * @retval       ResultOutOfRange    内部データが範囲外をさしています。
    * @retval       上記以外            ストレージからのデータ読み込みに失敗しました。
    *
    * @pre          マウントしている。
    *
    * @details      ファイルデータの前のブロックを取得します。
    */
    Result IteratePrevious(
               uint32_t* outIndex,
               bool* outFinished,
               uint32_t index
           ) NN_NOEXCEPT;

private:
    typedef FileSystemObjectTemplate<TDirectoryName, TFileName> FileSystemObjectTempl;
    typedef typename FileSystemObjectTempl::FileKey FileKey;

private:
    /**
    * @brief        ファイルオブジェクトを初期化します。
    *
    * @param[in]    pFileSystem         ファイルシステム
    * @param[in]    key                 ファイルキー
    *
    * @pre          pFileSystem != nullptr
    *
    * @details      ファイルオブジェクトを初期化します。
    */
    void Initialize(FileSystemObjectTempl* pFileSystem, const FileKey& key) NN_NOEXCEPT;

    /**
    * @brief        ファイル情報を取得します。
    *
    * @return       ファイル情報。
    *
    * @details      ファイル情報を取得します。
    */
    inline detail::FileInfo& GetFileInfo() NN_NOEXCEPT
    {
        return m_FileInfo;
    }

private:
    FileSystemObjectTempl* m_pFileSystem;   //! ファイルシステム
    uint32_t m_CurrBlockCount;              //! イテレーション中の連続ブロック数
    detail::FileInfo m_FileInfo;            //! ファイル情報
    FileKey m_FileKey;                      //! ファイルキー
    uint32_t m_CurrIndex;                   //! イテレーション中のインデックス
    uint32_t m_NextIndex;                   //! 次の連続ブロックのインデックス

private:
    // ファイルシステムが初期化処理を呼び出せるようにします。
    friend class FileSystemObjectTemplate<TDirectoryName, TFileName>;

private:
    // テスト用に private を公開します。
    friend class FileObjectTestSuite;
};

}}}
