﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <mutex>
#include <nn/nn_Common.h>
#include <nn/nn_Allocator.h>
#include <nn/lmem/lmem_ExpHeap.h>
#include <nn/os/os_Mutex.h>

namespace nn { namespace fssrv {

    /**
       @brief nn::lmem::ExpHeap を使用する nn::MemoryResource の実装です。
       @details コンストラクタに初期化済みのヒープハンドルを渡す必要があります。
    */
    class MemoryResourceFromExpHeap : public nn::MemoryResource
    {
    public:
        explicit MemoryResourceFromExpHeap(nn::lmem::HeapHandle heapHandle) NN_NOEXCEPT;
        ~MemoryResourceFromExpHeap() NN_NOEXCEPT;
    protected:
        virtual void* do_allocate(std::size_t bytes, std::size_t alignment) NN_NOEXCEPT NN_OVERRIDE;
        virtual void do_deallocate(void* p, std::size_t bytes, std::size_t alignment) NN_NOEXCEPT NN_OVERRIDE;
        virtual bool do_is_equal(const nn::MemoryResource& other) const NN_NOEXCEPT NN_OVERRIDE
        {
            NN_UNUSED(other);
            return false;
        }
    private:
        nn::lmem::HeapHandle m_HeapHandle;
    };

    class PeakCheckableMemoryResourceFromExpHeap : public nn::MemoryResource
    {
    public:
        explicit PeakCheckableMemoryResourceFromExpHeap(std::size_t heapSize) NN_NOEXCEPT;
        ~PeakCheckableMemoryResourceFromExpHeap() NN_NOEXCEPT;
    public:
        void SetHeapHandle(nn::lmem::HeapHandle heapHandle) NN_NOEXCEPT
        {
            m_HeapHandle = heapHandle;
        }
        std::size_t GetFreeSizePeak() NN_NOEXCEPT
        {
            return m_FreeSizePeak;
        }
        std::size_t GetFreeSizeCurrent() NN_NOEXCEPT
        {
            return m_FreeSizeCurrent;
        }
        void ClearPeak() NN_NOEXCEPT;

        std::unique_lock<os::Mutex> GetScopedLock()
        {
            return std::unique_lock<os::Mutex>(m_Mutex);
        }

        void OnAllocate(void* p, std::size_t size) NN_NOEXCEPT;
        void OnDeallocate(void* p, std::size_t size) NN_NOEXCEPT;
    protected:
        virtual void* do_allocate(std::size_t bytes, std::size_t alignment) NN_NOEXCEPT NN_OVERRIDE;
        virtual void do_deallocate(void* p, std::size_t bytes, std::size_t alignment) NN_NOEXCEPT NN_OVERRIDE;
        virtual bool do_is_equal(const nn::MemoryResource& other) const NN_NOEXCEPT NN_OVERRIDE
        {
            NN_UNUSED(other);
            return false;
        }
    private:
        nn::lmem::HeapHandle m_HeapHandle;
        nn::os::Mutex m_Mutex;
        std::size_t m_FreeSizePeak;
        std::size_t m_FreeSizeCurrent;
    };

}}
