﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <algorithm>
#include <nn/nn_Common.h>
#include <nn/nn_Result.h>
#include <nn/fs/fs_File.h>
#include <nn/fs/fs_FileSystem.h>
#include <nn/fs/fs_Result.h>
#include <nn/fs/fs_ResultPrivate.h>
#include <nn/fs/fs_OperateRange.h>
#include <nn/fssystem/fs_Assert.h>

namespace nn { namespace fs { namespace fsa {

    /**
    * @brief    ファイル操作のためのインターフェースです。
    *
    * @details  読み込み関数（Read・GetSize）同士が複数スレッドから同時に呼び出された場合に正しく動作するよう実装する必要があります。
    *           書き込み関数（Write・Flush・SetSize）同士および読み込み関数が複数スレッドから同時に呼び出された場合の動作は未定義でかまいません。
    */
    class IFile
    {
    public:

        Result Read(size_t* outValue, int64_t offset, void* buffer, size_t size, const ReadOption& option) NN_NOEXCEPT
        {
            NN_FSP_REQUIRES( outValue != nullptr, nn::fs::ResultNullptrArgument() );
            if( size == 0 )
            {
                *outValue = 0;
                NN_RESULT_SUCCESS;
            }
            NN_FSP_REQUIRES( buffer   != nullptr, nn::fs::ResultNullptrArgument() );
            NN_FSP_REQUIRES( offset >= 0, nn::fs::ResultOutOfRange()); // TODO: ResultInvalidOffset?
            int64_t castedSize = static_cast<int64_t>(size);
            NN_FSP_REQUIRES(castedSize >= 0, nn::fs::ResultOutOfRange()); // TODO: ResultInvalidSize?
            NN_FSP_REQUIRES((INT64_MAX - offset) >= castedSize, nn::fs::ResultOutOfRange()); // TODO: ResultInvalidSize?
            return DoRead(outValue, offset, buffer, size, option);
        }

        Result Write(int64_t offset, const void* buffer, size_t size, const WriteOption& option) NN_NOEXCEPT
        {
            if( size == 0 )
            {
                if( (option.flags & nn::fs::WriteOptionFlag_Flush) != 0 )
                {
                    NN_RESULT_DO(Flush());
                }
                NN_RESULT_SUCCESS;
            }
            NN_FSP_REQUIRES( buffer != nullptr, nn::fs::ResultNullptrArgument() );
            NN_FSP_REQUIRES( offset >= 0, nn::fs::ResultOutOfRange()); // TODO: ResultInvalidOffset?
            int64_t castedSize = static_cast<int64_t>(size);
            NN_FSP_REQUIRES(castedSize >= 0, nn::fs::ResultOutOfRange()); // TODO: ResultInvalidSize?
            NN_FSP_REQUIRES((INT64_MAX - offset) >= castedSize, nn::fs::ResultOutOfRange()); // TODO: ResultInvalidSize?
            return DoWrite(offset, buffer, size, option);
        }

        Result Flush() NN_NOEXCEPT
        {
            return DoFlush();
        }

        Result SetSize(int64_t size) NN_NOEXCEPT
        {
            NN_FSP_REQUIRES(size >= 0, nn::fs::ResultOutOfRange()); // TODO: ResultInvalidSize?
            return DoSetSize(size);
        }

        Result GetSize(int64_t* outValue) NN_NOEXCEPT
        {
            NN_FSP_REQUIRES( outValue != nullptr, nn::fs::ResultNullptrArgument() );
            return DoGetSize(outValue);
        }

        Result OperateRange(
            void* outBuffer,
            size_t outBufferSize,
            OperationId operationId,
            int64_t offset,
            int64_t size,
            const void* inBuffer,
            size_t inBufferSize) NN_NOEXCEPT
        {
            return DoOperateRange(
                outBuffer, outBufferSize, operationId, offset, size, inBuffer, inBufferSize);
        }

        Result OperateRange(
            OperationId operationId,
            int64_t offset,
            int64_t size) NN_NOEXCEPT
        {
            return DoOperateRange(nullptr, 0, operationId, offset, size, nullptr, 0);
        }

    public:
        /**
        * @brief    nn::fs::CloseFile に応じて呼び出されるデストラクタです。
        *
        * @details  実装クラスはファイルのクローズ処理をデストラクタで実装してください。
        */
        virtual ~IFile() NN_NOEXCEPT {}

    protected:
        Result DryRead(size_t* outValue, int64_t offset, size_t size, const ReadOption& option, nn::fs::OpenMode mode) NN_NOEXCEPT
        {
            // NOTE: outValue, offset, size は Read() で事前条件をクリアしているものとする。
            NN_UNUSED(option);
            NN_FSP_REQUIRES((mode & nn::fs::OpenMode::OpenMode_Read) != 0, nn::fs::ResultInvalidOperationForOpenMode());
            int64_t fileSize = 0;
            NN_RESULT_DO(GetSize(&fileSize));
            NN_FSP_REQUIRES(offset <= fileSize, ResultOutOfRange());
            const auto readableSize = fileSize - offset;
            *outValue = static_cast<size_t>(std::min<int64_t>(readableSize, size));
            NN_RESULT_SUCCESS;
        }

        Result DrySetSize(int64_t size, nn::fs::OpenMode mode) NN_NOEXCEPT
        {
            NN_FSP_REQUIRES((mode & nn::fs::OpenMode_Write) != 0, nn::fs::ResultInvalidOperationForOpenMode());

            // SetSize でチェックされている
            NN_UNUSED(size);
            NN_SDK_ASSERT_GREATER_EQUAL(size, 0);

            NN_RESULT_SUCCESS;
        }

    private:
        /**
        * @brief    nn::fs::ReadFile に応じて呼び出されるファイル読み込み関数です。
        *
        * @details  実装クラスはこの関数の実装でファイルの読み込み処理を実装してください。
        *           nn::fs::ReadFile の FileHandle 以外の引数が委譲されます。
        */
        virtual Result DoRead(size_t* outValue, int64_t offset, void* buffer, size_t size, const ReadOption& option) NN_NOEXCEPT = 0;

        /**
        * @brief    nn::fs::WriteFile に応じて呼び出されるファイル書き込み関数です。
        *
        * @details  実装クラスはこの関数の実装でファイルの書き込み処理を実装してください。
        *           nn::fs::WriteFile の FileHandle 以外の引数が委譲されます。
        */
        virtual Result DoWrite(int64_t offset, const void* buffer, size_t size, const WriteOption& option) NN_NOEXCEPT = 0;

        /**
        * @brief    nn::fs::FlushFile に応じて呼び出されるファイルフラッシュ関数です。
        *
        * @details  実装クラスはこの関数の実装でファイルのフラッシュを実装してください。
        */
        virtual Result DoFlush() NN_NOEXCEPT = 0;

        /**
        * @brief    nn::fs::SetFileSize に応じて呼び出されるファイルサイズ変更関数です。
        *
        * @details  実装クラスはこの関数の実装でファイルのフラッシュを実装してください。
        *           nn::fs::SetFileSize の FileHandle 以外の引数が委譲されます。
        */
        virtual Result DoSetSize(int64_t size) NN_NOEXCEPT = 0;

        /**
        * @brief    nn::fs::GetFileSize に応じて呼び出されるファイルサイズ取得関数です。
        *
        * @details  実装クラスはこの関数の実装でファイルのフラッシュを実装してください。
        *           nn::fs::GetFileSize の FileHandle 以外の引数が委譲されます。
        */
        virtual Result DoGetSize(int64_t* outValue) NN_NOEXCEPT = 0;

        /**
        * @brief    範囲操作関数です。
        *
        * @details  実装クラスはこの関数の実装で範囲操作を実装してください。
        */
        virtual Result DoOperateRange(
            void* outBuffer,
            size_t outBufferSize,
            OperationId operationId,
            int64_t offset,
            int64_t size,
            const void* inBuffer,
            size_t inBufferSize) NN_NOEXCEPT = 0;
    };

}}}
