﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
*   @file
*   @brief  セーブデータ管理 API の宣言
*/

#pragma once

#include <nn/nn_Common.h>
#include <nn/nn_Result.h>
#include <nn/fs/fs_SaveDataTypes.h>
#include <nn/fs/fs_SaveDataManagement.h>

namespace nn { namespace time {
    struct PosixTime;
}}

namespace nn { namespace fs {

using SaveDataCommitId = int64_t;

//--------------------------------------------------------------------------
/**
*   @brief      次回起動時にアトミックに削除するセーブデータを登録します。
*
*   @param[in]  saveDataIdArray      セーブデータ識別子の配列
*   @param[in]  saveDataIdArrayCount saveDataIdArray の要素数
*
*   @return     処理の結果が返ります。
*   @retval     ResultSuccess               成功しました。
*
*   @pre
*       - saveDataIdArray != nullptr
*/
Result RegisterSaveDataAtomicDeletion(SaveDataId* saveDataIdArray, int saveDataIdArrayCount) NN_NOEXCEPT;

//--------------------------------------------------------------------------
/**
*   @brief      セーブデータのファイルシステム上での占有サイズを取得します。
*
*   @param[out] outValue             セーブデータのファイルシステム上での占有バイトサイズ
*   @param[in]  saveDataSize         セーブデータのデータ保存領域のバイトサイズ
*   @param[in]  saveDataJournalSize  セーブデータのジャーナリング領域のバイトサイズ
*
*   @return     処理の結果が返ります。
*   @retval     ResultSuccess                   成功しました。
*/
Result QuerySaveDataTotalSize(int64_t* outValue, int64_t saveDataSize, int64_t saveDataJournalSize);


//--------------------------------------------------------------------------
/**
*   @brief      セーブデータの所有者の識別子を取得します。
*
*   @param[out] outValue        所有者の識別子
*   @param[in]  saveDataSpaceId セーブデータの配置場所の識別子
*   @param[in]  saveDataId      セーブデータの識別子
*
*   @return     処理の結果が返ります。
*   @retval     ResultSuccess               成功しました。
*   @retval     ResultTargetNotFound        指定したセーブデータが存在しません。
*
*   @pre
*       - outValue が有効なメモリを指している。
*/
Result GetSaveDataOwnerId(Bit64* outValue, SaveDataSpaceId saveDataSpaceId, SaveDataId saveDataId) NN_NOEXCEPT;

//--------------------------------------------------------------------------
/**
*   @brief      セーブデータに設定されているフラグを取得します。
*
*   @param[out] outValue        セーブデータのフラグ
*   @param[in]  saveDataSpaceId セーブデータの配置場所の識別子
*   @param[in]  saveDataId      セーブデータの識別子
*
*   @return     処理の結果が返ります。
*   @retval     ResultSuccess               成功しました。
*   @retval     ResultTargetNotFound        指定したセーブデータが存在しません。
*
*   @pre
*       - outValue が有効なメモリを指している。
*/
Result GetSaveDataFlags(uint32_t* outValue, SaveDataSpaceId saveDataSpaceId, SaveDataId saveDataId) NN_NOEXCEPT;

//--------------------------------------------------------------------------
/**
*   @brief      セーブデータのタイムスタンプを取得します。
*
*   @param[out] outValue        セーブデータの更新時刻を指す POSIX 時間
*   @param[in]  saveDataSpaceId セーブデータの配置場所の識別子
*   @param[in]  saveDataId      セーブデータの識別子
*
*   @return     処理の結果が返ります。
*   @retval     ResultSuccess               成功しました。
*   @retval     ResultTargetNotFound        指定したセーブデータが存在しません。
*
*   @pre
*       - outValue が有効なメモリを指している。
*/
Result GetSaveDataTimeStamp(nn::time::PosixTime* outValue, SaveDataSpaceId saveDataSpaceId, SaveDataId saveDataId) NN_NOEXCEPT;

//--------------------------------------------------------------------------
/**
*   @brief      セーブデータのタイムスタンプを設定します。
*
*   @param[in]  saveDataSpaceId   セーブデータの配置場所の識別子
*   @param[in]  saveDataId        セーブデータの識別子
*   @param[in]  saveDataTimeStamp セーブデータの更新時刻を指す POSIX 時間
*
*   @return     処理の結果が返ります。
*   @retval     ResultSuccess               成功しました。
*   @retval     ResultTargetNotFound        指定したセーブデータが存在しません。
*
*/
Result SetSaveDataTimeStamp(SaveDataSpaceId saveDataSpaceId, SaveDataId saveDataId, const nn::time::PosixTime& saveDataTimeStamp) NN_NOEXCEPT;

//--------------------------------------------------------------------------
/**
*   @brief      セーブデータのデータ保存領域のサイズを取得します。
*
*   @param[out] outValue        セーブデータのデータ保存領域のバイトサイズ
*   @param[in]  saveDataSpaceId セーブデータの配置場所の識別子
*   @param[in]  saveDataId      セーブデータの識別子
*
*   @return     処理の結果が返ります。
*   @retval     ResultSuccess               成功しました。
*   @retval     ResultTargetNotFound        指定したセーブデータが存在しません。
*
*/
Result GetSaveDataAvailableSize(int64_t* outValue, SaveDataSpaceId saveDataSpaceId, SaveDataId saveDataId) NN_NOEXCEPT;

//--------------------------------------------------------------------------
/**
*   @brief      セーブデータのジャーナリング領域のサイズを取得します。
*
*   @param[out] outValue        セーブデータのジャーナリング領域のバイトサイズ
*   @param[in]  saveDataSpaceId セーブデータの配置場所の識別子
*   @param[in]  saveDataId      セーブデータの識別子
*
*   @return     処理の結果が返ります。
*   @retval     ResultSuccess               成功しました。
*   @retval     ResultTargetNotFound        指定したセーブデータが存在しません。
*
*/
Result GetSaveDataJournalSize(int64_t* outValue, SaveDataSpaceId saveDataSpaceId, SaveDataId saveDataId) NN_NOEXCEPT;

//--------------------------------------------------------------------------
/**
*   @brief      セーブデータのコミット識別子を取得します。
*
*   @param[out] outValue        セーブデータのコミット識別子
*   @param[in]  saveDataSpaceId セーブデータ空間の識別子
*   @param[in]  saveDataId      セーブデータの識別子
*
*   @return     処理の結果が返ります。
*   @retval     ResultSuccess               成功しました。
*   @retval     ResultTargetNotFound        指定したセーブデータが存在しません。
*
*   @pre
*       - outValue が有効なメモリを指している。
*/
Result GetSaveDataCommitId(SaveDataCommitId* outValue, SaveDataSpaceId saveDataSpaceId, SaveDataId saveDataId) NN_NOEXCEPT;

//--------------------------------------------------------------------------
/**
*   @brief      セーブデータのコミット識別子を設定します。
*
*   @param[in]  saveDataSpaceId セーブデータ空間の識別子
*   @param[in]  saveDataId      セーブデータの識別子
*   @param[in]  commitId        セーブデータのコミット識別子
*
*   @return     処理の結果が返ります。
*   @retval     ResultSuccess               成功しました。
*   @retval     ResultTargetNotFound        指定したセーブデータが存在しません。
*
*/
Result SetSaveDataCommitId(SaveDataSpaceId saveDataSpaceId, SaveDataId saveDataId, const SaveDataCommitId& saveDataCommitId) NN_NOEXCEPT;

//--------------------------------------------------------------------------
/**
*   @brief      アプリケーション、所有者を指定して一時ストレージを作成します。
*
*   @param[in]  applicationId       アプリケーションの識別子
*   @param[in]  saveDataOwnerId     所有者の識別子
*   @param[in]  size                一時ストレージのバイトサイズ
*   @param[in]  flags               一時ストレージのフラグ集合
*
*   @pre
*       - size の値が有効範囲内である
*
*   @return     処理の結果が返ります。
*   @retval     ResultSuccess               成功しました。
*
*/
Result CreateTemporaryStorage(nn::ncm::ApplicationId applicationId, uint64_t saveDataOwnerId, int64_t size, uint32_t flags) NN_NOEXCEPT;

//--------------------------------------------------------------------------
/**
*   @brief      フィルタ情報に該当するセーブデータを取得します。
*
*   @param[out] pOutValue       セーブデータ情報の格納先バッファ
*   @param[in]  saveDataSpaceId セーブデータの配置場所の識別子
*   @param[in]  filter          セーブデータのフィルタ情報
*
*   @return     処理の結果が返ります。
*   @retval     ResultSuccess               成功しました。
*   @retval     ResultTargetNotFound        フィルタ情報に該当するセーブデータが存在しません。
*
*   @pre
*       - pOutValue が有効なメモリを指している。
*/
Result FindSaveDataWithFilter(SaveDataInfo* pOutValue, SaveDataSpaceId saveDataSpaceId, const SaveDataFilter& filter) NN_NOEXCEPT;

Result OpenSaveDataIterator(std::unique_ptr<SaveDataIterator>* outValue, SaveDataSpaceId saveDataSpaceId, const SaveDataFilter& filter) NN_NOEXCEPT;

Result QuerySaveDataInternalStorageTotalSize(int64_t* outValue, SaveDataSpaceId saveDataSpaceId, SaveDataId saveDataId) NN_NOEXCEPT;

namespace detail {
    Result ReadSaveDataFileSystemExtraData(SaveDataExtraData *outValue, SaveDataSpaceId saveDataSpaceId, SaveDataId saveDataId) NN_NOEXCEPT;
    Result WriteSaveDataFileSystemExtraData(SaveDataSpaceId spaceId, SaveDataId saveDataId, const SaveDataExtraData& extraDataValue) NN_NOEXCEPT;
    Result WriteSaveDataFileSystemExtraData(SaveDataSpaceId spaceId, SaveDataId saveDataId, const SaveDataExtraData& extraDataValue, const SaveDataExtraData& extraDataMask) NN_NOEXCEPT;
}


//! @}

// セーブデータ Raw イメージのエクスポート・インポート用定義

struct RawSaveDataInfo
{
    nn::ncm::ApplicationId applicationId;
    UserId userId;
    Bit64 ownerId;
    int64_t availableSize;
    int64_t journalSize;
    uint32_t flags;
    SaveDataType type;
    uint8_t padding[3];
    SaveDataId id;
};

const char RawSaveDataInfoFileName[] = "__BackupSaveDataInfo";

Result UpdateSaveDataMacForDebug(uint8_t saveDataSpaceId, uint64_t saveDataId) NN_NOEXCEPT;

}} // namespace nn::fs

