﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
*   @file
*   @brief  セーブデータ管理 API の宣言
*/

#pragma once

#include <nn/nn_Common.h>
#include <nn/nn_Result.h>
#include <nn/fs/fs_SaveDataTypes.h>
#include <nn/fs/detail/fs_Newable.h>
#include <nn/sf/sf_ISharedObject.h>

namespace nn { namespace fssrv { namespace sf {
    class ISaveDataInfoReader;
}}}

namespace nn { namespace time {
    struct PosixTime;
}}

namespace nn { namespace ncm {
    struct ApplicationId;
}}

namespace nn { namespace fs {

/**
* @brief セーブデータ情報のイテレータです。
*/
class SaveDataIterator : public detail::Newable
{
    NN_DISALLOW_COPY(SaveDataIterator);
public:
    NN_IMPLICIT SaveDataIterator(nn::sf::SharedPointer<nn::fssrv::sf::ISaveDataInfoReader>&& pReader) NN_NOEXCEPT;

    /**
    *   @brief      セーブデータ情報を列挙します。
    *
    *   @param[out] outCount         格納されたセーブデータ情報の個数
    *   @param[in]  outInfoBuffer    セーブデータ情報の格納先バッファ
    *   @param[in]  infoBufferCount  outInfoBuffer の要素数
    *
    *   @return     処理の結果が返ります。
    *   @retval     ResultSuccess               成功しました。
    *
    *   @pre
    *       - outCount が有効なメモリを指している。
    *       - outInfoBuffer が有効なメモリを指している。
    */
    Result ReadSaveDataInfo(int64_t* outCount, SaveDataInfo* outInfoBuffer, int64_t infoBufferCount) NN_NOEXCEPT;

private:
    nn::sf::SharedPointer<nn::fssrv::sf::ISaveDataInfoReader> m_pReader;
};

//! @name セーブデータ管理 API
//! @{

//--------------------------------------------------------------------------
/**
*   @brief      セーブデータを削除します。
*
*   @param[in]  saveDataId      セーブデータの識別子
*
*   @return     処理の結果が返ります。
*   @retval     ResultSuccess               成功しました。
*   @retval     ResultTargetNotFound        指定したセーブデータが存在しません。
*/
Result DeleteSaveData(SaveDataId saveDataId) NN_NOEXCEPT;

//--------------------------------------------------------------------------
/**
*   @brief      指定した配置場所のセーブデータを削除します。
*
*   @param[in]  saveDataSpaceId セーブデータの配置場所の識別子
*   @param[in]  saveDataId      セーブデータの識別子
*
*   @return     処理の結果が返ります。
*   @retval     ResultSuccess               成功しました。
*   @retval     ResultTargetNotFound        指定したセーブデータが存在しません。
*/
Result DeleteSaveData(nn::fs::SaveDataSpaceId saveDataSpaceId, nn::fs::SaveDataId saveDataId) NN_NOEXCEPT;

//--------------------------------------------------------------------------
/**
*   @brief      セーブデータの所有者の識別子を取得します。
*
*   @param[out] outValue   所有者の識別子
*   @param[in]  saveDataId セーブデータの識別子
*
*   @return     処理の結果が返ります。
*   @retval     ResultSuccess               成功しました。
*   @retval     ResultTargetNotFound        指定したセーブデータが存在しません。
*
*   @pre
*       - outValue が有効なメモリを指している。
*/
Result GetSaveDataOwnerId(Bit64* outValue, SaveDataId saveDataId) NN_NOEXCEPT;

//--------------------------------------------------------------------------
/**
*   @brief      セーブデータに設定されているフラグを取得します。
*
*   @param[out] outValue   セーブデータのフラグ
*   @param[in]  saveDataId セーブデータの識別子
*
*   @return     処理の結果が返ります。
*   @retval     ResultSuccess               成功しました。
*   @retval     ResultTargetNotFound        指定したセーブデータが存在しません。
*
*   @pre
*       - outValue が有効なメモリを指している。
*/
Result GetSaveDataFlags(uint32_t* outValue, SaveDataId saveDataId) NN_NOEXCEPT;

//--------------------------------------------------------------------------
/**
*   @brief      セーブデータにフラグを設定します。
*
*   @param[in]  saveDataId      セーブデータの識別子
*   @param[in]  saveDataSpaceId セーブデータ空間の識別子
*   @param[in]  flags           セーブデータのフラグ
*
*   @return     処理の結果が返ります。
*   @retval     ResultSuccess               成功しました。
*   @retval     ResultTargetNotFound        指定したセーブデータが存在しません。
*
*/
Result SetSaveDataFlags(SaveDataId saveDataId, SaveDataSpaceId saveDataSpaceId, uint32_t flags) NN_NOEXCEPT;

//--------------------------------------------------------------------------
/**
*   @brief      セーブデータのタイムスタンプを取得します。
*
*   @param[out] outValue   セーブデータの更新時刻を指す POSIX 時間
*   @param[in]  saveDataId セーブデータの識別子
*
*   @return     処理の結果が返ります。
*   @retval     ResultSuccess               成功しました。
*   @retval     ResultTargetNotFound        指定したセーブデータが存在しません。
*
*   @details    セーブデータタイムスタンプが無効値である場合、取得には成功し outValue には 0 が格納されます。
*
*   @pre
*       - outValue が有効なメモリを指している。
*/
Result GetSaveDataTimeStamp(nn::time::PosixTime* outValue, SaveDataId saveDataId) NN_NOEXCEPT;

//--------------------------------------------------------------------------
/**
*   @brief      セーブデータのデータ保存領域のサイズを取得します。
*
*   @param[out] outValue   セーブデータのデータ保存領域のバイトサイズ
*   @param[in]  saveDataId セーブデータの識別子
*
*   @return     処理の結果が返ります。
*   @retval     ResultSuccess               成功しました。
*   @retval     ResultTargetNotFound        指定したセーブデータが存在しません。
*
*/
Result GetSaveDataAvailableSize(int64_t* outValue, SaveDataId saveDataId) NN_NOEXCEPT;

//--------------------------------------------------------------------------
/**
*   @brief      セーブデータのジャーナリング領域のサイズを取得します。
*
*   @param[out] outValue   セーブデータのジャーナリング領域のバイトサイズ
*   @param[in]  saveDataId セーブデータの識別子
*
*   @return     処理の結果が返ります。
*   @retval     ResultSuccess               成功しました。
*   @retval     ResultTargetNotFound        指定したセーブデータが存在しません。
*
*/
Result GetSaveDataJournalSize(int64_t* outValue, SaveDataId saveDataId) NN_NOEXCEPT;

//--------------------------------------------------------------------------
/**
*   @brief      アプリケーション、ユーザー、所有者を指定してセーブデータを作成します。
*
*   @param[in]  applicationId       アプリケーションの識別子
*   @param[in]  userId              ユーザーの識別子
*   @param[in]  saveDataOwnerId     所有者の識別子
*   @param[in]  size                セーブデータのデータ保存領域のバイトサイズ
*   @param[in]  journalSize         セーブデータのジャーナリング領域のバイトサイズ
*   @param[in]  flags               セーブデータのフラグ集合
*
*   @pre
*       - size, journalSize の値が有効範囲内である
*
*   @return     処理の結果が返ります。
*   @retval     ResultSuccess               成功しました。
*
*   @pre
*       - size >= 0
*       - journalSize >= 0
*/
Result CreateSaveData(nn::ncm::ApplicationId applicationId, UserId userId, uint64_t saveDataOwnerId, int64_t size, int64_t journalSize, uint32_t flags) NN_NOEXCEPT;

//--------------------------------------------------------------------------
/**
*   @brief      アプリケーション、所有者を指定して本体セーブデータを作成します。
*
*   @param[in]  applicationId       アプリケーションの識別子
*   @param[in]  saveDataOwnerId     所有者の識別子
*   @param[in]  size                セーブデータのデータ保存領域のバイトサイズ
*   @param[in]  journalSize         セーブデータのジャーナリング領域のバイトサイズ
*   @param[in]  flags               セーブデータのフラグ集合
*
*   @pre
*       - size, journalSize の値が有効範囲内である
*
*   @return     処理の結果が返ります。
*   @retval     ResultSuccess               成功しました。
*
*   @pre
*       - size >= 0
*       - journalSize >= 0
*/
Result CreateDeviceSaveData(nn::ncm::ApplicationId applicationId, uint64_t saveDataOwnerId, int64_t size, int64_t journalSize, uint32_t flags) NN_NOEXCEPT;


//--------------------------------------------------------------------------
/**
*   @brief      アプリケーションを指定して bcat セーブデータを作成します。
*
*   @param[in]  applicationId       アプリケーションの識別子
*   @param[in]  size                セーブデータのデータ保存領域のバイトサイズ
*
*   @pre
*       - size の値が有効である
*
*   @return     処理の結果が返ります。
*   @retval     ResultSuccess               成功しました。
*
*   @pre
*       - size >= 0
*/
Result CreateBcatSaveData(nn::ncm::ApplicationId applicationId, int64_t size) NN_NOEXCEPT;


//--------------------------------------------------------------------------
/**
*   @brief      アプリケーション、所有者を指定して本体保存メモリーにキャッシュストレージを作成します。
*
*   @param[in]  applicationId       アプリケーションの識別子
*   @param[in]  saveDataOwnerId     所有者の識別子
*   @param[in]  size                キャッシュストレージのデータ保存領域のバイトサイズ
*   @param[in]  journalSize         キャッシュストレージのジャーナリング領域のバイトサイズ
*   @param[in]  flags               セーブデータのフラグ集合
*
*   @pre
*       - size, journalSize の値が有効範囲内である
*
*   @return     処理の結果が返ります。
*   @retval     ResultSuccess               成功しました。
*
*/
Result CreateCacheStorage(nn::ncm::ApplicationId applicationId, uint64_t saveDataOwnerId, int64_t size, int64_t journalSize, uint32_t flags) NN_NOEXCEPT;

//--------------------------------------------------------------------------
/**
*   @brief      アプリケーション、所有者を指定してキャッシュストレージを作成します。
*
*   @param[in]  applicationId       アプリケーションの識別子
*   @param[in]  spaceId             キャッシュストレージの配置場所
*   @param[in]  saveDataOwnerId     所有者の識別子
*   @param[in]  size                キャッシュストレージのデータ保存領域のバイトサイズ
*   @param[in]  journalSize         キャッシュストレージのジャーナリング領域のバイトサイズ
*   @param[in]  flags               セーブデータのフラグ集合
*
*   @pre
*       - size, journalSize の値が有効範囲内である
*
*   @return     処理の結果が返ります。
*   @retval     ResultSuccess               成功しました。
*
*/
Result CreateCacheStorage(nn::ncm::ApplicationId applicationId, nn::fs::SaveDataSpaceId spaceId, uint64_t saveDataOwnerId, int64_t size, int64_t journalSize, uint32_t flags) NN_NOEXCEPT;

//--------------------------------------------------------------------------
/**
*   @brief      アプリケーション、所有者を指定してキャッシュストレージを作成します。
*
*   @param[in]  applicationId       アプリケーションの識別子
*   @param[in]  spaceId             キャッシュストレージの配置場所
*   @param[in]  saveDataOwnerId     所有者の識別子
*   @param[in]  index               キャッシュストレージのインデックス
*   @param[in]  size                キャッシュストレージのデータ保存領域のバイトサイズ
*   @param[in]  journalSize         キャッシュストレージのジャーナリング領域のバイトサイズ
*   @param[in]  flags               セーブデータのフラグ集合
*
*   @pre
*       - size, journalSize の値が有効範囲内である
*
*   @return     処理の結果が返ります。
*   @retval     ResultSuccess               成功しました。
*
*/
Result CreateCacheStorage(nn::ncm::ApplicationId applicationId, SaveDataSpaceId spaceId, uint64_t saveDataOwnerId, uint16_t index, int64_t size, int64_t journalSize, uint32_t flags) NN_NOEXCEPT;

//--------------------------------------------------------------------------
/**
*   @brief      セーブデータを拡張します。
*
*   @param[in]  spaceId                 セーブデータの配置場所の識別子
*   @param[in]  saveDataId              セーブデータの識別子
*   @param[in]  saveDataAvailableSize   拡張後のセーブデータのデータ保存領域のバイトサイズ
*   @param[in]  saveDataJournalSize     拡張後のセーブデータのジャーナリング領域のバイトサイズ
*
*   @return     処理の結果が返ります。
*   @retval     ResultSuccess               成功しました。
*   @retval     ResultTargetLocked          セーブデータが使用中です。
*   @retval     ResultTargetNotFound        指定したセーブデータが存在しません。
*
*   @pre
*       - saveDataAvailableSize, saveDataJournalSize の値が有効範囲内である
*/
Result ExtendSaveData(
    nn::fs::SaveDataSpaceId spaceId,
    nn::fs::SaveDataId saveDataId,
    int64_t saveDataAvailableSize,
    int64_t saveDataJournalSize) NN_NOEXCEPT;

//--------------------------------------------------------------------------
/**
*   @brief      セーブデータ情報のイテレータをオープンします。
*
*   @param[out] outValue        イテレータ
*   @param[in]  saveDataSpaceId セーブデータ空間の識別子
*
*   @return     処理の結果が返ります。
*   @retval     ResultSuccess               成功しました。
*
*   @pre
*       - outValue が有効なメモリを指している。
*/
Result OpenSaveDataIterator(std::unique_ptr<SaveDataIterator>* outValue, SaveDataSpaceId saveDataSpaceId) NN_NOEXCEPT;


//--------------------------------------------------------------------------
/**
*   @brief      セーブデータを検証します。
*
*   @param[out] pOutIsValid     セーブデータの検証結果
*   @param[in]  saveDataId      セーブデータの識別子
*   @param[in]  workBuffer      ワークバッファ
*   @param[in]  workBufferSize  workBuffer のサイズ
*
*   @return     処理の結果が返ります。
*   @retval     ResultSuccess                   成功しました。
*   @retval     ResultTargetNotFound            指定したセーブデータが存在しません。
*   @retval     ResultAllocationMemoryFailed    メモリが不足しています。
*
*   @details    対象に破損が見られなかった場合、pOutIsValid に true が返ります。対象が破損していた場合、false が返ります。
*
*   @pre
*           - 対象のセーブデータがマウントされていない
*/
Result VerifySaveData(bool* pOutIsValid, SaveDataId saveDataId, void* workBuffer, size_t workBufferSize) NN_NOEXCEPT;

//--------------------------------------------------------------------------
/**
*   @brief      配置場所を指定してセーブデータを検証します。
*
*   @param[out] pOutIsValid     セーブデータの検証結果
*   @param[in]  spaceId         セーブデータの配置場所の識別子
*   @param[in]  saveDataId      セーブデータの識別子
*   @param[in]  workBuffer      ワークバッファ
*   @param[in]  workBufferSize  workBuffer のサイズ
*
*   @return     処理の結果が返ります。
*   @retval     ResultSuccess                   成功しました。
*   @retval     ResultTargetNotFound            指定したセーブデータが存在しません。
*   @retval     ResultAllocationMemoryFailed    メモリが不足しています。
*
*   @details    対象に破損が見られなかった場合、pOutIsValid に true が返ります。対象が破損していた場合、false が返ります。
*
*   @pre
*           - 対象のセーブデータがマウントされていない
*/
Result VerifySaveData(bool* pOutIsValid, SaveDataSpaceId spaceId, SaveDataId saveDataId, void* workBuffer, size_t workBufferSize) NN_NOEXCEPT;


//--------------------------------------------------------------------------
/**
*   @brief      セーブデータを破壊します。
*
*   @param[in]  saveDataId      セーブデータの識別子
*
*   @return     処理の結果が返ります。
*   @retval     ResultSuccess                   成功しました。
*   @retval     ResultTargetNotFound            指定したセーブデータが存在しません。
*   @retval     ResultDataCorrupted             ファイルシステムが壊れています。
*   @retval     ResultAllocationMemoryFailed    メモリが不足しています。
*
*   @details    この API はデバッグ用途に限り使用可能です。@n
*               この API で破壊したセーブデータは、マウント API が ResultDataCorrupted で失敗し、 nn::fs::VerifySaveData() の検証結果が false になります。
*
*   @pre
*           - 対象のセーブデータがマウントされていない
*/
Result CorruptSaveDataForDebug(SaveDataId saveDataId) NN_NOEXCEPT;

//--------------------------------------------------------------------------
/**
*   @brief      配置場所を指定してセーブデータを破壊します。
*
*   @param[in]  spaceId         セーブデータの配置場所の識別子
*   @param[in]  saveDataId      セーブデータの識別子
*
*   @return     処理の結果が返ります。
*   @retval     ResultSuccess                   成功しました。
*   @retval     ResultTargetNotFound            指定したセーブデータが存在しません。
*   @retval     ResultDataCorrupted             ファイルシステムが壊れています。
*   @retval     ResultAllocationMemoryFailed    メモリが不足しています。
*
*   @details    この API はデバッグ用途に限り使用可能です。@n
*               この API で破壊したセーブデータは、マウント API が ResultDataCorrupted で失敗し、 nn::fs::VerifySaveData() の検証結果が false になります。
*
*   @pre
*           - 対象のセーブデータがマウントされていない
*/
Result CorruptSaveDataForDebug(SaveDataSpaceId spaceId, SaveDataId saveDataId) NN_NOEXCEPT;

//! @}

}} // namespace nn::fs

