﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <memory>
#include <nn/nn_Common.h>
#include <nn/fs/fs_Result.h>
#include <nn/result/result_HandlingUtility.h>

#include <nn/fs/detail/fs_Newable.h>
#include <nn/fs/detail/fs_ResultHandlingUtility.h>
#include <nn/fs/fsa/fs_IFile.h>
#include <nn/fs/fsa/fs_IFileSystem.h>

namespace nn { namespace fs {

namespace
{
    class ReadOnlyFile : public nn::fs::fsa::IFile, public nn::fs::detail::Newable
    {
    public:
        explicit ReadOnlyFile(std::unique_ptr<nn::fs::fsa::IFile>&& pBaseFile) NN_NOEXCEPT
            : m_pUniqueBaseFile(std::move(pBaseFile))
        {
            NN_SDK_REQUIRES_NOT_NULL(m_pUniqueBaseFile);
        }

        virtual Result DoRead(
            size_t* pOutValue,
            int64_t offset,
            void* pBuffer,
            size_t size,
            const nn::fs::ReadOption& option
        ) NN_NOEXCEPT NN_OVERRIDE
        {
            return m_pUniqueBaseFile->Read(pOutValue, offset, pBuffer, size, option);
        }

        virtual Result DoGetSize(int64_t* pOutValue) NN_NOEXCEPT NN_OVERRIDE
        {
            return m_pUniqueBaseFile->GetSize(pOutValue);
        }

        virtual Result DoWrite(
            int64_t offset, const void* pBuffer, size_t size, const nn::fs::WriteOption& option
        ) NN_NOEXCEPT NN_OVERRIDE
        {
            NN_UNUSED(offset);
            NN_UNUSED(pBuffer);
            NN_UNUSED(size);
            NN_UNUSED(option);
            return ResultUnsupportedOperation();
        }

        virtual Result DoFlush() NN_NOEXCEPT NN_OVERRIDE
        {
            NN_RESULT_SUCCESS;
        }

        virtual Result DoSetSize(int64_t size) NN_NOEXCEPT NN_OVERRIDE
        {
            NN_UNUSED(size);
            return ResultUnsupportedOperation();
        }

        virtual Result DoOperateRange(
            void* outBuffer,
            size_t outBufferSize,
            OperationId operationId,
            int64_t offset,
            int64_t size,
            const void* inBuffer,
            size_t inBufferSize) NN_NOEXCEPT NN_OVERRIDE
        {
            switch( operationId )
            {
            case OperationId::Invalidate:
            case OperationId::QueryRange:
                return m_pUniqueBaseFile->OperateRange(
                    outBuffer,
                    outBufferSize,
                    operationId,
                    offset,
                    size,
                    inBuffer,
                    inBufferSize);
            default:
                return nn::fs::ResultUnsupportedOperation();
            }
        }

        virtual ~ReadOnlyFile() NN_NOEXCEPT NN_OVERRIDE
        {
        }

    private:
        std::unique_ptr<nn::fs::fsa::IFile> m_pUniqueBaseFile;
    };
}

template <typename T>
class ReadOnlyFileSystemTemplate : public nn::fs::fsa::IFileSystem, public nn::fs::detail::Newable
{
public:
    explicit ReadOnlyFileSystemTemplate(
        T&& pBaseFileSystem) NN_NOEXCEPT
        : m_pBaseFileSystem(std::move(pBaseFileSystem))
    {
        NN_SDK_REQUIRES_NOT_NULL(m_pBaseFileSystem);
    }

    virtual ~ReadOnlyFileSystemTemplate() NN_NOEXCEPT NN_OVERRIDE
    {
    }

private:
    virtual Result DoOpenFile(
        std::unique_ptr<nn::fs::fsa::IFile>* pOutValue, const char* path, nn::fs::OpenMode mode
    ) NN_NOEXCEPT NN_OVERRIDE
    {
        // 書き込みモードで開こうとしたら失敗
        NN_RESULT_THROW_UNLESS(
            (mode & (OpenMode_Read | OpenMode_Write | OpenMode_AllowAppend)) == OpenMode_Read,
            ResultInvalidArgument()
        );

        std::unique_ptr<nn::fs::fsa::IFile> baseFile;
        NN_RESULT_DO(m_pBaseFileSystem->OpenFile(&baseFile, path, mode));

        std::unique_ptr<nn::fs::fsa::IFile> file(new ReadOnlyFile(std::move(baseFile)));
        NN_RESULT_THROW_UNLESS(file, ResultAllocationMemoryFailedInReadOnlyFileSystemA());

        *pOutValue = std::move(file);
        NN_RESULT_SUCCESS;
    }

    virtual Result DoOpenDirectory(
        std::unique_ptr<nn::fs::fsa::IDirectory>* pOutValue,
        const char* path,
        nn::fs::OpenDirectoryMode mode
    ) NN_NOEXCEPT NN_OVERRIDE
    {
        return m_pBaseFileSystem->OpenDirectory(pOutValue, path, mode);
    }

    virtual Result DoGetEntryType(
        nn::fs::DirectoryEntryType* pOutValue, const char* path
    ) NN_NOEXCEPT NN_OVERRIDE
    {
        return m_pBaseFileSystem->GetEntryType(pOutValue, path);
    }

    virtual Result DoCreateFile(
        const char* path, int64_t size, int option) NN_NOEXCEPT NN_OVERRIDE
    {
        NN_UNUSED(path);
        NN_UNUSED(size);
        NN_UNUSED(option);
        return ResultUnsupportedOperation();
    }

    virtual Result DoDeleteFile(const char* path) NN_NOEXCEPT NN_OVERRIDE
    {
        NN_UNUSED(path);
        return ResultUnsupportedOperation();
    }

    virtual Result DoCreateDirectory(const char* path) NN_NOEXCEPT NN_OVERRIDE
    {
        NN_UNUSED(path);
        return ResultUnsupportedOperation();
    }

    virtual Result DoDeleteDirectory(const char* path) NN_NOEXCEPT NN_OVERRIDE
    {
        NN_UNUSED(path);
        return ResultUnsupportedOperation();
    }

    virtual Result DoDeleteDirectoryRecursively(const char* path) NN_NOEXCEPT NN_OVERRIDE
    {
        NN_UNUSED(path);
        return ResultUnsupportedOperation();
    }

    virtual Result DoCleanDirectoryRecursively(const char* path) NN_NOEXCEPT NN_OVERRIDE
    {
        NN_UNUSED(path);
        return ResultUnsupportedOperation();
    }

    virtual Result DoRenameFile(
        const char* currentPath, const char* newPath) NN_NOEXCEPT NN_OVERRIDE
    {
        NN_UNUSED(currentPath);
        NN_UNUSED(newPath);
        return ResultUnsupportedOperation();
    }

    virtual Result DoRenameDirectory(
        const char* currentPath, const char* newPath
    ) NN_NOEXCEPT NN_OVERRIDE
    {
        NN_UNUSED(currentPath);
        NN_UNUSED(newPath);
        return ResultUnsupportedOperation();
    }

    virtual Result DoCommit() NN_NOEXCEPT NN_OVERRIDE
    {
        NN_RESULT_SUCCESS;
    }

    virtual Result DoGetFreeSpaceSize(
        int64_t* pOutValue, const char* path) NN_NOEXCEPT NN_OVERRIDE
    {
        NN_UNUSED(pOutValue);
        NN_UNUSED(path);
        return ResultUnsupportedOperation();
    }

    virtual Result DoGetTotalSpaceSize(
        int64_t* pOutValue, const char* path) NN_NOEXCEPT NN_OVERRIDE
    {
        NN_UNUSED(pOutValue);
        NN_UNUSED(path);
        return ResultUnsupportedOperation();
    }

private:
    T m_pBaseFileSystem;
};

using ReadOnlyFileSystem = ReadOnlyFileSystemTemplate<std::unique_ptr<nn::fs::fsa::IFileSystem> >;
using ReadOnlyFileSystemShared = ReadOnlyFileSystemTemplate<std::shared_ptr<nn::fs::fsa::IFileSystem> >;

}}
