﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/


 /**
 *   @file
 *   @brief  ファイルシステムに関連するファイルシステム API の宣言
 */

#pragma once

#include <nn/nn_Common.h>
#include <nn/nn_Result.h>
#include <nn/nn_StaticAssert.h>

namespace nn {
namespace fs {

//--------------------------------------------------------------------------
/**
*   @brief      ファイルのタイムスタンプ情報を定義した構造体です
*
*   @details
*       それぞれ UNIX 時間(nn::time::PosixTime の value) で入っています。
*       isLocalTime が true の時は、時差を含んだ現地時間を取得しています。
*       isLocalTime が false の時は、時差を含まない協定世界時(UTC)を取得しています。
*
*/
struct FileTimeStampRaw
{
    uint64_t create;
    uint64_t modify;
    uint64_t access;
    bool isLocalTime;
    char reserved[7];
};
NN_STATIC_ASSERT(std::is_pod<FileTimeStampRaw>::value);


//! @name ファイルシステム関連 API
//! @{

//--------------------------------------------------------------------------
/**
*   @brief      対象のパスのファイルのタイムスタンプを取得します。
*
*   @param[out] outValue    ファイルのタイムスタンプ
*   @param[in]  path        タイムスタンプを取得したいパス
*
*   @return     処理の結果が返ります。
*   @retval     ResultSuccess           成功しました。
*   @retval     ResultPathNotFound      対象のエントリが存在しません。
*
*   @pre
*       - outValue が有効なメモリを指している。
*       - path が @link entry_path 有効なパス @endlink である。
*/
Result GetFileTimeStampRawForDebug(FileTimeStampRaw* outTimeStamp, const char* path) NN_NOEXCEPT;

//! @}

}
} // namespace nn::fs
