﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Common.h>
#include <nn/nn_Abort.h>
#include <nn/util/util_IntrusiveList.h>


namespace nn { namespace fs { namespace detail {

template<typename TKey, typename TValue>
class LruList
{
public:
    struct LruListTag;
    struct HashTableTag;

    class EntryNode
        : public nn::util::IntrusiveListBaseNode<EntryNode, LruListTag>
        , public nn::util::IntrusiveListBaseNode<EntryNode, HashTableTag>
    {
        friend class LruList;
    private:
        TKey key;
        TValue* value;
    private:
        EntryNode() {}
        ~EntryNode() {}
        NN_DISALLOW_COPY(EntryNode);
        NN_DISALLOW_MOVE(EntryNode);
    public:
        TValue* Get() const NN_NOEXCEPT { return value; }
    };

    typedef EntryNode NodeType;
    typedef nn::util::IntrusiveList<EntryNode, nn::util::IntrusiveListBaseNodeTraits<EntryNode, HashTableTag>> BucketType;

private:
    EntryNode* m_pEntryNodes;

    // IntrusiveList の size() は O(n) かかるので、O(1) で済むようカウントは自分で行う
    int m_UsedCount;
    int m_FreeCount;
    nn::util::IntrusiveList<EntryNode, nn::util::IntrusiveListBaseNodeTraits<EntryNode, LruListTag>> m_LruList;
    nn::util::IntrusiveList<EntryNode, nn::util::IntrusiveListBaseNodeTraits<EntryNode, LruListTag>> m_FreeList;

    // 木構造にすると挿入が O(log(n)) になってしまうはず。連結リストなら先頭、末尾への挿入に限り O(1) で済ませられる
    // リスト内のノード数は、key が一様ならば (nodeCount / bucketCount) 個で収まるようになっている
    // CacheSystem 側で bucketCount を調整して検索と削除もできるだけ定数時間に近づけるようにする (その分メモリを犠牲にする)
    int m_BucketCount;
    nn::util::IntrusiveList<EntryNode, nn::util::IntrusiveListBaseNodeTraits<EntryNode, HashTableTag>>* m_pHashTableBuckets;

public:
    LruList() NN_NOEXCEPT
        : m_pEntryNodes(nullptr)
        , m_UsedCount(0)
        , m_FreeCount(0)
        , m_BucketCount(0)
        , m_pHashTableBuckets(nullptr)
    {
    }

    ~LruList() NN_NOEXCEPT
    {
        Finalize();
    }

    void Initialize(void* pBuffer, size_t bufferSize, size_t nodeCount, size_t bucketCount) NN_NOEXCEPT
    {
        NN_ABORT_UNLESS_NOT_NULL(pBuffer);
        NN_ABORT_UNLESS((bucketCount & (bucketCount - 1)) == 0);

        uintptr_t p = reinterpret_cast<uintptr_t>(pBuffer);
        const uintptr_t nodeStart = util::align_up(p, NN_ALIGNOF(NodeType));
        p = nodeStart + sizeof(NodeType) * nodeCount;
        const uintptr_t bucketStart = util::align_up(p, NN_ALIGNOF(BucketType));
        p = bucketStart + sizeof(BucketType) * bucketCount;

        NN_ABORT_UNLESS(p <= reinterpret_cast<uintptr_t>(pBuffer) + bufferSize);

        NN_ABORT_UNLESS(m_pEntryNodes == nullptr);
        m_pEntryNodes = reinterpret_cast<NodeType*>(nodeStart);
        m_UsedCount = 0;
        NN_ABORT_UNLESS(nodeCount <= INT_MAX);
        m_FreeCount = static_cast<int>(nodeCount);
        for (int i = 0; i < m_FreeCount; i++)
        {
            new (&m_pEntryNodes[i]) NodeType();
            m_FreeList.push_back(m_pEntryNodes[i]);
        }

        NN_ABORT_UNLESS(m_pHashTableBuckets == nullptr);
        m_pHashTableBuckets = reinterpret_cast<BucketType*>(bucketStart);
        NN_ABORT_UNLESS(bucketCount <= INT_MAX);
        m_BucketCount = static_cast<int>(bucketCount);
        for (int i = 0; i < m_BucketCount; i++)
        {
            new (&m_pHashTableBuckets[i]) BucketType();
        }
    }

    void Finalize() NN_NOEXCEPT
    {
        m_LruList.clear();
        m_FreeList.clear();

        if (m_pHashTableBuckets)
        {
            for (int i = 0; i < m_BucketCount; i++)
            {
                m_pHashTableBuckets[i].clear();
                m_pHashTableBuckets[i].~IntrusiveList();
            }
            m_BucketCount = 0;
            m_pHashTableBuckets = nullptr;
        }
        if (m_pEntryNodes)
        {
            for (int i = 0; i < (m_UsedCount + m_FreeCount); i++)
            {
                m_pEntryNodes[i].~EntryNode();
            }
            m_FreeCount = 0;
            m_UsedCount = 0;
            m_pEntryNodes = nullptr;
        }
    }

    EntryNode* Get(TKey key) NN_NOEXCEPT
    {
        auto& bucket = m_pHashTableBuckets[GetHashKey(key)];
        for (auto& entry : bucket)
        {
            if (entry.key == key)
            {
                return &entry;
            }
        }
        return nullptr;
    }

    bool ContainsKey(TKey key) NN_NOEXCEPT
    {
        return Get(key) != nullptr;
    }

    int CountUsed() const NN_NOEXCEPT
    {
        return m_UsedCount;
    }

    int CountFree() const NN_NOEXCEPT
    {
        return m_FreeCount;
    }

    int CountTotal() const NN_NOEXCEPT
    {
        return m_UsedCount + m_FreeCount;
    }

    void Add(TKey key, TValue* value) NN_NOEXCEPT
    {
        NN_SDK_ASSERT(Get(key) == nullptr);  // abort すべき？ けど Get 関数は遅い
        NN_SDK_ASSERT(!m_FreeList.empty());

        EntryNode& node = m_FreeList.front();
        m_FreeList.pop_front();
        m_FreeCount--;

        node.key = key;
        node.value = value;
        m_LruList.push_front(node);
        m_UsedCount++;

        auto& bucket = m_pHashTableBuckets[GetHashKey(key)];
        bucket.push_front(node);
    }

    void Promote(EntryNode* pNode) NN_NOEXCEPT
    {
        NN_SDK_ASSERT_NOT_NULL(pNode);
        m_LruList.erase(m_LruList.iterator_to(*pNode));
        m_LruList.push_front(*pNode);
    }

    TValue* Remove() NN_NOEXCEPT
    {
        NN_SDK_ASSERT(!m_LruList.empty());
        return Remove(m_LruList.back());
    }

    TValue* Remove(TKey key) NN_NOEXCEPT
    {
        EntryNode* pNode = Get(key);
        NN_SDK_ASSERT_NOT_NULL(pNode);
        return Remove(*pNode);
    }

    TValue* Remove(EntryNode& node) NN_NOEXCEPT
    {
        m_LruList.erase(m_LruList.iterator_to(node));
        m_UsedCount--;

        m_FreeList.push_front(node);
        m_FreeCount++;

        auto& bucket = m_pHashTableBuckets[GetHashKey(node.key)];
        bucket.erase(bucket.iterator_to(node));

        return node.value;
    }

    // callback から Add や Remove を呼ばないこと
    template<typename TCallback>
    void Remove(TCallback callback) NN_NOEXCEPT
    {
        for (auto it = m_LruList.begin(); it != m_LruList.end(); )
        {
            EntryNode& node = *it;
            bool shouldRemove = callback(node.key, node.value);

            if (shouldRemove)
            {
                it = m_LruList.erase(it);
                m_UsedCount--;

                m_FreeList.push_front(node);
                m_FreeCount++;

                auto& bucket = m_pHashTableBuckets[GetHashKey(node.key)];
                bucket.erase(bucket.iterator_to(node));
            }
            else
            {
                it++;
            }
        }
    }

    void Clear() NN_NOEXCEPT
    {
        for (auto it = m_LruList.begin(); it != m_LruList.end(); )
        {
            EntryNode& entry = *it;
            it = m_LruList.erase(it);
            m_FreeList.push_front(entry);
        }
        NN_SDK_ASSERT(m_LruList.size() == 0);
        m_FreeCount += m_UsedCount;
        m_UsedCount = 0;
        for (int i = 0; i < m_BucketCount; i++)
        {
            m_pHashTableBuckets[i].clear();
        }
    }

    // デバッグ用
    template<typename TCallback>
    void EnumerateUsed(TCallback callback) NN_NOEXCEPT
    {
        for (auto& entry : m_LruList)
        {
            callback(entry.key, entry.value);
        }
    }

private:
    int GetHashKey(TKey key) NN_NOEXCEPT
    {
        NN_SDK_ASSERT((m_BucketCount & (m_BucketCount - 1)) == 0);
        return key.GetKey() & (m_BucketCount - 1);
    }
};

}}}  // namespace nn::fs::detail
