﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Common.h>
#include <nn/nn_Result.h>

#include <nn/fs/fs_Result.h>
#include <nn/fs/fs_ResultPrivate.h>
#include <nn/fs/detail/fs_BufferRegion.h>
#include <nn/fssystem/fs_Assert.h>


namespace nn { namespace fs {

struct ReadOption;


namespace fsa {

class IFile;
class IFileSystem;

}  // namespace nn::fs::fsa


namespace detail {

class FileDataCacheAccessResult
{
public:
    static const int MaxCacheFetchedRegionCount = 8;

private:
    static const int FileDataCacheNotUsed = -1;

private:
    int m_CacheFetchedRegionCount;
    BufferRegion m_CacheFetchedRegions[MaxCacheFetchedRegionCount];

    bool m_ExceededMaxCacheFetchedRegionCount;

public:
    FileDataCacheAccessResult() NN_NOEXCEPT
        : m_CacheFetchedRegionCount(FileDataCacheNotUsed)
        , m_ExceededMaxCacheFetchedRegionCount(false)
    {
    }

    bool IsFileDataCacheUsed() const NN_NOEXCEPT
    {
        return m_CacheFetchedRegionCount >= 0;
    }
    void SetFileDataCacheUsed(bool flag) NN_NOEXCEPT
    {
        if (flag)
        {
            m_CacheFetchedRegionCount = std::max(m_CacheFetchedRegionCount, 0);
        }
        else
        {
            m_CacheFetchedRegionCount = FileDataCacheNotUsed;
        }
    }

    int GetCacheFetchedRegionCount() const NN_NOEXCEPT
    {
        NN_SDK_ASSERT(m_CacheFetchedRegionCount >= 0);
        return m_CacheFetchedRegionCount;
    }

    bool ExceededMaxCacheFetchedRegionCount() const NN_NOEXCEPT
    {
        return m_ExceededMaxCacheFetchedRegionCount;
    }

    void AddCacheFetchedRegion(const BufferRegion& region) NN_NOEXCEPT
    {
        if (m_CacheFetchedRegionCount == FileDataCacheNotUsed)
        {
            m_CacheFetchedRegionCount = 0;
        }
        if (region.size > 0)
        {
            if (m_CacheFetchedRegionCount < MaxCacheFetchedRegionCount)
            {
                m_CacheFetchedRegions[m_CacheFetchedRegionCount++] = region;
            }
            else
            {
                m_CacheFetchedRegions[MaxCacheFetchedRegionCount - 1] = region;
                m_ExceededMaxCacheFetchedRegionCount = true;
            }
        }
    }

    const BufferRegion& GetCacheFetchedRegion(int index) const NN_NOEXCEPT
    {
        NN_SDK_ASSERT(IsFileDataCacheUsed());
        NN_SDK_ASSERT(index >= 0);
        NN_SDK_ASSERT(index < m_CacheFetchedRegionCount);
        return m_CacheFetchedRegions[index];
    }
};

class IFileDataCache
{
public:
    virtual ~IFileDataCache() NN_NOEXCEPT {}

    Result Read(
        fsa::IFile* pFile,
        size_t* outValue,
        int64_t offset,
        void* pBuffer,
        size_t size,
        const ReadOption& option,
        FileDataCacheAccessResult* pAccessResult) NN_NOEXCEPT
    {
        // TORIAEZU: nn::fs::fsa::IFile から借用 (SIGLO-80734)
        NN_FSP_REQUIRES(outValue != nullptr, nn::fs::ResultNullptrArgument());
        if (size == 0)
        {
            *outValue = 0;
            NN_RESULT_SUCCESS;
        }
        NN_FSP_REQUIRES(pBuffer != nullptr, nn::fs::ResultNullptrArgument());
        NN_FSP_REQUIRES(offset >= 0, nn::fs::ResultOutOfRange()); // TODO: ResultInvalidOffset?
        int64_t castedSize = static_cast<int64_t>(size);
        NN_FSP_REQUIRES(castedSize >= 0, nn::fs::ResultOutOfRange()); // TODO: ResultInvalidSize?
        NN_FSP_REQUIRES((INT64_MAX - offset) >= castedSize, nn::fs::ResultOutOfRange()); // TODO: ResultInvalidSize?
        return DoRead(pFile, outValue, offset, pBuffer, size, option, pAccessResult);
    }

    virtual void Purge(fsa::IFileSystem* pFileSystem) NN_NOEXCEPT = 0;

protected:
    virtual Result DoRead(
        fsa::IFile* pFile,
        size_t* outValue,
        int64_t offset,
        void* pBuffer,
        size_t size,
        const ReadOption& option,
        FileDataCacheAccessResult* pAccessResult) NN_NOEXCEPT = 0;
};

}}}  // namespace nn::fs::detail
