﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/os.h>

namespace nn  {
namespace eth {

// Maximum length of the ethernet device name
const int MaxInterfaceNameLength = 32;

// Maximum number of supported media types
const int MaxInterfaceMediaCount = 32;

// Maximum number of ethernet adapters in the system
const int MaxInterfaceCount      = 1;

// Fields in Media type
//      LinkState    bits [31 - 24]
//      Options      bits [23 - 16]
//      SubType      bits [15 -  8]
//      Type         bits [7  -  0]
enum EthShift
{
    Type_Shift           = 0,
    SubType_Shift        = 8,
    Option_Shift         = 16,
    LinkState_Shift      = 24,
    Type_Mask            = (0xff<<Type_Shift),
    SubType_Mask         = (0xff<<SubType_Shift),
    Option_Mask          = (0xff<<Option_Shift),
    LinkState_Mask       = (0xff<<LinkState_Shift),
};

// Ethernet types
enum EthType
{
    Type_10_T            = (0<<Type_Shift),         // 10BaseT - RJ45
    Type_100_TX          = (1<<Type_Shift),         // 100BaseTX - RJ45
    Type_100_T4          = (2<<Type_Shift),         // 100BaseT4 - 4 pair cat 3
    Type_1000_T          = (3<<Type_Shift),         // 1000baseT - 4 pair cat 5
    Type_UNKNOWN         = (4<<Type_Shift),         // media types not defined yet
    Type_INPROGRESS      = (5<<Type_Shift),         // Configuration is in progress
};

// Shared media sub-types
enum EthSubType
{
    SubType_AUTO         = (2<<SubType_Shift),      // Autoselect best media
    SubType_MANUAL       = (1<<SubType_Shift),      // Manual selection
    SubType_NONE         = (0<<SubType_Shift),      // Deselect all media
};

// Shared options (may be combined)
enum EthOption
{
    Option_FDX           = (0x01<<Option_Shift),    // Force full duplex
    Option_HDX           = (0x02<<Option_Shift),    // Force half duplex
    Option_FLOW          = (0x04<<Option_Shift),    // enable hardware flow control
    Option_LOOP          = (0x08<<Option_Shift),    // Put hardware in loopback
    Option_MASTER        = (0x10<<Option_Shift),    // master mode (1000baseT)
    Option_TXPAUSE       = (0x20<<Option_Shift),
    Option_RXPAUSE       = (0x40<<Option_Shift),
};

enum EthLink
{
    LinkState_Unknown    = (0<<LinkState_Shift),    // Link state not known
    LinkState_Up         = (1<<LinkState_Shift),    // Link is UP
    LinkState_Down       = (2<<LinkState_Shift)     // Link is Down
};

// Media type keyword
#define EthMakeMediaType(type, subtype, options, link) \
                                    ((type) | (subtype) | (options) | (link))

#define EthMediaTypeOptions(media)  ((media)  & nn::eth::Option_Mask)
#define EthMediaType(media)         ((media)  & nn::eth::Type_Mask)
#define EthMediaSubType(media)      ((media)  & nn::eth::SubType_Mask)
#define EthMediaLink(media)         ((media)  & nn::eth::LinkState_Mask)
#define EthLinkUp(media)            (EthMediaLink(media) == nn::eth::LinkState_Up ? true : false)
#define EthFullDuplex(media)        ((media)  & nn::eth::Option_FDX  ? true : false)
#define EthHalfDuplex(media)        ((media)  & nn::eth::Option_HDX  ? true : false)
#define EthFlow(media)              ((media)  & nn::eth::Option_FLOW ? true : false)
#define EthAuto(media)              (EthMediaSubType(media) == nn::eth::SubType_AUTO ? true : false)

#define EthSpeed(media)                 \
({                                      \
    uint32_t speed = 0;                 \
    switch (EthMediaType(media))        \
    {                                   \
    case nn::eth::Type_10_T:            \
        speed = 10;                     \
        break;                          \
    case nn::eth::Type_100_TX:          \
    case nn::eth::Type_100_T4:          \
        speed = 100;                    \
        break;                          \
    case nn::eth::Type_1000_T:          \
        speed = 1000;                   \
        break;                          \
    default:                            \
        break;                          \
    }                                   \
    speed;                              \
})

// Ethernet media types supported by EthCfg module.
enum MediaType
{
    // Unknown
    MediaType_Unknown                = EthMakeMediaType(Type_UNKNOWN, SubType_MANUAL, 0, LinkState_Unknown),
    // Being configured
    MediaType_ConfigInProgress       = EthMakeMediaType(Type_INPROGRESS, SubType_MANUAL, 0, LinkState_Down),
    // Link lost
    MediaType_LinkLost               = EthMakeMediaType(Type_UNKNOWN, SubType_MANUAL, 0, LinkState_Down),
    // No link
    MediaType_None                   = EthMakeMediaType(Type_UNKNOWN, SubType_NONE,   0, LinkState_Down),
    // Auto
    MediaType_AUTO                   = EthMakeMediaType(Type_UNKNOWN, SubType_AUTO,   0, LinkState_Up),
    // 10BaseT
    MediaType_10_T                   = EthMakeMediaType(Type_10_T,    SubType_MANUAL, 0, LinkState_Up),
    // 10BaseT, Full duplex
    MediaType_10_T_FDX               = EthMakeMediaType(Type_10_T,    SubType_MANUAL, Option_FDX, LinkState_Up),
    // 10BaseT, Full duplex, Flow control
    MediaType_10_T_FDX_FLOW          = EthMakeMediaType(Type_10_T,    SubType_MANUAL, Option_FDX | Option_FLOW, LinkState_Up),
    // 100BaseTX
    MediaType_100_TX                 = EthMakeMediaType(Type_100_TX,  SubType_MANUAL, 0, LinkState_Up),
    // 100BaseTX, Full duplex
    MediaType_100_TX_FDX             = EthMakeMediaType(Type_100_TX,  SubType_MANUAL, Option_FDX, LinkState_Up),
    // 100BaseTX, Full duplex, Flow control
    MediaType_100_TX_FDX_FLOW        = EthMakeMediaType(Type_100_TX,  SubType_MANUAL, Option_FDX | Option_FLOW, LinkState_Up),
    // 100BaseT4
    MediaType_100_T4                 = EthMakeMediaType(Type_100_T4,  SubType_MANUAL, 0, LinkState_Up),
    // 1000BaseT
    MediaType_1000_T                 = EthMakeMediaType(Type_1000_T,  SubType_MANUAL, 0, LinkState_Up),
    // 1000BaseT, Master
    MediaType_1000_T_MASTER          = EthMakeMediaType(Type_1000_T,  SubType_MANUAL, Option_MASTER, LinkState_Up),
    // 1000BaseT, Full duplex
    MediaType_1000_T_FDX             = EthMakeMediaType(Type_1000_T,  SubType_MANUAL, Option_FDX, LinkState_Up),
    // 1000BaseT, Full duplex, Flow control
    MediaType_1000_T_FDX_FLOW        = EthMakeMediaType(Type_1000_T,  SubType_MANUAL, Option_FDX | Option_FLOW, LinkState_Up),
    // 1000BaseT, Master, Full duplex
    MediaType_1000_T_FDX_MASTER      = EthMakeMediaType(Type_1000_T,  SubType_MANUAL, Option_MASTER | Option_FDX, LinkState_Up),
    // 1000BaseT, Master, Full duplex, Flow control
    MediaType_1000_T_FDX_MASTER_FLOW = EthMakeMediaType(Type_1000_T,  SubType_MANUAL, Option_MASTER | Option_FDX | Option_FLOW, LinkState_Up),
};

typedef struct InterfaceInfo
{
    char      interfaceName[MaxInterfaceNameLength];
    uint8_t   interfaceMacAddress[6];
    uint32_t  interfaceModel;
    uint32_t  interfaceRevision;
    uint32_t  interfaceUniqueIdentifier;
}
InterfaceInfo;

// Information about all attached ethernet adapters in the system.
typedef struct InterfaceList
{
    // Number of adapters in the system
    uint32_t      adapterCount;
    InterfaceInfo adapterInfo[MaxInterfaceCount];
}
InterfaceList;

// Information about adapter capabilities, lists media types
// supported by this ethernet adapter.
typedef struct MediaList
{
    uint32_t   mediaCount;
    MediaType  mediaType[MaxInterfaceMediaCount];
}
MediaList;

}}
