﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * @file
 * @brief   ヘッダ内で使用する ForEach ユーティリティ実装の宣言と定義
 */

#pragma once

#include <mutex>
#include <nn/nn_Common.h>
#include <nn/result/result_HandlingUtility.h>

namespace nn { namespace ddsf { namespace detail {

/**
* @brief 指定したリストの各要素について指定した関数を呼び出します（リザルトハンドリングあり）
*
* @tparam   RangeT  range-based for loop で T 型を列挙する range-expression として使用できる型
* @tparam   FuncT   RangeT の要素型を T として、 T* または const T* を引数に取って nn::Result を返す関数として振る舞える型
* @tparam   LockT   C++ 標準の Lockable 要件を満たす型
*
* @param[in] pList              メソッドを適用するリスト
* @param[in] func               呼び出すメソッド
* @param[in] pLockable          リストの走査中にロックする同期オブジェクト
* @param[in] returnOnFailure    @a func が失敗したときの挙動オプション
*   returnOnFailure | 動作
*   --------------- | ------
*   true            | func のひとつが nn::ResultSuccess 以外を返した時点で以降のメソッド呼び出しを中止する
*   false           | func のひとつが nn::ResultSuccess 以外を返しても以降のメソッド呼び出しを続ける
*
* @return
*   すべての @a func の呼び出しが nn::ResultSuccess を返したとき、 nn::ResultSuccess を返します。 @n
*   いずれかの @a func の呼び出しが nn::ResultSuccess 以外のリザルトを返したとき、そのリザルトを返します。 @n
*   @a returnOnFailure が false で、複数の func 呼び出しが失敗した場合は、最初の失敗のリザルトを返します。
*/
template<typename RangeT, typename FuncT, typename LockT>
inline nn::Result ForEach(RangeT* pList, FuncT func, LockT* pLockable, bool returnOnFailure) NN_NOEXCEPT
{
    std::lock_guard<LockT> lock(*pLockable);
    nn::Result result = nn::ResultSuccess();
    for ( auto&& item : *pList )
    {
        auto currentResult = func(&item);
        if ( currentResult.IsFailure() )
        {
            if ( returnOnFailure )
            {
                return currentResult;
            }
            else if ( result.IsSuccess() )
            {
                result = currentResult;
            }
        }
    }
    return result;
}

/**
* @brief 指定したリストの各要素について指定した関数を呼び出します（リザルトハンドリングなし）
*
* @tparam   RangeT  range-based for loop で T 型を列挙する range-expression として使用できる型
* @tparam   FuncT   RangeT の要素型を T として、 T* または const T* を引数に取って bool を返す関数として振る舞える型
* @tparam   LockT   C++ 標準の Lockable 要件を満たす型
*
* @param[in] pList              メソッドを適用するリスト
* @param[in] func               呼び出すメソッド
* @param[in] pLockable          リストの走査中にロックする同期オブジェクト
*
* @return   func が true を返した回数
*
* @detail
*   指定したリストの各要素について指定した関数を呼び出します。 @n
*   呼び出された @a func が true を返す限り、続く要素へのメソッド呼び出しを継続します。 @n
*   @a func が false を返した時点で以降のメソッド呼び出しを中止します。 @n
*/
template<typename FuncT, typename RangeT, typename LockT>
inline int ForEach(RangeT* pList, FuncT func, LockT* pLockable) NN_NOEXCEPT
{
    int count = 0;
    std::lock_guard<LockT> lock(*pLockable);
    for ( auto&& item : *pList )
    {
        if ( !func(&item) )
        {
            return count;
        }
        ++count;
    }
    return count;
}

}}} // namespace nn::ddsf::detail
