﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * @file
 * @brief   デバイスドライバ向けアドレス変換に関する公開ヘッダファイル
 */

#pragma once

#include <nn/nn_Common.h>
#include <nn/nn_Macro.h>
#include <nn/dd/dd_Result.h>

namespace nn { namespace dd {

//--------------------------------------------------------------------------
/**
 * @brief   物理メモリアドレスを表す型です
 */
typedef uint64_t PhysicalAddress;

//--------------------------------------------------------------------------
/**
 * @brief   仮想メモリ領域に対応する物理メモリ領域を取得するための構造体です
 */
struct PhysicalMemoryInfo
{
    PhysicalAddress   physicalAddress;
    uintptr_t         virtualAddress;
    size_t            size;

    uintptr_t         _endVirtualAddress;
};

//! @name アドレス変換関連 API
//! @{

/**
 * @brief  PhysicalMemoryInfo オブジェクトを初期化します
 *
 * @param[out] pOutInfo  PhysicalMemoryInfo オブジェクトへのポインタ
 * @param[in]  addr      仮想メモリ領域の先頭アドレス
 * @param[in]  size      仮想メモリ領域のサイズ
 *
 * @details
 *  指定された仮想メモリ領域に対応する物理メモリ情報を得るため、PhysicalMemoryInfo オブジェクトを初期化します。
 *  初期化した PhysicalMemoryInfo オブジェクトを用いて QueryNextPhysicalMemoryInfo() を呼び出すことで
 *  物理メモリ情報を得ます。
 */
void InitializePhysicalMemoryInfo( PhysicalMemoryInfo* pOutInfo, const void* addr, size_t size ) NN_NOEXCEPT;

/**
 * @brief   仮想メモリ領域に対応する物理メモリ情報を得ます
 *
 * @param[out] pOutInfo  PhysicalMemoryInfo オブジェクトへのポインタ
 *
 * @retresult
 *   @handleresult{nn::dd::ResultEndOfQuery}
 *   @handleresult{nn::dd::ResultInvalidCurrentMemory}
 * @endretresult
 *
 * @pre
 *  - PhysicalMemoryInfo オブジェクトが初期化済み
 *  - PhysicalMemoryInfo オブジェクトが適切なメモリ領域に配置されている
 *
 * @details
 *  指定された仮想メモリ領域の物理メモリ情報を得ます。
 *
 *  物理アドレス情報は、システムが管理する単位に区切られて取得します。
 *  指定された仮想メモリ領域に対応する全ての物理メモリ情報を取得するためには、
 *  本 API の返り値が false になるまで繰り返し呼び出す必要があります。
 *
 *  使用例：
 *      @code
 *      nn::Result result;
 *      nn::dd::PhysicalMemoryInfo pmi;
 *      nn::dd::InitializePhysicalMemoryInfo(&pmi, buffer, bufferSize);
 *      while ((result = nn::dd::QueryNextPhysicalMemoryInfo(&pmi)).IsSuccess())
 *      {
 *          // nn::dd::GetPhysicalAddress(&pmi) などを用いて物理アドレスに対する処理を行う
 *      }
 *
 *      if (nn::dd::ResultEndOfQuery::Includes(result))
 *      {
 *          // 成功
 *      }
 *      else
 *      {
 *          // エラー処理
 *      }
 *      @endcode
 */
Result QueryNextPhysicalMemoryInfo( PhysicalMemoryInfo* pOutInfo ) NN_NOEXCEPT;

/**
 * @brief  単体領域で管理されている物理メモリ領域の物理アドレスを得ます
 *
 * @param[out] pOutAddress  物理アドレス型へのポインタ
 * @param[in]  addr         仮想メモリ領域の先頭アドレス
 * @param[in]  size         仮想メモリ領域のサイズ
 *
 * @retresult
 *   @handleresult{nn::dd::ResultInvalidCurrentMemory}
 *   @handleresult{nn::dd::ResultNotSingleRegion}
 * @endretresult
 *
 * @details
 *  指定された仮想メモリ領域の物理メモリ情報を得ます。
 *  ただし、物理メモリ領域が複数の領域に分かれて管理されている場合エラーが返ります。
 *
 *  使用例：
 *      @code
 *      nn::dd::PhysicalAddress address;
 *      nn::Result result = nn::dd::QuerySinglePhysicalAddress(&address, buffer, bufferSize);
 *      if (result.IsSuccess())
 *      {
 *          // 物理アドレスに対する処理を行う
 *      }
 *      else
 *      {
 *          // エラー処理
 *      }
 *      @endcode
 */
Result QuerySinglePhysicalAddress( PhysicalAddress* pOutAddress, const void* addr, size_t size ) NN_NOEXCEPT;

/**
 * @brief  PhysicalMemoryInfo オブジェクトから物理メモリ領域の物理アドレスを得ます
 *
 * @param[in] pmi  PhysicalMemoryInfo オブジェクトへのポインタ
 *
 * @return PhysicalMemoryInfo オブジェクトが持つ物理メモリ領域の物理アドレス
 *
 * @pre
 *  - QueryNextPhysicalMemoryInfo() で物理メモリ情報を取得済み
 *
 * @details
 *  QueryNextPhysicalMemoryInfo() によって取得された物理メモリ領域の物理アドレスを得ます。
 */
inline PhysicalAddress GetPhysicalAddress( const PhysicalMemoryInfo* pmi ) NN_NOEXCEPT
{
    return pmi->physicalAddress;
}

/**
 * @brief  PhysicalMemoryInfo オブジェクトから物理メモリ領域に対応する仮想アドレスを得ます
 *
 * @param[in] pmi  PhysicalMemoryInfo オブジェクトへのポインタ
 *
 * @return PhysicalMemoryInfo オブジェクトが持つ物理メモリ領域に対応する仮想アドレス
 *
 * @pre
 *  - QueryNextPhysicalMemoryInfo() で物理メモリ情報を取得済み
 *
 * @details
 *  QueryNextPhysicalMemoryInfo() によって取得された物理メモリ領域に対応する仮想アドレスを得ます。
 */
inline const void* GetVirtualAddress( const PhysicalMemoryInfo* pmi ) NN_NOEXCEPT
{
    return reinterpret_cast<void*>(pmi->virtualAddress);
}

/**
 * @brief  PhysicalMemoryInfo オブジェクトから物理メモリ領域のサイズを得ます
 *
 * @param[in] pmi  PhysicalMemoryInfo オブジェクトへのポインタ
 *
 * @return PhysicalMemoryInfo オブジェクトが持つ物理メモリ領域のサイズ
 *
 * @pre
 *  - QueryNextPhysicalMemoryInfo() で物理メモリ情報を取得済み
 *
 * @details
 *  QueryNextPhysicalMemoryInfo() によって取得された物理メモリ領域のサイズを得ます。
 */
inline size_t GetSize( const PhysicalMemoryInfo* pmi ) NN_NOEXCEPT
{
    return pmi->size;
}

//! @}

}}  // namespace nn::dd
